// Copyright 2024 XMOS LIMITED.
// This Software is subject to the terms of the XMOS Public Licence: Version 1.
#include <xcore/parallel.h>
#include <xcore/hwtimer.h>
#include <xs1.h>
#include <stdlib.h>
#include <stdio.h>
#include <print.h>
#include <platform.h>
#include <xscope.h>

#include <string.h>

#include "src.h"


int fs_code(int frequency) {
    if(frequency == 44100) {
        return  FS_CODE_44;
    } else if(frequency == 48000) {
        return  FS_CODE_48;
    } else if(frequency == 88200) {
        return  FS_CODE_88;
    } else if(frequency == 96000) {
        return  FS_CODE_96;
    } else if(frequency == 176400) {
        return  FS_CODE_176;
    } else if(frequency == 192000) {
        return  FS_CODE_192;
    } else {
        printf("Illegal freq %d\n", frequency);
        exit(1);
    }
}


#define     SRC_N_CHANNELS                (1)   // Total number of audio channels to be processed by SRC (minimum 1)
#define     SRC_N_INSTANCES               (1)   // Number of instances (each usually run a logical core) used to process audio (minimum 1)
#define     SRC_CHANNELS_PER_INSTANCE     (SRC_N_CHANNELS/SRC_N_INSTANCES) // Calculated number of audio channels processed by each core
#define     SRC_N_IN_SAMPLES              (4)   // Number of samples per channel in each block passed into SRC each call
                                                // Must be a power of 2 and minimum value is 4 (due to two /2 decimation stages)
#define     SRC_N_OUT_IN_RATIO_MAX        (5)   // Max ratio between samples out:in per processing step (44.1->192 is worst case)
#define     SRC_DITHER_SETTING            (0)   // Enables or disables quantisation of output with dithering to 24b
#define     SRC_MAX_NUM_SAMPS_OUT         (SRC_N_OUT_IN_RATIO_MAX * SRC_N_IN_SAMPLES)
#define     SRC_OUT_BUFF_SIZE             (SRC_CHANNELS_PER_INSTANCE * SRC_MAX_NUM_SAMPS_OUT) // Size of output buffer for SRC for each instance

/* Stuff that must be defined for lib_src */
#define ASRC_N_IN_SAMPLES                 (SRC_N_IN_SAMPLES) /* Used by SRC_STACK_LENGTH_MULT in src_mrhf_asrc.h */

#define ASRC_N_CHANNELS                   (SRC_CHANNELS_PER_INSTANCE) /* Used by SRC_STACK_LENGTH_MULT in src_mrhf_asrc.h */



int32_t sine_wave[16] = {
          0,
  821806413,
 1518500249,
 1984016188,
 2147483647,
 1984016188,
 1518500249,
  821806413,
          0,
 -821806413,
-1518500249,
-1984016188,
-2147483647,
-1984016188,
-1518500250,
 -821806413,
};

int32_t sine2_wave[19] = {
          0,
  697286800,
 1319011756,
 1797801322,
 2081771218,
 2140148902,
 1966608243,
 1579955067,
 1022089244,
  353464191,
 -353464191,
-1022089243,
-1579955067,
-1966608243,
-2140148902,
-2081771219,
-1797801322,
-1319011757,
 -697286801,
};



int expected_wave_48_192[] = {
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
-23, -44, -47, -25, -3, -25, -17, 350,
1778, 5429, 12566, 23861, 38432, 53084, 62292, 59348,
38608, -1797, -58045, -120764, -177826, -219112, -241321, -250349,
-259439, -283140, -329454, -394079, -460370, -506146, -514853, -485646,
-436478, -397140, -394380, -436165, -503972, -558948, -560795, -490638,
-365357, -233690, -153379, -159526, -241106, -340569, -380309, -304701,
-115617, 120183, 300187, 340137, 225023, 26936, -125201, -112963,
101385, 442343, 750185, 862632, 708407, 359646, 7473, -134555,
48293, 493610, 977269, 1224660, 1064195, 538551, -95996, -494460,
-413443, 136717, 870749, 1364756, 1287656, 603178, -378264, -1143758,
-1254546, -607797, 470332, 1371478, 1530345, 757043, -616049, -1876431,
-2308045, -1605936, -91706, 1414359, 2022546, 1283281, -511466, -2429237,
-3379025, -2729228, -715348, 1618263, 2950975, 2426535, 190908, -2602806,
-4372892, -3987685, -1461806, 1949846, 4368475, 4324378, 1645381, -2326742,
-5377463, -5624326, -2638997, 2170276, 6204679, 7070291, 3985666, -1610962,
-6681335, -8193088, -4880128, 1843252, 8426105, 11016600, 7665547, -331891,
-8820357, -12863252, -9514811, 222112, 11468605, 17779992, 14663243, 2465347,
-13083407, -23308141, -21087374, -5237078, 17637910, 35244935, 35227575, 12041701,
-28089292, -66753291, -78213639, -37960264, 68404642, 239606685, 458803798, 699316679,
933586692, 1141670391, 1315937143, 1460491636, 1586240495, 1704139523, 1819437171, 1929226106,
2024522892, 2095748188, 2137856994, 2147483647, 2141517241, 2112512913, 2068532325, 2011040154,
1938368567, 1846689055, 1732700024, 1596144372, 1440339863, 1270660436, 1092203382, 907978978,
718429151, 522367551, 318727356, 108126814, -106568546, -320906687, -530289194, -731372061,
-922700129, -1104312894, -1276627193, -1439258338, -1590420115, -1727182313, -1846386660, -1945686818,
-2024155943, -2082179263, -2120752220, -2140609954, -2141668070, -2123048682, -2083630519, -2022787233,
-1940903978, -1839413883, -1720371281, -1585823767, -1437328371, -1275851822, -1102068222, -916855647,
-721702111, -518805270, -310829440, -100469955, 109937895, 318571105, 523983826, 724791333,
919367509, 1105723793, 1281627484, 1444883521, 1593625352, 1726468295, 1842465681, 1940916156,
2021144416, 2082378430, 2123783077, 2144620744, 2144444804, 2123223081, 2081334368, 2019452756,
1938390280, 1838982543, 1722068910, 1588564054, 1439568401, 1276450745, 1100856029, 914636071,
719739079, 518112144, 311655885, 102240236, -108244224, -317838850, -524507946, -726177192,
-920804587, -1106452503, -1281334013, -1443821410, -1592426513, -1725773737, -1842588296, -1941709730,
-2022126807, -2083018244, -2123782563, -2144046984, -2143656417, -2122652439, -2081254626, -2019852993,
-1939012129, -1839481470, -1722202357, -1588305629, -1439097192, -1276035593, -1100706852, -914802974,
-720105719, -518475488, -311840847, -102186026, 108466783, 318079099, 524620798, 726094029,
920555304, 1106134013, 1281048717, 1443621466, 1592291960, 1725631356, 1842356261, 1941341760,
2021633286, 2082456422, 2123224690, 2143545164, 2143222088, 2122258625, 2080856661, 2019414939,
1938525126, 1838966271, 1721697135, 1587847133, 1438705268, 1275707920, 1100424780, 914543972,
719855571, 518234606, 311622729, 102009783, -108585611, -318135230, -524621057, -726054472,
-920495619, -1106071860, -1280996043, -1443583515, -1592268512, -1725619077, -1842350994, -1941340054,
-2021632960, -2082456433, -2123224708, -2143545169, -2143222115, -2122258673, -2080856706, -2019414963,
-1938525133, -1838966271, -1721697135, -1587847135, -1438705271, -1275707923, -1100424782, -914543974,
-719855573, -518234608, -311622731, -102009785, 108585609, 318135228, 524621055, 726054470,
920495617, 1106071858, 1280996041, 1443583513, 1592268510, 1725619075, 1842350992, 1941340052,
2021632958, 2082456431, 2123224706, 2143545167, 2143222114, 2122258672, 2080856704, 2019414960,
1938525130, 1838966268, 1721697132, 1587847132, 1438705268, 1275707920, 1100424780, 914543972,
719855571, 518234606, 311622729, 102009783, -108585611, -318135230, -524621057, -726054472,
-920495619, -1106071860, -1280996043, -1443583515, -1592268512, -1725619077, -1842350994, -1941340054,
-2021632960, -2082456433, -2123224708, -2143545169,
};

int expected_wave_192_176[] = {
0, 0, -3, 49, 250, -14249, -213106, -527848,
-405969, -775573, -301298, -357497, -336513, 919167, -1030364, 2683007,
-2241030, 3722680, -2622945, 2103758, 232658, -4435221, 9507697, -19548016,
33312169, -71395972, 291387234, 1228506666, 1823787865, 2132583644, 2049481317, 1588743650,
859587041, -44643092, -921864923, -1648756329, -2067758933, -2121704717, -1790435028, -1137929327,
-281274546, 627235130, 1421603895, 1961170377, 2147275153, 1947268590, 1396865379, 594887247,
-314160354, -1166681086, -1809281066, -2126337506, -2060802505, -1624467783, -895842915, -6029338,
884869271, 1616553465, 2057371537, 2128007211, 1815751023, 1176787123, 326084238, -583290955,
-1387714705, -1942447284, -2147483648, -1966473393, -1431443902, -638855042, 268682841, 1127876731,
1784131817, 2119368766, 2073267996, 1654124587, 937354916, 51927157, -842843857, -1585962115,
-2043718589, -2133749224, -1839854851, -1214915811, -371377214, 538983200, 1352364468, 1922415090,
2146566013, 1984485836, 1465337468, 682531283,
};


int expected_wave_192_44[] = {
0, 0, -1, 0, 0, -8, -82, -211,
2208, 95538, 385483, -41732, -528136, 385728, 140766, -40118,
-632517, 842205, -347930, 105815, -606335, 1011340, -577464, -87181,
97495, 154304, 357379, -1558416, 2438950, -2832479, 3745046, -5668892,
7788799, -9453092, 11382069, -14544293, 18658954, -22945419, 27906825, -35238166,
46385448, -64199295, 93461330, -169545633, 1183694722, 1613881895, -1719076513, -1062562787,
1961136216, 556677952, -2141172833, 54586833, 2113580099, -631890381, -1943179032, 1171986264,
1618643345, -1621741530, -1166992530, 1941705336, 632878857, -2120747578, -41813462, 2128731244,
-544372039, -1980982034, 1095614564, 1675312166, -1557691300, -1245448689, 1903785982, 717329779,
-2101796881, -135704427, 2139650601, -456990372, -2013028054, 1014449959, 1732066800, -1493674093,
-1318244279, 1858751196, 803490318, -2081266100, -227114387, 2144162104, -366679651, -2042615587,
932351882, 1784414658, -1426519158, -1389361067, 1811282166, 887753222, -2057132332, -318060786,
2145214573, -276024728, -2068773593, 848941027, 1833671872, -1356749413, -1457940180, 1760504401,
970394479, -2029240848, -408426337, 2142348453, -184865604, -2091152639, 763979444, 1879579818,
-1284501320, -1523856123, 1706510783,
};

int expected_wave_44_192[] = {
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
-21, -42, -49, -42, -8, -6, -26, 58,
614, 2324, 6142, 13031, 23448, 36708, 50471, 60685,
62198, 50108, 21527, -22813, -77897, -135279, -185501, -221423,
-241234, -249779, -257238, -275226, -311590, -366040, -428814, -483553,
-513762, -510439, -477432, -431580, -396689, -393200, -427839, -488265,
-546115, -568297, -532307, -438724, -314358, -203105, -147165, -166390,
-245633, -337445, -380967, -329957, -177444, 35560, 233710, 339594,
311164, 166586, -17381, -135961, -104466, 97654, 408987, 706526,
858711, 788868, 518942, 169702, -87648, -110036, 144103, 583440,
1014382, 1225331, 1088731, 632457, 42120, -414040, -508566, -170547,
464138, 1099410, 1409013, 1194062, 493093, -414215, -1121664, -1284220,
-793073, 149362, 1100729, 1574586, 1275670, 262224, -1051132, -2065642,
-2276390, -1528435, -123024, 1291384, 2008223, 1605169, 175781, -1672974,
-3075861, -3312890, -2175087, -109457, 1946490, 2984947, 2417132, 396863,
-2194290, -4120664, -4370706, -2672324, 312921, 3238386, 4677714, 3818918,
913701, -2758572, -5434472, -5692826, -3181853, 1117730, 5261436, 7213997,
5841587, 1547463, -3790950, -7613516, -7874253, -4094373, 2256164, 8296369,
11014074, 8733652, 2074361, -6117197, -11930223, -12195515, -6125452, 4023466,
13693434, 17974704, 13966032, 2465347, -11898700, -22358481, -22968992, -11631707,
8210301, 28178671, 37951438, 29705708, 2354672, -36170781, -69735263, -77924907,
-41910226, 49729465, 197263910, 389490372, 606950567, 827540728, 1032537855, 1210982913,
1361066579, 1488371681, 1601973078, 1709956928, 1815882301, 1917403559, 2007681714, 2078963595,
2126252645, 2147483647, 2147483647, 2131921229, 2099990777, 2056111823, 2000768538, 1932203889,
1847329107, 1743597862, 1620666642, 1480814954, 1328059386, 1166627303, 999577608, 828120408,
651842119, 469659327, 281017888, 86776494, -110605816, -307611949, -500642688, -686956651,
-865158947, -1035057532, -1197007334, -1351082621, -1496472317, -1631358222, -1753300545, -1859921372,
-1949556753, -2021587724, -2076326911, -2114548548, -2136904125, -2143499546, -2133814623, -2106976393,
-2062234716, -1999409941, -1919107861, -1822623313, -1711596294, -1587594275, -1451812678, -1305017436,
-1147732699, -980565765, -804506165, -621061303, -432174060, -239972974, -46475988, 146621057,
337986167, 526525382, 711164752, 890652646, 1063474821, 1227917229, 1382240279, 1524882236,
1654601253, 1770503449, 1871956463, 1958442635, 2029425979, 2084294113, 2122397252, 2143159756,
2146209242, 2131464602, 2099148108, 2049723136, 1983791948, 1902001271, 1804997543, 1693441383,
1568070301, 1429771521, 1279629500, 1118925335, 949089637, 771629607, 588061120, 399869712,
208510288, 15435090, -177869662, -369861245, -558946938, -743517162, -921994717, -1092881126,
-1254786188, -1406434081, -1546651624, -1674349659, -1788511353, -1888194534, -1972550095, -2040848075,
-2092503144, -2127090405, -2144347688, -2144166604, -2126578331, -2091741192, -2039934853, -1971562058,
-1887155071, -1787381823, -1673046978, -1545087190, -1404557950, -1252618264, -1090514184, -919565329,
-741154664, -556721361, -367753989, -175782833, 17631051, 210908992, 402470874, 590750996,
774213080, 951364416, 1120767138, 1281048716, 1430911134, 1569140244, 1694614710, 1806315308,
1903333482, 1984879577, 2050289499, 2099030790, 2130706752, 2145059671, 2141972839, 2121471408,
2083722051, 2029031674, 1957844915, 1870740521, 1768426644, 1651735178, 1521614849, 1379123393,
1225419507, 1061752715, 889453713, 709923270, 524621056, 335053552, 142762032, -50690211,
-243730305, -434788897, -622312464, -804776530, -980697723, -1148645604, -1307254799, -1455235766,
-1591385430, -1714596832, -1823868268, -1918311317, -1997158178, -2059767768, -2105631122, -2134375255,
-2145766568, -2139712420, -2116262031, -2075606051, -2018075030, -1944136700, -1854392201, -1749571155,
-1630525788, -1498223926, -1353741282, -1198252537, -1033021637, -859392172, -678775666, -492640630,
-302500323, -109900656, 83592559, 276406118, 466972478, 653742239, 835197041, 1009861431,
1176315476, 1333205948, 1479257170, 1613281786, 1734190113, 1840999178, 1932840577, 2008967659,
2068761475, 2111735898, 2137541512, 2145968558, 2136948476, 2110554644, 2067001609, 2006643494,
1929971003, 1837607527, 1730303955, 1608932721, 1474480558, 1328040625, 1170803442, 1004047479,
829128475, 647468353, 460544301, 269875914, 77013413, -116475245, -309016906, -499046231,
-685018201, -865420882, -1038787515, -1203708684, -1358843379, -1502930459, -1634798521, -1753375331,
-1857696896, -1946915047, -2020304437, -2077268394, -2117343795, -2140204804,
};

int expected_wave_176_96[] = {
0, -1, -2, 1, 230, -304, -39420, -317025,
-551858, -293778, -276463, 461741, 228637, 660283, 167594, -323477,
197812, -1662153, 1535433, -2672857, 3831614, -3789295, 5290143, -5004003,
4656502, -4344534, 1378604, 1551789, -8413473, 18920311, -42253320, 911944867,
2041873121, 1992038037, 1041091809, -487836039, -1736108032, -2141491477, -1467645589, -66236870,
1366977132, 2123223952, 1815013207, 608542161, -907331331, -1965807735, -2048299602, -1107017618,
384412029, 1685061970, 2145487114, 1536333647, 161603044, -1293963142, -2104270929, -1865905361,
-697595601, 818438458, 1926497382, 2074235732, 1188008832, -290416829, -1624075475, -2147483648,
-1601436735, -256424306, 1216410539, 2082888938, 1911088644, 786648715, -729919976, -1882638070,
-2096898550, -1265896831, 196129264, 1560388844, 2146825608, 1663112527, 350370960, -1137023567,
-2057634462, -1952555544, -874166535, 639977113, 1835104887, 2115469396, 1341314509, -101458969,
-1493657206, -2141296984, -1721542875, -443633886, 1055417758, 2028364649, 1990212231, 959978447,
-548785376, -1783990624, -2129912044, -1414114696, 6590740, 1424010740, 2131589778, 1776613732,
536031092, -971752375, -1995136697, -2023984958, -1043917064,
};

int expected_wave_192_88[] = {
0, -1, -5, 107, -8, -46506, -406578, -688759,
-80096, -153494, 869358, 342084, -48030, 570740, -2408566, 2498264,
-4003766, 5165156, -4161509, 3938565, -1089572, -4335376, 10737489, -22989437,
42004241, -89649778, 654908190, 2016558843, 1903091092, 510045524, -1232345720, -2145358170,
-1558791461, 73026568, 1678094017, 2111181957, 1106090315, -666913592, -1978138046, -1930126115,
-556687925, 1201224111, 2131756166, 1598287731, -34451689, -1643254427, -2122581345, -1143373626,
621641183, 1959372981, 1950340909, 600752983, -1161759141, -2125742638, -1628608852, -12059954,
1612778306, 2129082454, 1181973717, -577558026, -1940108493, -1969136329, -644689317, 1122881246,
2118645689, 1658170891, 57961655, -1582087199, -2134697307, -1220035207, 533211249, 1919957937,
1987032273, 688331195, -1083490518, -2110581115, -1686975615, -103836908, 1550673544, 2139337197,
1257539480, -488620952, -1898930508, -2004020878, -731658712, 1043604930, 2101552593, 1715009862,
149664727, -1518551428, -2143000016, -1294469421,
};

int expected_wave_176_88[] = {
0, -1, -5, 64, 312, -11132, -200967, -658596,
-457545, -219145, 193032, 863686, -27371, 1085715, -1506095, 888432,
-2043335, 1026703, -186982, -276954, 3376865, -5351517, 8712788, -13829013,
17967245, -25563375, 35753375, -63089301, 279625068, 1647760364, 2147483647, 1343347652,
-204498368, -1676936456, -2127436347, -1365060992, 223310234, 1661507678, 2139353210, 1356488761,
-217689016, -1664714067, -2137965447, -1356638865, 217111899, 1665612160, 2137031214, 1357437742,
-217699415, -1665235905, -2137216489, -1357643303, 217108519, 1664677430, 2137101123, 1357639983,
-217108248, -1664677412, -2137101127, -1357639985, 217108246, 1664677410, 2137101125, 1357639983,
-217108248, -1664677412, -2137101127, -1357639985, 217108246, 1664677410, 2137101125, 1357639983,
-217108248, -1664677412, -2137101127, -1357639985, 217108246, 1664677410, 2137101125, 1357639983,
-217108248, -1664677412, -2137101127, -1357639985, 217108246, 1664677410, 2137101125, 1357639983,
-217108248, -1664677412, -2137101127, -1357639985, 217108246, 1664677410, 2137101125, 1357639983,
-217108248, -1664677412, -2137101127, -1357639985,
};

int expected_wave_88_48[] = {
0, -1, -2, 19, 87, -1105, -35018, -154862,
-182742, -117113, -15224, 148336, 136948, 222250, -50755, 51229,
-344034, 44825, -358800, 356737, -222801, 601807, -331165, 498999,
-618899, 315050, -617564, 331797, -194502, 263633, 319264, -362954,
844739, -1545966, 1889898, -3057270, 3987475, -5293472, 7438967, -9679180,
13808051, -20611886, 31213200, -62981402, 827384335, 2004668397, 2022833033, 1125968405,
-397553005, -1674207306, -2147483648, -1530178255, -167720081, 1298027150, 2100844118, 1869181659,
695640861, -816728787, -1928075774, -2073739678, -1189004893, 290722265, 1624051245, 2147483647,
1601856877, 256548191, -1216382228, -2083395132, -1911361225, -787103044, 730055555, 1882786240,
2097440691, 1266172487, -195936082, -1560628555, -2147186683, -1663552773, -350577819, 1137094142,
2057984114, 1952987945, 874369266, -640125679, -1835531116, -2115960754, -1341626055, 101482533,
1494004132, 2141794335, 1721942731, 443736925, -1055662899, -2028835774, -1990674495, -960201422,
548912839, 1784404984, 2130406751, 1414443146, -6592273, -1424341494, -2132084879, -1777026384,
-536155597, 971978080, 1995600100, 2024455062, 1044159529,
};

int expected_wave_96_48[] = {
0, 0, 2, -44, -12, 7961, 69537, 1326,
-378532, -443788, -440317, 4077, 240543, 526400, 354618, 38253,
-223337, -640621, -104426, -473222, 900101, -342336, 1465483, -1345933,
1464542, -2659977, 2275640, -3238678, 4077089, -3959248, 5554694, -5952694,
6658254, -8552239, 9062306, -11282938, 14094242, -17492597, 25972615, -51693927,
265445210, 1661125265, 2137061225, 1356966125, -215270434, -1666627214, -2134732876, -1360515313,
219612599, 1661762773, 2139894631, 1355243132, -214380845, -1666832233, -2135078838, -1359735120,
218501339, 1663113815, 2138381926, 1356849343, -216020758, -1665207952, -2136646910, -1358257733,
217137691, 1664345580, 2137292510, 1357791965, -216819757, -1664552317, -2137181821, -1357865479,
216784479, 1664449783, 2136784326, 1357410308, -217071088, -1664392182, -2136734918, -1357407344,
217071043, 1664392157, 2136734916, 1357407342, -217071045, -1664392158, -2136734919, -1357407344,
217071043, 1664392157, 2136734916, 1357407342, -217071045, -1664392158, -2136734919, -1357407344,
217071043, 1664392157, 2136734916, 1357407342,
};

int unit_test_spline() {
//    int DWORD_ALIGNED iH[3] = {0x12345678, 0x87654321, -12314516};
    int DWORD_ALIGNED iH[3] = { -12314516, 0x87654321,0x12345678};
    int           *piPhase0 = &iADFirCoefs[0][63];
    int DWORD_ALIGNED piADCoefs1[16];
    int out63[16] = {
        0xffffde68,
        0xfff9c91a,
        0x001fb061,
        0x00387050,
        0xfe1347a3,
        0x054d01a1,
        0xf7101d35,
        0xd71ca2dd,
        0xf6dd5890,
        0x05573f10,
        0xfe13e0a5,
        0x0036b73e,
        0x00205038,
        0xfff9bbd0,
        0xffffdd51,
        0x00000000        
    };
    ASRC_prepare_coefs();
    src_mrhf_spline_coeff_gen_inner_loop_asm_xs3(piPhase0, iH, piADCoefs1, FILTER_DEFS_ADFIR_PHASE_N_TAPS);
//    src_mrhf_spline_coeff_gen_inner_loop_asm(piPhase0, iH, piADCoefs2, FILTER_DEFS_ADFIR_PHASE_N_TAPS);

    int errors = 0;
    for(int i =0 ; i < 16; i++) {
        if (abs(piADCoefs1[i] - out63[i]) > 2) {
            printf("%08x should be 0x%08x,\n", piADCoefs1[i], out63[i]);
            errors++;
        }
    }

    printf("Spline done\n");
    return errors;
}



int unit_test_asrc(int test_case) {
    int input_frequency;
    int output_frequency;
    int *expected_wave;
    int steps = 100;
    int old_time = 0;
    if (test_case == 0) {
        old_time = 124751;
        input_frequency = 48000;
        output_frequency = 192000;
        expected_wave = expected_wave_48_192;   // 6 OS2 UP_N_TAPS FILTER_DEFS_ASRC_FIR_UP_ID
    } else if (test_case == 1) {
        old_time = 50695;
        input_frequency = 192000;
        output_frequency = 176400;
        expected_wave = expected_wave_192_176;  // 9 FILTER_DEFS_ASRC_FIR_UP192176_ID
    } else if (test_case == 2) {
        old_time = 163022;
        input_frequency = 192000;
        output_frequency = 44100;
        expected_wave = expected_wave_192_44;   // 10 FILTER_DEFS_ASRC_FIR_DS_ID,           1  FILTER_DEFS_ASRC_FIR_BL9644_ID
        steps = 500;
    } else if (test_case == 3) {
        old_time = 132734;
        input_frequency = 44100;
        output_frequency = 192000;
        expected_wave = expected_wave_44_192;   // 6 FILTER_DEFS_ASRC_FIR_UP_ID
    } else if (test_case == 4) {
        old_time = 79768;
        input_frequency = 176400;
        output_frequency = 96000;
        expected_wave = expected_wave_176_96;   // 5 FILTER_DEFS_ASRC_FIR_BL17696_ID
        steps = 200;
    } else if (test_case == 5) {
        old_time = 76162;
        input_frequency = 192000;
        output_frequency = 88200;
        expected_wave = expected_wave_192_88;   // 4 FILTER_DEFS_ASRC_FIR_BL19288_ID
        steps = 200;
    } else if (test_case == 6) {
        old_time = 77709;
        input_frequency = 176400;
        output_frequency = 88200;
        expected_wave = expected_wave_176_88;   // 3 FILTER_DEFS_ASRC_FIR_BLF_ID
        steps = 200;
    } else if (test_case == 7) {
        old_time = 97773;
        input_frequency = 88200;
        output_frequency = 48000;
        expected_wave = expected_wave_88_48;   // 2 FILTER_DEFS_ASRC_FIR_BL8848_ID
        steps = 200;
    } else if (test_case == 8) {
        old_time = 95707;
        input_frequency = 96000;
        output_frequency = 48000;
        expected_wave = expected_wave_96_48;   // 0 FILTER_DEFS_ASRC_FIR_BL_ID
        steps = 200;
    } else {
        printf("Bad test case %d\n", test_case);
        return 1;
    }

    asrc_state_t sASRCState[SRC_CHANNELS_PER_INSTANCE];                                   // ASRC state machine state
    int iASRCStack[SRC_CHANNELS_PER_INSTANCE][ASRC_STACK_LENGTH_MULT * SRC_N_IN_SAMPLES * 100]; // Buffer between filter stages
    asrc_ctrl_t sASRCCtrl[SRC_CHANNELS_PER_INSTANCE];                                     // Control structure
    asrc_adfir_coefs_t asrc_adfir_coefs;                                                  // Adaptive filter coefficients

    uint64_t fs_ratio = 0;
    int sine_cnt = 0;

    int inputFsCode = fs_code(input_frequency);
    int outputFsCode = fs_code(output_frequency);

    for(int ui = 0; ui < SRC_CHANNELS_PER_INSTANCE; ui++)
    {
            // Set state, stack and coefs into ctrl structure
            sASRCCtrl[ui].psState                   = &sASRCState[ui];
            sASRCCtrl[ui].piStack                   = iASRCStack[ui];
            sASRCCtrl[ui].piADCoefs                 = asrc_adfir_coefs.iASRCADFIRCoefs;
    }

    fs_ratio = asrc_init(inputFsCode, outputFsCode, sASRCCtrl, SRC_CHANNELS_PER_INSTANCE, SRC_N_IN_SAMPLES, SRC_DITHER_SETTING);

    int in_samples[4];
    int out_samples[20];
    int exp_cnt = 0;
    int errors = 0;
    int total_time = 0, t0, t1;
    for(int32_t i = 0; i < steps; i+=4) {
        for(int j = 0; j < 4; j++) {
            in_samples[j] = sine_wave[sine_cnt];
            sine_cnt++;
            if (sine_cnt == 16) sine_cnt = 0;
        }
        asm volatile("gettime %0" : "=r" (t0));
        int num_samples = asrc_process(in_samples, out_samples, fs_ratio, sASRCCtrl);
        asm volatile("gettime %0" : "=r" (t1));
        total_time += t1-t0;
        for(int j = 0; j < num_samples; j++) {
            if (abs(out_samples[j] - expected_wave[exp_cnt]) > 25) {
                printf("saw %d expected %d\n", out_samples[j], expected_wave[exp_cnt]);
                errors++;
            }
            exp_cnt++;
        }
    }
    printf("%d -> %d: %d errors time %d old_time %d %f speedup\n", input_frequency, output_frequency, errors, total_time, old_time, old_time / (float) total_time);
    return errors;
}

#define LOCAL_SRC_CHANNELS_PER_INSTANCE 2
int expected_wave_stereo_96_48[] = {
0, 0, 0, 0, 2, 2, -44, -37,
-13, -16, 7840, 6674, 69148, 60448, 3086, 14469,
-376349, -305879, -444232, -417874, -441103, -517010, -2075, -244735,
241189, -39325, 520048, 334661, 365879, 457454, 31982, 415533,
-204121, 301883, -658857, -234517, -84989, -111391, -507230, -851573,
926350, 122491, -378107, -863546, 1506932, 1092239, -1374786, -785535,
1504388, 1970249, -2684912, -1474316, 2283201, 2285156, -3240635, -2861141,
4032943, 2653571, -3882614, -4076174, 5426768, 3877011, -5732182, -4795464,
6355285, 5794299, -8102580, -5837078, 8420074, 7720952, -10395499, -8305601,
12790018, 10246969, -15550029, -13587955, 22853577, 18052027, -45338850, -38126184,
228682974, 191547051, 1614394278, 1411923721, 2146369142, 2113683591, 1404241654, 1909792245,
-146675498, 912379725, -1624207421, -479721775, -2139351553, -1659062479, -1416122080, -2147483648,
147865757, -1726325566, 1613024202, -585675069, 2146807507, 811831714, 1413883590, 1857803092,
-135324316, 2130784562, -1615094729, 1498331604, -2141579750, 242721343, -1423563336, -1121888285,
133863981, -2008190932, 1607000369, -2054277949, 2145509650, -1231094330, 1425056819, 106640333,
-125219912, 1402543183, -1605109990, 2105528115, -2144123428, 1924045026, -1430980484, 931086860,
120232522, -452761349, 1600254661, -1646767421, 2145051125, -2146997394, 1435097286, -1743673684,
-113904062, -606127871, -1596354830, 786538340, -2145303838, 1848059607, -1439648875, 2131619523,
107778362, 1517693329, 1592211159, 264865344, 2145216456, -1099745666, 1443699589, -2001399834,
-102025101, -2060446426, -1588082867, -1252066341, -2145450191, 83418700, -1448185860, 1383785439,
95958941, 2101595060, 1583988189, 1934658290, 2145721505, 953253513, 1452664479, -429459590,
-89892063, -1631375348, -1579880880, -2146501590, -2145975674, -1757976331, -1457131488, -629364061,
83824460, 764202955, 1575760941, 1836049665, 2146212685, 2134941423, 1461586846, 1535048370,
-77756191, 288918310, -1571628411, -1078842910, -2146432542, -1992424162, -1466030523, -2067218670,
71687297, -1271738837, 1567483311, 59127839, 2146635236, 1365102403, 1470462476, 2096385179,
};

int expected_wave_stereo_176_176[] = {
0, 0, 0, 0, -2, -1, 12, 10,
192, 163, -1607, -1356, -85281, -72404, -411645, -352533,
-474876, -424345, -590243, -555723, -699346, -697776, -215234, -351206,
-802525, -903444, 630229, 252507, -790703, -958854, 1397542, 893825,
-443529, -588734, 1186373, 880227, 755621, 658523, -897516, -574107,
3439216, 3221808, -5292523, -3975452, 8024117, 7282171, -11769190, -9305951,
14608967, 12769983, -19974236, -16375478, 23887599, 20233111, -35249573, -29658629,
66240376, 55401453, 898025858, 762644234, 1585729057, 1381080770, 2004627918, 1826169311,
2147483647, 2105731301, 1940857848, 2124132088, 1467164283, 1943667029, 739161478, 1526503334,
-76289715, 965298622, -898506964, 284097857, -1571827047, -417355400, -2013531758, -1079818846,
-2145934176, -1622671812, -1952761990, -1990458306, -1463950138, -2143884840, -750769484, -2063797484,
74498224, -1762167918, 890165135, -1268460559, 1569303672, -638601030, 2010714313, 60982334,
2146199643, 753667074, 1955609159, 1364896589, 1467898590, 1828545628, 756862255, 2094185866,
-69243254, 2133252089, -884826349, 1941559032, -1565941721, 1539843172, -2009080083, 971558900,
-2146897441, 298178835, -1958449620, -407455917, -1472375139, -1069015223, -762543002, -1614936667,
63173210, -1986166659, 879288962, -2142548396, 1561778680, -2067165685, 2006924044, -1768172864,
2147076060, -1277912676, 1960935750, -649417703, 1476790958, 49326290, 768217438, 742734542,
-57102502, 1355799469, -873744550, 1822204426, -1557603157, 2091497292, -2004751961, 2134548036,
-2147237510, 1946699759, -1963406204, 1548272449, -1481194981, 982364912, -773885743, 310192675,
51031336, -395533824, 868193153, -1058474510, 1553415181, -1606917548, 2002563850, -1981536577,
2147381802, -2141808201, 1965860957, -2070395444, 1485587149, -1775023207, 779547852, -1287642590,
-44959762, -660974678, -862634815, 37192399, -1549214782, 731336287, -2000359735, 1346369832,
-2147483648, 1815763421, -1968300001, 2088741664, -1489967451, 2135775867, -785203730, 1951778230,
38887828, 1556652210, 857069585, 993139501, 1545002003, 322196584, 1998139624, -383599083,
2147483647, -1047899947, 1970723299, -1598847039, 1494335830, -1976843132, 790853326, -2140999511,
-32815594, -2073558993, -851497509, -1781816786, -1540776876, -1297331316, -1995903543, -672510502,
-2147483648, 25057334, -1973130844, 719914659, -1498692269, 1336897145, -796496600, 1809264357,
26743089, 2085919239, 845918618, 2136935393, 1536539433, 1956794278, 1993651506, 1564982176,
2147483647, 1003882322, 1975522619, 334190176, 1503036716, -371652093, 802133499, -1037291892,
-20670383, -1590725417, -840332981, -1972086475, -1532289711, -2140122358, -1991383534, -2076656228,
-2147483648, -1788553371, -1977898597, -1306978542, -1507369148, -684024802, -807763984, 12921486,
14597514, 708470027, 834740619, 1327381717, 1528027735, 1802707444, 1989099638, 2083030113,
2147483647, 2138026578, 1980258759, 1961747737, 1511689526, 1573262082, 813388003, 1014593011,
-8524537, 346173057, -829141599, -359693240, -1523753554, -1026650681, -1986799850, -1582552935,
-2147483648, -1967266771, -1982603071, -2139176785, -1515997739, -2079687051, -819005348, -1795232694,
2451726, -1316583823, 823536146, -695517052, 1519467287, 785347, 1984484179, 697002733,
2147483647, 1317823726, 1984931753, 1796092772, 1520294337, 2080074364, 824616927, 2139049500,
3621967, 1966638650, -817923415, 1581491896, -1515168489, 1025271486, -1982152546, 358145089,
-2147483648, -347722673, -1987244277, -1015976463, -1524578176, -1574329716, -830221091, -1962384065,
-9694854, -2138162766, 812304714, -2082651434, 1510857895, -1801854858, 1979805165, -1326147450,
2147483647, -706987592, 1989541005, -11351245, 1528850004, 685512996, 835818835, 1308223704,
15767815, 1789420814, -806679444, 2077052114, -1506535190, 2140003861, -1977441958, 1971466417,
-2147483648, 1589670867, -1991821829, 1035916927, -1533109622, 370105442, -841409900, -335741208,
-21840663, -1005269947, 801047717, -1566056306, 1502200438, -1957438716, 1975062935, -2137080431,
2147483647, -2085549187, 1994086734, -1808419243, 1537356981, -1335668388, 846994243, -718435174,
27913331, -23487164, -795409589, 674001509, -1497853675, 1298581872, -1972668120, 1782691617,
-2147483648, 2073963471, -1996335696, 2140889871, -1541592058, 1976231244, -852571821, 1597799084,
-33985787, 1046529292, 789765093, 382053990, 1493494934, -323748979, 1970257533, -994531267,
2147483647, -1557732802, 1998568703, -1952430759, 1545814802, -2135929741, 858142583, -2088380246,
40057967, -1814925801, -784114284, -1345146619, -1489124253, -729859802, -1967831198, -35622355,
-2147483648, 662468446, -2000785730, 1288898497, -1550025200, 1775905401, -863706491, 2070808494,
-46129839, 2141707421, 778457202, 1980932873, 1484741664, 1605876221, 1965389129, 1057108203,
2147355329, 393990340, 2002986766, -311746381, 1554223206, -983760768, 869263491, -1549359469,
52201333, -1947360359, -772793895, -2134710750, -1480347207, -2091144523, -1962931345, -1821374327,
-2147207737, -1354581850, -2005171786, -741261099, -1558408785, -47756416, -874813550, 650914188,
-58272417, 1279173893, 767124408, 1769062391, 1475940910, 2067587292, 1960457870, 2142456477,
2147042973, 1985571165, 2007340777, 1613902007, 1562581903, 1067653317, 880356609, 405914093,
64343035, -299733805, -761448789, -972958803, -1471522818, -1540936584, -1957968720, -1942227673,
-2146861045, -2133423485, -2009493724, -2093841925, -1566742539, -1827764614, -885892634, -1363973763,
-70413140, -752638699, 755767078, -59888956, 1467092957, 639339110, 1955463918, 1269408380,
2146661959, 1762162800, 2011630601, 2064299969, 1570890645, 2143137021, 891421576, 1990145960,
76482684, 1621876186, -750079324, 1078164301, -1462651370, 417824871, -1952943483, -287711638,
-2146445704, -962125715, -2013751399, -1532464417, -1575026195, -1937032883, -896943399, -2132068006,
-82551618, -2096472374, 744385576, -1834096455, 1458198086, -1373322065, 1950407435, -763992235,
2146212297, -72019590, 2015856102, 627743581, 1579149155, 1259602269, 902458041, 1755206854,
88619889, 2060946639, -738685877, 2143749032, -1453733146, 1994657117, -1947855792, 1629798503,
-2145961733, 1088640804, -2017944692, 429722297, -1583259490, -275680269, -907965475, -951261862,
-94687454, -1523943249, 732980272, -1931776151, 1449256585, -2130644340, 1945288583, -2099035786,
2145694007, -1840369647, 2020017139, -1382626452, 1587357163, -775321343, 913465648, -84147918,
100754261, 616127978, -727268809, 1249755879, -1444768439, 1748194786, -1942705822, 2057527401,
-2145409133, 2144292494, -2022073443, 1999104484, -1591442150, 1637668702, -918958522, 1099082494,
-106820261, 441605974, 721551529, -263640087, 1440268737, -940367585, 1940107522, -1515373344,
2145107105, -1926457645, 2024113581, -2129152541, 1595514415, -2101532073, 924444047, -1846583988,
112885411, -1391886627, -715828485, -786625653, -1435757532, -96273561, -1937493720, 604492671,
-2144787927, 1239869525, -2026137542, 1741126813, -1599573923, 2054042361, -929922184, 2144767387,
-118949654, 2003487931, 710099713, 1645486535, 1431234844, 1109489040, 1934864431, 453475531,
2144451605, -251591474, 2028145300, -929443247, 1603620645, -1506754987, 935392879, -1921077533,
125012950, -2127592664, -704365269, -2103961153, -1426700709, -1852739279, -1932219666, -1401102290,
-2144098139, -797904810, -2030136844, -108396117, -1607654543, 592838041, -940856102, 1229943525,
-131075244, 1734003163, 698625192, 2050491649, 1422155174, 2145173692, 1929559456, 2007807311,
2143727533, 1653251748, 2032112159, 1119860103, 1611675589, 465330587, 946311802, -239534816,
137136486, -918489186, -692879532, -1498088442, -1417598270, -1915635994, -1926883827, -2125964751,
-2143339788, -2106322959, -2034071232, -1858835321, -1615683756, -1410273147, -951759938, -809158452,
-143196636, -120515203, 687128330, 581164449, 1413030033, 1219978197, 1924192786, 1726824068,
2142934904, 2046875362, 2036014040, 2145511403, 1619679001, 2012062483, 957200459, 1660964092,
149255640, 1130195355, -681371641, 477170763, -1408450501, -227470501, -1921486370, -907505755,
-2142512902, -1489374002, -2037940577, -1910133201, -1623661305, -2124268855, -962633336, -2108617411,
-155313454, -1864871924, 675609502, -1419398906, 1403859709, -820386213, 1918764589, -132630438,
2142073760, 569472279, 2039850810, 1209973857, 1627630616, 1719589755, 968058509, 2043193626,
161370020, 2145780503, -669841963, 2016253316, -1399257697, 1668623322, -1916027466, 1140494466,
-2141617499, 488995672, -2041744745, -215398915, -1631586925, -896493307, -973475949, -1480611927,
-167425297, -1904569325, 664069064, -2122505026, 1394644494, -2110844435, 1913275028, -1870848891,
2141144113, -1428479277, 2043622351, -831587741, 1635530189, -144741429, 978885604, 557761902,
173479234, 1199930832, -658290867, 1712300456, -1390020144, 2039446561, -1910507298, 2145980989,
-2140653617, 2020379676, -2045483631, 1676229193, -1639460380, 1150757109, -984287431, 500804945,
-179531788, -203320446, 652507398, -885452192, 1385384681, -1471802521, 1907724296, -1898944550,
2140145999, -2120673332, 2047328545, -2113003955, 1643377452, -1876766030, 989681389, -1437513968,
185582902, -842762676, -646718718, -156847789, -1380738141, 546033687, -1904926034, 1189849430,
-2139621276, 1704956399, -2049157100, 2035634270, -1647281397, 2146112852, -995067438, 2024441420,
-191632534, 1683781455, 640924870, 1160982942, 1376080568, 512598204, 1902112552, -191235473,
2139079448, -874382767, 2050969270, -1462946043, 1651172171, -1893259046, 1000445526, -2118773823,
197680635, -2115095910, -635125894, -1882623160, -1371411991, -1446502690, -1899283866, -853910658,
-2138520523, -168949130, -2052765047, 534288018, -1655049747, 1179729990, -1005815628, 1697557828,
-203727157, 2031756889, 629321841, 2146176089, 1366732452, 2028438439, 1896439989, 1691279885,
2137944495, 1171171655, 2054544411, 524375070, 1658914083, -179144387, 1011177676, -863285380,
209772044, -1454042789, -623512758, -1887513006, -1362041986, -2116806557, -1893580951, -2117120229,
-2137351383, -1888420087, -2056307352, -1455445154, -1662765168, -865031337, -1016531648, -181045069,
-215815263, 522525259, 617698690, 1169572822, 1357340632, 1690104971, 1890706777, 2027814540,
2136741173, 2146170696, 2058053851, 2032370587, 1666602953, 1698724225, 1021877492, 1181322919,
221856749, 536135165, -611879687, -167047574, -1352628428, -852160387, -1887817489, -1445093033,
-2136113895, -1881706609, -2059783900, -2114771610, -1670427418, -2119076848, -1027215166, -1894156622,
-227896470, -1464341079, 606055786, -876124354, 1347905407, -193135221, 1884913105, 510745793,
2135469523, 1159378246, 2061497476, 1682598069, 1674238523, 2023807339, 1032544627, 2146096673,
233934355, 2036237743, -600227047, 1706114243, -1343171616, 1191436403, -1881993660, 547878120,
-2134808097, -154945417, -2063194580, -841008143, -1678036256, -1436097072, -1037865838, -1875840039,
-239970385, -2112669027, 594393503, -2120965706, 1338427078, -1899832593, 1879059165, -1473190175,
2134129597, -887189352, 2064875186, -205219198, 1681820562, 498949999, 1043178745, 1149146608,
246004480, 1675037363, -588555211, 2019735433, -1333671851, 2145954028, -1876109650, 2040039789,
-2133434037, 1713449707, -2066539284, 1201511789, -1685591431, 559603554, -1048483322, -142838303,
-252036627, -829829010, 582712208, -1427055183, 1328905951, -1869913478, 1873145136, -2110498898,
2132721417, -2122786738, 2068186864, -1905447810, 1689348822, -1481992167, 1053779511, -898225986,
258066753, -217296614, -576864554, 487138240, -1324129436, 1138878216, -1870165644, 1667423091,
-2131991756, 2015598936, -2069817910, 2145742757, -1693092707, 2043776603, -1059067277, 1720730382,
-264094814, 1211548761, 571012283, 571311093, 1319342328, -130726624, 1867171202, -818623336,
2131245041, -1417967660, 2071432407, -1863927128, 1696823057, -2108261271, 1064346572, -2124539892,
270120765, -1911002099, -565155450, -1490746766, -1314544678, -909233894, -1864161835, -229367081,
-2130481299, 475310909, -2073030349, 1128573404, -1700539845, 1659755497, -1069617368, 2011397982,
-276144558, 2145462872, 559294093, 2047448060, 1309736516, 1727956033, 1861137560, 1221546991,
2129700513, 583000366, 2074611712, -118610757, 1704243037, -807391483, 1074879608, -1408834794,
282166141, -1857881172, -553428272, -2105956231, -1304917884, -2126225108, -1858098414, -1916495277,
-2128902713, -1499453701, -2076176498, -920212735, -1707932601, -241430212, -1080133261, 463468369,
-288185472, 1118232504, 547558022, 1652034831, 1300088820, 2007132709, 1855044407, 2145114381,
2128087884, 2051054047, 2077724679, 1735126431, 1711608515, 1231506157, 1085378268, 594670998,
294202495, -106491098, -541683394, -796133808, -1295249360, -1399656874, -1851975575, -1851775806,
-2127256049, -2103583848, -2079256252, -2127842333, -1715270751, -1921927176, -1090614612, -1508112688,
-300217170, -931162145, 535804431, -253485635, 1290399546, 451611012, 1848891935, 1107855842,
2126407202, 1644261333, 2080771204, 2002803255, 1718919265, 2144697294, 1095842227, 2054594449,
306229446, 1742241345, -529921195, 1241425949, -1285539420, 606322622, -1845793519, -94368034,
-2125541360, -784850673, -2082269523, -1390434200, -1722554049, -1845611226, -1101061088, -2101144196,
-312239272, -2129391519, 524033713, -1927297613, 1280669005, -1516723446, 1842680332, -942081784,
2124658520, -265532945, 2083751192, 439739211, 1726175051, 1097443751, 1106271140, 1636435252,
318246602, 1998409754, -518142048, 2144211626, -1275788369, 2058069150, -1839552432, 1749300544,
-2123758702, 1251306040, -2085216204, 617954852, -1729782263, -82241947, -1111472356, -773542442,
-324251391, -1381167057, 512246234, -1839387623, 1270897520, -2098637356, 1836409815, -2130872610,
2122841902, -1932606424, 2086664546, -1525285711, 1733375634, -952971299, 1116664682, -277571770,
330253583, 427853346, -506346332, 1086996567, -1265996519, 1628556850, -1833252522, 1993952349,
-2121908134, 2143657391, -2088096209, 2061478035, -1736955157, 1756303806, -1121848080, 1261146117,
-336253137, 629567323, 500442378, -70113232, 1261085389, -762209473, 1830080569, -1371855751,
2120957399, -1833105200, 2089511175, -2096063406, 1740520790, -2132285558, 1127022511, -1937853431,
342250006, -1533799197, -494534426, -963830338, -1256164186, -289601716, -1826893988, 415953800,
-2119989712, 1076514624, -2090909441, 1620626362, -1744072514, 1989431181, -1132187938, 2143034609,
-348244136, 2064820999, 488622516, 1763250909, 1251232931, 1270945872, 1823692801, 641159667,
2119005073, -57982273, 2092290986, -750852132, 1747610289, -1362500570, 1137344300, -1826764160,
354235481, -2093422425, -482706703, -2133630324, -1246291686, -1943038474, -1820477040, -1542263636,
-2118003495, -974658555, -2093655809, -301622399, -1751134099, 404040957, -1142491582, 1065998259,
-360223988, 1612644059, 476787034, 1984846399, 1241340472, 2142343294, 1817246718, 2068097939,
2116984991, 1770141625, 2095003890, 1280704977, 1754643902, 652731497, 1147629719, -45849470,
366209623, -739470785, -470863554, -1353101833, -1236379336, -1820364710, -1814001876, -2090714506,
-2115949557, -2134906860, -2096335225, -1948161375, -1758139686, -1550678753, -1152758691, -985455604,
-372192327, -313633431, 464936309, 392115199, 1231408313, 1055447809, 1810742527, 1604610184,
2114897205, 1980198144, 2097649797, 2141583470, 1761621401, 2071308737, 1157878435, 1776975723,
378172050, 1290423120, -459005351, 664282447, -1226427457, -33715205, -1807468713, -728065800,
-2113827956, -1343659828, -2098947606, -1813907049, -1765089044, -2087939733, -1162988927, -2136115121,
-384148747, -1953221972, 453070725, -1559044271, 1221436788, -996221123, 1804180440, -325634423,
2112741797, 380176911, 2100228628, 1044863618, 1768542563, 1596525009, 1168090114, 1975486571,
390122374, 2140755165, -447132482, 2074453296, -1216436364, 1783752993, -1800877752, 1300099987,
-2111638756, 675812143, -2101492862, -21579875, -1771981948, -716637542, -1173181966, -1334174862,
-396092880, -1807391387, 441190661, -2085098185, 1211426212, -2137255070, 1797560663, -1958220106,
2110518830, -1567359919, 2102740295, -1006954772, 1775407164, -337624985, 1178264432, 368226480,
402060208, 1034246031, -435245323, 1588388781, -1206406386, 1970711827, -1794229210, 2139858397,
-2109382034, 2077531506, -2103970917, 1790473205, -1778818182, 1309735257, -1183337479, 687320206,
-408024325, -9443873, 429296509, -705186387, 1201376916, -1324647249, 1790883412, -1800817936,
2108228372, -2082189961, 2105184715, -2138326663, 1782214978, -1963155597, 1188401049, -1575625426,
413985134, -1017656168, -423344354, -349604650, -1196337975, 356264445, -1787523445, 1023595603,
-2107057937, 1580201995, -2106381686, 1965874235, -1785597540, 2138893319, -1193455292, 2080543427,
-419943048, 1797136420, 417388209, 1319329029, 1191288825, 698806725, 1784148678, 2692822,
2105870446, -693712390, 2107561880, -1315077078, 1788965947, -1794186787, 1198499870, -2079215120,
425897118, -2139329912, -411429446, -1968028418, -1186230852, -1583840715, -1780760115, -1028325281,
-2104666275, -361573507, -2108725153, 344290513, -1792319910, 1012911827, -1203534827, 1571964115,
-431847899, 1960973393, 405467191, 2137859618, 1181163212, 2083488603, 1777357213, 1803741781,
2103445258, 1328880018, 2109871579, 710270219, 1795659554, 14828834, 1208560168, -682216617,
437795227, -1305465116, -399501693, -1787498414, -1176086127, -2076173863, -1773940102, -2140264752,
-2102207432, -1972838235, -2111001143, -1592005219, -1798984857, -1038961300, -1213575852, -373530521,
-443739065, 332305921, 393532996, 1002196062, 1170999630, 1563676326, 1770508807, 1956010086,
2100952790, 2136757689, 2112113831, 2086367254, 1802295772, 1810289618, 1218581836, 1338388749,
449679351, 721711258, -387561147, 26964588, -1165903769, -670698909, -1767063353, -1295811342,
-2099681359, -1780752860, -2113209632, -2073066196, -1805592282, -2141131147, -1223578086, -1977584970,
-455616055, -1600118829, 381586197, -1049564116, 1160798583, -385475613, 1763603775, 320310683,
2098393136, 991448229, 2114288537, 1555338511, 1808874357, 1950984218, 1228564547, 2135587428,
461549110, 2089179178, -375608198, 1816779567, -1155684122, 1347854687, -1760130094, 733129236,
-2097088139, 39099492, -2115350549, -659159739, -1812141976, -1286116126, -1233541198, -1773950346,
-467478485, -2069892235, 369627183, -2141929074, 1150560412, -1982268464, 1756642345, -1608181273,
2095766374, -1060133375, 2116395642, -397408388, 1815395105, 308305189, 1238507981, 980668672,
473404117, 1546950954, -363643224, 1945895952, -1145427509, 2134348866, -1753140548, 2091924304,
-2094427857, 1823211427, -2117423821, 1357277538, -1818633726, 744523780, -1243464869, 51233156,
-479325974, -647599473, 357656354, -1276379773, 1140285443, -1767091095, 1749624735, -2066652076,
2093072596, -2142658500, 2118435065, -1986888571, 1821857803, -1616192295, 1248411809, -1070668741,
485243991, -409328462, -351666623, 296289828, -1135134266, 969857756, -1746094933, 1538513919,
-2091700602, 1940745458, -2119429381, 2133042049, -1825067320, 2094602528, -1253348781, 1829584986,
-491158142, 1366656987, 345674072, 755894521, 1129974010, 63365189, 1742551174, -636018498,
2090311880, -1266602586, 2120406751, -1760175336, 1828262245, -2063345826, 1258275725, -2143319408,
497068353, -1991445145, -339678769, -1624151636, -1124804718, -1081169874, -1738993479, -421235456,
-2088906458, 284264983, -2121367173, 959015817, -1831442557, 1530027682, -1263192615, 1935532897,
-502974603, 2131667029, 333680736, 2097213777, 1119626432, 1835900040, 1735421888, 1375992731,
2087484327, 767241094, 2122310630, 75495203, 1834608228, -624417177, 1268099402, -1256784909,
508876824, -1753203290, -327680049, -2059973597, -1114439205, -2143911778, -1731836421, -1995938039,
-2086045513, -1632059048, -2123237127, -1091636439, -1837759231, -433128976, -1272996061, 272231051,
-514774981, 948143211, 321676731, 1521492521, 1109243057, 1930258448, 1728237113, 2130223839,
2084590022, 2099757963, 2124146643, 1842156386, 1840895539, 1385284479, 1277882529, 778563127,
520669017, 87622799, -315670856, -612795896, -1104038046, -1246927027, -1724623982, -1746175174,
-2083117865, -2056535494, -2125039183, -2144435590, -1844017133, -2000367105, -1282758796, -1639914268,
-526558896, -1102068090, 309662442, -445008650, 1098824211, 260188412, 1720997066, 937240289,
2081629054, 1512908706, 2125914732, 1924922271, 1847123987, 2128712524, 1287624797, 2102234999,
532444563, 1848353825, -303651566, 1394531926, -1093601593, 789860273, -1717356392, 99747595,
-2080123606, -601155013, -2126773286, -1237029284, -1850216074, -1739091226, -1292480507, -2053031628,
-538325979, -2144890827, 297638251, -2004732210, 1088370227, -1647717044, 1713701986, -1112464500,
2078601516, -456874089, 2127614838, 248137456, 1853293362, 926307395, 1297325884, 1504276509,
544203081, 1919524539, -291622565, 2127133140, -1083130170, 2104644809, -1710033888, 1854492158,
-2077062821, 1403734781, -2128439385, 801132156, -1856355842, 111869192, -1302160894, -589494909,
-550075838, -1227091983, 285604541, -1731951663, 1077881447, -2049462114, 1706352111, -2145277477,
2075507517, -2009033200, 2129246907, -1655467130, 1859403478, -1122825335, 1306985486, -468724918,
555944194, 236078566, -279584243, 915344881, -1072624110, 1495596216, -1702656700, 1914065428,
-2073935624, 2125485741, -2130037416, 2106987323, -1862436249, 1860571181, -1311799631, 1412892741,
-561808109, 812378413, 273561703, 123987219, 1067358193, -577815960, 1698947676, -1217115444,
2072347144, -1724756719, 2130810893, -2045827063, 1865454137, -2145595531, 1316603293, -2013269950,
567667531, -1663164282, -267536983, -1133150264, -1062083751, -480560754, -1695225070, 224012130,
-2070742108, 904353099, -2131567333, 1486868098, -1868457108, 1908545116, -1321396430, 2123770376,
-573522411, 2109262459, 261510116, 1866590720, 1056800812, 1422005529, 1691488912, 823598696,
2069120512, 136101275, 2132306736, -566118536, 1871445137, -1207099987, 1326178996, -1717506633,
579372709, -2042126587, -255481167, -2145844974, -1051509432, -2017442318, -1687739229, -1670808240,
-2067482379, -1143438961, -2133029091, -492381223, -1874418210, 211938534, -1330950968, 893332405,
-585218375, 1478092435, 249450170, 1902963766, 1046209640, 2121987095, 1683976055, 2111470143,
2065827705, 1872550559, 2133734396, 1431072840, 1877376295, 834792638, 1335712292, 148210977,
591059364, -554403006, -243417186, -1197045932, -1040901488, -1710201616, -1680199424, -2038360819,
-2064156537, -2146025800, -2134422642, -2021550176, -1880319372, -1678398774, -1340462946, -1153691086,
-596895623, -504185945, 237382249, 199858159, 1035585014, 882283142, 1676409355, 1469269505,
2062468846, 1897321571, 2135093818, 2120135955, 1883247417, 2113610311, 1345202873, 1878450523,
602727115, 1440094386, -231345421, 845959885, -1030260259, 160315940, -1672605886, -542669753,
-2060764679, -1186953597, -2135747938, -1702841915, -1886160410, -2034529863, -1349932054, -2146137994,
-608553787, -2025593388, 225306736, -1685935639, 1024927270, -1163906317, 1668789043, -515974545,
2059044035, 187771395, 2136384974, 871205669, 1889058318, 1460399597, 1354650441, 1891618711,
614375587, 2118217031, -219266258, 2115682891, -1019586090, 1884290419, -1664958867, 1449069877,
-2057306933, 857100078, -2137004938, 172415772, -1891941133, -530919150, -1359357998, -1176823310,
-620192485, -1695427763, 213224020, -2030633853, 1014236754, -2146181563, 1661115373, -2029571826,
2055553380, -1693418591, 2137607810, -1174084331, 1894808813, -527746641, 1364054691, 175678625,
626004419, 860100331, -207180085, 1451482986, -1008879315, 1885855351, -1657258605, 2116230363,
-2053783395, 2117687817, -2138193600, 1890070059, -1897661352, 1457999033, -1368740473, 868212867,
-631811350, 184510093, 201134485, -519151564, 1003513806, -1166655392, 1653388581, -1687959394,
2051996989, -2026672905, 2138762292, -2146156503, 1900498717, -2033485362, 1373415314, -1700847392,
637613230, -1184224800, -195087286, -539501863, -998140281, 163580247, -1649505347, 848967496,
-2050194188, 1442519962,
};

int unit_test_asrc_stereo(int test_case) {
    int input_frequency;
    int output_frequency;
    int *expected_wave;
    int steps = 100;
    int old_time = 0;
    if (test_case == 8) {
        old_time = 172686;
        input_frequency = 96000;
        output_frequency = 48000;
        expected_wave = (int *)expected_wave_stereo_96_48;   // 0 FILTER_DEFS_ASRC_FIR_BL_ID
        steps = 200;
    } else if (test_case == 9) {
        old_time = 866676;
        input_frequency = 176400;
        output_frequency = 176400;
        expected_wave = (int *)expected_wave_stereo_176_176;
        steps = 1000;
    } else {
        printf("Bad test case %d\n", test_case);
        return 1;
    }

    asrc_state_t sASRCState[LOCAL_SRC_CHANNELS_PER_INSTANCE];                                   // ASRC state machine state
    int iASRCStack[LOCAL_SRC_CHANNELS_PER_INSTANCE][ASRC_STACK_LENGTH_MULT * SRC_N_IN_SAMPLES * 100]; // Buffer between filter stages
    asrc_ctrl_t sASRCCtrl[LOCAL_SRC_CHANNELS_PER_INSTANCE];                                     // Control structure
    asrc_adfir_coefs_t asrc_adfir_coefs;                                                  // Adaptive filter coefficients

    uint64_t fs_ratio = 0;
    int sine_cnt = 0;
    int sine2_cnt = 0;

    int inputFsCode = fs_code(input_frequency);
    int outputFsCode = fs_code(output_frequency);

    for(int ui = 0; ui < LOCAL_SRC_CHANNELS_PER_INSTANCE; ui++)
    {
            // Set state, stack and coefs into ctrl structure
            sASRCCtrl[ui].psState                   = &sASRCState[ui];
            sASRCCtrl[ui].piStack                   = iASRCStack[ui];
            sASRCCtrl[ui].piADCoefs                 = asrc_adfir_coefs.iASRCADFIRCoefs;
    }

    fs_ratio = asrc_init(inputFsCode, outputFsCode, sASRCCtrl, LOCAL_SRC_CHANNELS_PER_INSTANCE, SRC_N_IN_SAMPLES, SRC_DITHER_SETTING);
    fs_ratio *= 0.9991;
    int in_samples[4][LOCAL_SRC_CHANNELS_PER_INSTANCE];
    int out_samples[20][LOCAL_SRC_CHANNELS_PER_INSTANCE];
    int exp_cnt = 0;
    int errors = 0;
    int total_time = 0, t0, t1;
    for(int32_t i = 0; i < steps; i+=4) {
        for(int j = 0; j < 4; j++) {
            in_samples[j][0] = sine_wave[sine_cnt];
            sine_cnt++;
            if (sine_cnt == 16) sine_cnt = 0;
            in_samples[j][1] = sine2_wave[sine2_cnt];
            sine2_cnt++;
            if (sine2_cnt == 19) sine2_cnt = 0;
        }
        asm volatile("gettime %0" : "=r" (t0));
        int num_samples = asrc_process((int *)in_samples, (int *) out_samples, fs_ratio, sASRCCtrl);
        asm volatile("gettime %0" : "=r" (t1));
        total_time += t1-t0;
        for(int j = 0; j < num_samples; j++) {
            for(int k = 0; k < 2; k++) {
                int idx = exp_cnt*LOCAL_SRC_CHANNELS_PER_INSTANCE+k;
                if (abs(out_samples[j][k] - expected_wave[idx]) > 25) {
                    printf("saw %d expected %d diff %d\n", out_samples[j][k], expected_wave[idx], out_samples[j][k] - expected_wave[idx]);
                    errors++;
                }
            }
            exp_cnt++;
        }
    }
    printf("%d -> %d stereo: %d errors time %d old_time %d %f speedup\n", input_frequency, output_frequency, errors, total_time, old_time, old_time / (float) total_time);
    return errors;
}

#define ONE  0x1000000

int unit_test_fir() {
    int        DWORD_ALIGNED piData[64]  = {
        1*ONE, 2*ONE, 4*ONE, 8*ONE, -1*ONE, -8*ONE, -4*ONE, -9*ONE+6,
        1*ONE, 2*ONE, 4*ONE, 8*ONE, -4*ONE, -8*ONE, -4*ONE+3, -9*ONE,
        1*ONE, 2*ONE, 4*ONE, 8*ONE, -1*ONE, -1*ONE-12345, -4*ONE, -9*ONE,
        1*ONE, 2*ONE, 4*ONE, 8*ONE, -1*ONE, -8*ONE, -5*ONE, -9*ONE,
        4*ONE, 2*ONE, 4*ONE, 8*ONE, -1*ONE, -8*ONE, -4*ONE, -9*ONE,
        1*ONE, -3*ONE, 4*ONE, 8*ONE, -4*ONE, -8*ONE, -4*ONE, -9*ONE,
        1*ONE, 2*ONE, 1*ONE, 8*ONE, -1*ONE, -1*ONE, -4*ONE, -9*ONE,
        1*ONE, 2*ONE, 4*ONE, -7*ONE, -1*ONE, -8*ONE, -5*ONE, -9*ONE,
    };
    int        DWORD_ALIGNED piCoefs[64] = {
        ONE, 2*ONE,
        -1*ONE, -3*ONE,
        5*ONE, 7*ONE,
        8*ONE, -7*ONE,
        ONE, 2*ONE,
        -1*ONE, -3*ONE,
        5*ONE, 7*ONE,
        8*ONE, -9*ONE,

        ONE, 5*ONE,
        -5*ONE, -3*ONE,
        5*ONE, 7*ONE,
        8*ONE, -7*ONE,
        ONE, 2*ONE,
        -1*ONE, -3*ONE,
        5*ONE, 7*ONE,
        8*ONE, -8*ONE,

        ONE, 5*ONE,
        -5*ONE, -3*ONE,
        5*ONE, 7*ONE,
        8*ONE, -7*ONE,
        ONE, 2*ONE,
        -1*ONE, -3*ONE,
        6*ONE, 1*ONE,
        9*ONE, -8*ONE,
    
        ONE, 5*ONE,
        -5*ONE, -3*ONE,
        5*ONE, 7*ONE,
        2*ONE, -7*ONE,
        ONE, 2*ONE,
        -1*ONE, -3*ONE,
        5*ONE, 7*ONE,
        8*ONE, -1*ONE
    };
    int        DWORD_ALIGNED piCoefs2[64] = {
        ONE, -1*ONE, 5*ONE, 8*ONE, ONE, -1*ONE, 5*ONE, 8*ONE,
        2*ONE, -3*ONE, 7*ONE, -7*ONE, 2*ONE, -3*ONE, 7*ONE,  -9*ONE,
        ONE, -5*ONE, 5*ONE, 8*ONE, ONE, -1*ONE, 5*ONE, 8*ONE,
        5*ONE, -3*ONE, 7*ONE, -7*ONE, 2*ONE, -3*ONE, 7*ONE,  -8*ONE,
        ONE, -5*ONE, 5*ONE, 8*ONE, ONE, -1*ONE, 6*ONE, 9*ONE,
        5*ONE, -3*ONE, 7*ONE, -7*ONE, 2*ONE, -3*ONE, 1*ONE,  -8*ONE,
        ONE, -5*ONE, 5*ONE, 2*ONE, ONE, -1*ONE, 5*ONE, 8*ONE,
        5*ONE, -3*ONE, 7*ONE, -7*ONE, 2*ONE, -3*ONE, 7*ONE,  -1*ONE,
    };
    for(int i = 0; i < 64; i++) {
        int o = (i&1) * 8 + (i&~15) + (i&14)/2;
//        printf("%08x  %08x\n", piCoefs
        piCoefs2[o] = piCoefs[i];
    }
    int        DWORD_ALIGNED        iData1[2];
    int        DWORD_ALIGNED        iData2[2];

    iData1[0] = 0x1234;
    iData1[1] = 0x2234;
    iData2[0] = 0x3234;
    iData2[1] = 0x4234;
    int t0, t1, t2, t3;
    int errors = 0;
    asm volatile("gettime %0" : "=r" (t0));
    src_mrhf_fir_os_inner_loop_asm(piData, piCoefs, iData1, 16);
    asm volatile("gettime %0" : "=r" (t1));
    printf("Fir os inner loop: time old %d  ", t1-t0);
    asm volatile("gettime %0" : "=r" (t2));
    src_mrhf_fir_os_inner_loop_asm_xs3(piData, piCoefs2, iData2, 16);
    asm volatile("gettime %0" : "=r" (t3));
    printf(" new %d speedup %f\n", t3-t2, (t1-t0)/(float)(t3-t2));
    if (abs(iData1[0] - iData2[0]) > 2 || abs(iData1[1] - iData2[1]) > 2) {
        printf("expected %08x %08x ", iData1[0], iData1[1]);
        printf("saw %08x %08x\n", iData2[0], iData2[1]);
        errors++;
    }
    asm volatile("gettime %0" : "=r" (t0));
    src_mrhf_fir_inner_loop_asm(piData, piCoefs, iData1, 32);
    asm volatile("gettime %0" : "=r" (t1));
    printf("fir_inner_loop time old %d    ", t1-t0);
    asm volatile("gettime %0" : "=r" (t2));
    src_mrhf_fir_inner_loop_asm_xs3(piData, piCoefs, iData2, 32);
    asm volatile("gettime %0" : "=r" (t3));
    printf(" new %d speedup %f\n", t3-t2, (t1-t0)/(float)(t3-t2));
    if (abs(iData1[0] - iData2[0]) > 2) {
        printf("expected %08x", iData1[0]);
        printf("  saw %08x \n", iData2[0]);
        errors++;
    }

    asm volatile("gettime %0" : "=r" (t0));
    src_mrhf_adfir_inner_loop_asm(piData, piCoefs, iData1, 32);
    asm volatile("gettime %0" : "=r" (t1));
    printf("adfir_inner_loop time old %d   ", t1-t0);
    asm volatile("gettime %0" : "=r" (t2));
    src_mrhf_adfir_inner_loop_asm_xs3(piData, piCoefs, iData2, 32);
    asm volatile("gettime %0" : "=r" (t3));
    printf(" new %d speedup %f\n", t3-t2, (t1-t0)/(float)(t3-t2));
    if (abs(iData1[0] - iData2[0]) > 1) {
        printf("expected %08x", iData1[0]);
        printf("  saw %08x \n", iData2[0]);
        errors++;
    }
    return errors;
}


int main(void) {
    int errors = 0;
//    errors += unit_test_asrc(8);
//    return errors;
    errors += unit_test_spline();
    errors += unit_test_fir();
    errors += unit_test_asrc(0);
    errors += unit_test_asrc(1);
    errors += unit_test_asrc(2);
    errors += unit_test_asrc(3);
    errors += unit_test_asrc(4);
    errors += unit_test_asrc(5);
    errors += unit_test_asrc(6);
    errors += unit_test_asrc(7);
    errors += unit_test_asrc(8);
    errors += unit_test_asrc_stereo(8);
    errors += unit_test_asrc_stereo(9);

    if (errors == 0) {
        printf("PASS\n");
    } else {
        printf("FAIL: %d errors\n", errors);
    }

}


