# Copyright 2023-2024 XMOS LIMITED.
# This Software is subject to the terms of the XMOS Public Licence: Version 1.
"""
This script generates filter coefficient for a 48 - 16 kHz fixed-factor-of-3 sample rate conversion
"""
import argparse
import numpy as np
from scipy import signal
import matplotlib.pyplot as plt
from pathlib import Path

fs = 48000.0
NUM_PHASES = 3
NUM_TAPS_PER_PHASE = 32

def test_bounds(Y, F, freq, min, max):
    # This will find the closest frequency we can get and test the responce
    idx = (np.abs(F - freq)).argmin()
    #print(f"{Y[idx]} dB at {F[idx]}")
    assert Y[idx] < max
    if min != None:
        assert Y[idx] > min


def test_filter(taps_fl):
    # This function will check that the shape of the filter is appropriate
    w, h = signal.freqz(taps_fl)
    Y = 20 * np.log10(np.abs(h))
    F = 0.5 * fs * w / np.pi

    test_bounds(Y, F, 10500,  None, -6  )
    test_bounds(Y, F, 7500,  -2,    0.05)
    test_bounds(Y, F, 6300,  -0.05, 0.05)
    test_bounds(Y, F, 5000,  -0.05, 0.05)
    test_bounds(Y, F, 200,   -0.05, 0.05)
    test_bounds(Y, F, 180,   -0.05, 0.05)
    test_bounds(Y, F, 100,   -0.05, 0.05)

def gen_coefs(num_taps_per_phase = NUM_TAPS_PER_PHASE, num_phases = NUM_PHASES):
    """
    Get 8 kHz low pass filter coefficients for the 48 - 16 kHz fixed-factor-of-3 polyphase filtering

    Returns:
        taps[num_phases * num_taps_per_phase] in float for plotting and debug

        taps[num_phases * num_taps_per_phase] in int32 for debug and testing

        taps[num_phases][num_taps_per_phase]  in int32 for polyphase implementation
    """
    taps_fl = signal.firwin2((num_phases * num_taps_per_phase), [0, 7600, 8400, 0.5 * fs], [1, 1, 0, 0], window=("kaiser", 4), fs=fs)

    test_filter(taps_fl)

    mixed_taps_fl = np.zeros([num_phases, num_taps_per_phase], np.float32)
    phase = 0
    for step in range(num_phases - 1, -1, -1):
        tap = 0
        for j in range(step, len(taps_fl), num_phases):
            mixed_taps_fl[phase][tap] = taps_fl[j]
            tap += 1
        phase += 1

    taps_int = np.zeros(len(taps_fl), np.int32)
    for i in range(len(taps_int)): taps_int[i] = (taps_fl[i] * 2 ** 31).astype(np.int32)

    mixed_taps_int = np.zeros([num_phases, num_taps_per_phase], np.int32)
    for ph in range(num_phases): mixed_taps_int[ph] = (mixed_taps_fl[ph] * 2 ** 30).astype(np.int32)

    return taps_fl, taps_int, mixed_taps_int

def plot_response(taps, passband = False, freq_domain = True):
    if passband and not freq_domain: return
    plt.figure()
    if freq_domain:
        w, h = signal.freqz(taps)
        #plt.plot(np.unwrap(np.angle(h)))
        plt.plot(0.5 * fs * w / np.pi, 20 * np.log10(np.abs(h)))
        plt.xlabel('Frequency (Hz)')
        plt.ylabel('Gain (dB)')
    else:
        plt.plot(taps)
        plt.xlabel('Time')
        plt.ylabel('Amplitude')

    if passband:
        plt.ylim(-0.2, 0.2)
        if freq_domain: plt.xlim(0, 0.25 * fs)
        title = "lpf_ff3_pb"
    else:
        if freq_domain: plt.xlim(0, 0.5 * fs)
        title = "lpf_ff3"
    plt.grid(True)
    plt.title(title)
    #plt.show()
    fig = plt.gcf()
    title += ".png"
    title = Path(__file__).parent / title
    fig.savefig(title, dpi = 200)

def generate_header_file(output_path, num_taps_per_phase = NUM_TAPS_PER_PHASE,
                         num_phases = NUM_PHASES, name=None):
    header_template = """\
// Copyright 2023 XMOS LIMITED.
// This Software is subject to the terms of the XCORE VocalFusion Licence.

/*********************************/
/* AUTOGENERATED. DO NOT MODIFY! */
/*********************************/

// Use src_ff3_fir_gen.py script to regenare this file
// python src_ff3_fir_gen.py -o <output_path> -gc True -ntp 32 -np 3

#ifndef _SRC_FF3_COEFS_H_
#define _SRC_FF3_COEFS_H_

#include <stdint.h>

#ifndef ALIGNMENT
#  ifdef __xcore__
#    define ALIGNMENT(N)  __attribute__((aligned (N)))
#  else
#    define ALIGNMENT(N)
#  endif
#endif

#define %(name_up)s_NUM_PHASES (%(phases)s)
#define %(name_up)s_TAPS_PER_PHASE (%(taps_per_phase)s)

/** q31 coefficients to use for debugging ff3 sample rate conversion */
extern const int32_t %(name)s_coefs_debug[%(name_up)s_NUM_PHASES * %(name_up)s_TAPS_PER_PHASE];

/** q30 coefficients to use for the ff3 48 - 16 kHz polyphase FIR filtering */
extern const int32_t %(name)s_coefs[%(name_up)s_NUM_PHASES][%(name_up)s_TAPS_PER_PHASE];

#endif // _SRC_FF3_COEFS_H_

"""
    filename = "src_ff3_fir_coefs.h" if name is None else name + "_coefs.h"
    header_path = Path(output_path) / filename
    with open(header_path, "w") as header_file:
        header_file.writelines(header_template % {  'name_up': name.upper(),
                                                    'name': name.rstrip(".h"),
                                                    'taps_per_phase':num_taps_per_phase,
                                                    'phases':num_phases})

def generate_c_file(output_path, taps, mixed_taps, num_taps_per_phase = NUM_TAPS_PER_PHASE,
                    num_phases = NUM_PHASES, name=None):
    c_template = """\
// Copyright 2023 XMOS LIMITED.
// This Software is subject to the terms of the XCORE VocalFusion Licence.

/*********************************/
/* AUTOGENERATED. DO NOT MODIFY! */
/*********************************/

// Use src_ff3_fir_gen.py script to regenare this file
// python src_ff3_fir_gen.py -o <output_path> -gc True -ntp 32 -np 3

#include "%(name)s_coefs.h"
#include <stdint.h>

/** q31 coefficients to use for debugging ff3 sample rate conversion */
const int32_t ALIGNMENT(8) %(name)s_coefs_debug[%(name_up)s_NUM_PHASES * %(name_up)s_TAPS_PER_PHASE] = {
%(coefs_debug)s
};

/** q30 coefficients to use for the ff3 48 - 16 kHz polyphase FIR filtering */
const int32_t ALIGNMENT(8) %(name)s_coefs[%(name_up)s_NUM_PHASES][%(name_up)s_TAPS_PER_PHASE] = {
%(coefs)s};

"""

    coefs_debug = ''

    for tap in range(len(taps)):
        coefs_debug += ' ' + str(taps[tap]).rjust(12) + ','
        if(((tap + 1) % 6) == 0):
            coefs_debug += '\n'

    coefs = ''

    for phase in range(num_phases):
        coefs += '    {\n    '
        for tap in range(num_taps_per_phase):
            coefs += ' ' + str(mixed_taps[phase][tap]).rjust(12)  + ','
            if(((tap + 1) % 6) == 0):
                coefs += '\n    '
        coefs += '},\n'

    filename = "src_ff3_fir_coefs.c" if name is None else name + "_coefs.c"
    c_path = Path(output_path) / filename

    with open(c_path, "w") as c_file:
        c_file.writelines(c_template % {    'name_up': name.upper(),
                                            'name': name,
                                            'coefs_debug':coefs_debug,
                                            'coefs':coefs})

if __name__ == "__main__":
    parser = argparse.ArgumentParser("Generate FIR coefficiens for a 48 - 16 kHz polyphase SRC")
    parser.add_argument('--output_dir','-o', help='output path for filter files')
    parser.add_argument('--gen_c_files','-gc', help='Generate .h and .c files', action='store_true')
    parser.add_argument('--name','-n', help='Name of the generated files', default=None)
    parser.add_argument('--gen_plots', '-gp', help='Generate .png files', action='store_true')
    parser.add_argument('--num_taps_per_phase', '-ntp', help='Number of filter taps per phase', default=NUM_TAPS_PER_PHASE, type=int)
    parser.add_argument('--num_phases', '-np', help='number of phases', default=NUM_PHASES, type=int)
    args = parser.parse_args()

    print(f"Running In src_ff3_fir_gen.py. name = {args.name}, num_taps_per_phase = {args.num_taps_per_phase}")

    taps_fl, taps, mixed_taps = gen_coefs(args.num_taps_per_phase, args.num_phases)

    if args.gen_c_files:
        Path(args.output_dir).mkdir(exist_ok=True, parents=True)
        generate_header_file(args.output_dir, args.num_taps_per_phase, args.num_phases, name=args.name)
        generate_c_file(args.output_dir, taps, mixed_taps, args.num_taps_per_phase, args.num_phases, name=args.name)
    if args.gen_plots:
        plot_response(taps_fl, False)
        plot_response(taps_fl, True)
