// Copyright 2024-2025 XMOS LIMITED.
// This Software is subject to the terms of the XMOS Public Licence: Version 1.
#include <string.h>
#include <stdlib.h>
#include <xcore/assert.h>
#include <debug_print.h>
#include "cmds.h" // Autogenerated
#include "cmd_offsets.h" // Autogenerated
#include "stages/crossfader.h"
#include <stdio.h>
#include "print.h"
#include "control/signal_chain.h"

void crossfader_process(int32_t **input, int32_t **output, void *app_data_state)
{
    crossfader_state_t *state = app_data_state;

    int32_t *in1 = input[0];
    int32_t *in2 = input[1];
    int32_t *out = output[0];
    int j = 0;
    do
    {
        *out++ = adsp_crossfader(*in1++, *in2++, state->gains[0], state->gains[1], 31);
    } while (++j < state->frame_size);
}


void crossfader_init(module_instance_t* instance, adsp_bump_allocator_t* allocator, uint8_t id, int n_inputs, int n_outputs, int frame_size)
{
    crossfader_state_t *state = instance->state;
    crossfader_config_t *config = instance->control.config;

    memset(state, 0, sizeof(crossfader_state_t));
    state->n_inputs = n_inputs;
    xassert(n_inputs == 2 && "crossfader must have 2 inputs");

    state->frame_size = frame_size;
    xassert(n_outputs == 1 && "crossfader should only have one outputs");
    state->n_outputs = n_outputs;

    memcpy(&state->config, config, sizeof(crossfader_config_t));
    adsp_crossfader_mix(state->gains, config->mix);

}

void crossfader_control(void *module_state, module_control_t *control)
{
    crossfader_state_t *state = module_state;
    crossfader_config_t *config = control->config;

    if(control->config_rw_state == config_write_pending)
    {
        // Finish the write by updating the working copy with the new config
        memcpy(&state->config, config, sizeof(crossfader_config_t));
        control->config_rw_state = config_none_pending;
        adsp_crossfader_mix(state->gains, config->mix);
    }
    else if(control->config_rw_state == config_read_pending)
    {
        memcpy(config, &state->config, sizeof(crossfader_config_t));
        control->config_rw_state = config_read_updated;
    }
    else
    {
        // nothing to do.
    }
}

