// Copyright 2025 XMOS LIMITED.
// This Software is subject to the terms of the XMOS Public Licence: Version 1.

#include <platform.h>
#include <xs1.h>
#include "config.h"
#include "i2c.h"
#include "i2s.h"
#include "xk_evk_xu316/board.h"

#if (ADD_AI_PROCESSING == 1)
#include "some_ai.hpp"
#include "some_gpo.h"
#endif

#if (ADD_DSP_PROCESSING == 1)
#include "some_dsp.h"
#include "some_gpi.h"
#endif

// GPIO resources
#if (ADD_AI_PROCESSING == 1)
on tile[0]: out port p_leds = PORT_LEDS;
#endif

#if (ADD_DSP_PROCESSING == 1)
on tile[0]: in port p_buttons = PORT_BUTTONS;
#endif

// I2S resources
on tile[1]: in port p_mclk =                                PORT_MCLK_IN;
on tile[1]: buffered out port:32 p_lrclk =                  PORT_I2S_LRCLK;
on tile[1]: out port p_bclk =                               PORT_I2S_BCLK;
on tile[1]: buffered out port:32 p_dac[NUM_I2S_LINES] =     {PORT_I2S_DAC_DATA};
on tile[1]: buffered in port:32 p_adc[NUM_I2S_LINES] =      {PORT_I2S_ADC_DATA};
on tile[1]: clock bclk =                                    XS1_CLKBLK_1;

// Board configuration from lib_board_support
static const xk_evk_xu316_config_t hw_config = {
        MASTER_CLOCK_FREQUENCY
};

void application(
    #if (ADD_AI_PROCESSING == 1)
        chanend c_ai,
    #endif
    #if (ADD_DSP_PROCESSING == 1)
        chanend c_dsp,
    #endif
    server i2s_frame_callback_if i_i2s
)
{
    xk_evk_xu316_AudioHwInit(hw_config);

    // Array used for looping back samples
    int32_t samples[BUFFER_SIZE] = {0};

    #if (ADD_DSP_PROCESSING == 1)
        size_t idx = 0;
        int32_t sample;
    #endif

    while (1) {
    select {
        case i_i2s.init(i2s_config_t &?i2s_config, tdm_config_t &?tdm_config):
            i2s_config.mode = I2S_MODE_I2S;
            i2s_config.mclk_bclk_ratio = (MASTER_CLOCK_FREQUENCY / (SAMPLE_FREQUENCY * CHANS_PER_FRAME * DATA_BITS));
            xk_evk_xu316_AudioHwConfig(SAMPLE_FREQUENCY, MASTER_CLOCK_FREQUENCY, 0, DATA_BITS, DATA_BITS);
            break;

        case i_i2s.receive(size_t n_chans, int32_t in_samps[n_chans]):
            for (size_t i = 0; i < n_chans; ++i){
                #if (ADD_AI_PROCESSING == 1)
                    // Send samples for AI processing
                    c_ai <: in_samps[i];
                #endif

                #if (ADD_DSP_PROCESSING == 1)
                    // Send samples to DSP processing
                    c_dsp <: in_samps[i];
                #else
                    samples[i] = in_samps[i]; // copy samples for loopback
                #endif
            }
            break;

        #if (ADD_DSP_PROCESSING == 1)
        // Receive processed samples from DSP
        case c_dsp :> sample:
            samples[idx] = sample;
            if (BUFFER_SIZE <= ++idx){
                idx = 0;
            }
            break;
        #endif

        case i_i2s.send(size_t n_chans, int32_t out_samps[n_chans]):
            for (size_t i = 0; i < n_chans; ++i){
                out_samps[i] = samples[i]; // loopback samples
            }
            break;

        case i_i2s.restart_check() -> i2s_restart_t restart:
            restart = I2S_NO_RESTART; // Keep on looping
            break;
        } // End select
    } // End while (1)
} // End application


int main(void)
{
    chan c_audio_hw_init; // Channel for cross-tile communication

    #if (ADD_AI_PROCESSING == 1)
        chan c_ai;  // Channel for application <-> AI communication
        chan c_gpo; // Channel for LED events
    #endif

    #if (ADD_DSP_PROCESSING == 1)
        chan c_dsp; // Channel for application <-> DSP communication
        chan c_gpi; // Channel for button press events
    #endif

    par {
        on tile[0]: {
            par {
                xk_evk_xu316_AudioHwRemote(c_audio_hw_init);

                #if (ADD_AI_PROCESSING == 1)
                    some_gpo(c_gpo, p_leds);
                #endif

                #if (ADD_DSP_PROCESSING == 1)
                    some_gpi(c_gpi, p_buttons);
                #endif
            } // Inner par
        } // tile[0]

        on tile[1]: {
            interface i2s_frame_callback_if i_i2s;

            xk_evk_xu316_AudioHwChanInit(c_audio_hw_init);

            par {
                application(
                    #if (ADD_AI_PROCESSING == 1)
                        c_ai,
                    #endif
                    #if (ADD_DSP_PROCESSING == 1)
                        c_dsp,
                    #endif
                    i_i2s);

                i2s_frame_master(
                    i_i2s,
                    p_dac,
                    NUM_I2S_LINES,
                    p_adc,
                    NUM_I2S_LINES,
                    DATA_BITS,
                    p_bclk,
                    p_lrclk,
                    p_mclk,
                    bclk);

                #if (ADD_AI_PROCESSING == 1)
                    some_ai(c_ai, c_gpo);
                #endif

                #if (ADD_DSP_PROCESSING == 1)
                    some_dsp(c_dsp, c_gpi);
                #endif
            } // Inner par
        } // tile[1]
    } // Outer par
    return 0;
}
