#include <string.h>
#include "downsample.h"
#include "xmath/filter.h"

#define US_COEFFICIENTS  31

// start:coefficients
#define CONVERT_FP(x)  ((int)((2*x) * (1<<30)))

int32_t us_coefficients_phase0[(US_COEFFICIENTS+1)/2] = {
    CONVERT_FP(0.0001144368956127061),
    CONVERT_FP(0.0015503056579715265),
    CONVERT_FP(0.004441310178041441),
    CONVERT_FP(-0.004624415198892449),
    CONVERT_FP(-0.028890699902555182),
    CONVERT_FP(0.005899831447759155),
    CONVERT_FP(0.14890438350333784),
    CONVERT_FP(0.24348744648423895),
    CONVERT_FP(0.14890438350333784),
    CONVERT_FP(0.005899831447759155),
    CONVERT_FP(-0.028890699902555182),
    CONVERT_FP(-0.004624415198892449),
    CONVERT_FP(0.004441310178041441),
    CONVERT_FP(0.0015503056579715265),
    CONVERT_FP(0.0001144368956127061),
    0
};

int32_t us_coefficients_phase1[(US_COEFFICIENTS+1)/2] = {
    CONVERT_FP(0.000012655763070268956),
    CONVERT_FP(0.0005222575167540901),
    CONVERT_FP(0.003206743194403911),
    CONVERT_FP(0.0027511574458392177),
    CONVERT_FP(-0.0174382058234831),
    CONVERT_FP(-0.025632812530463855),
    CONVERT_FP(0.06892545234870025),
    CONVERT_FP(0.2167916423815637),
    CONVERT_FP(0.2167916423815637),
    CONVERT_FP(0.06892545234870025),
    CONVERT_FP(-0.025632812530463855),
    CONVERT_FP(-0.0174382058234831),
    CONVERT_FP(0.0027511574458392177),
    CONVERT_FP(0.003206743194403911),
    CONVERT_FP(0.0005222575167540901),
    CONVERT_FP(0.000012655763070268956),
};
// end:coefficients

// start:simple
int32_t us_history[US_COEFFICIENTS/2];

static int us_inner_product(int32_t coefficients[]) {
    int64_t accumulator = 0;
    for(int i = 0; i < US_COEFFICIENTS/2; i++) {
        accumulator += ((coefficients[i] * (int64_t) us_history[i]));
    }
    return (accumulator + (1<<29)) >> 30;
}

void us_simple(int out[2], int in_sample) {
    us_history[US_COEFFICIENTS/2-1] = in_sample;
    out[0] = us_inner_product(us_coefficients_phase0);
    out[1] = us_inner_product(us_coefficients_phase1);
    memmove(us_history, us_history + 1, (US_COEFFICIENTS/2 - 1) * sizeof(int32_t));
}
// end:simple

// start:lib_xcore_math
filter_fir_s32_t filter0_fast;
filter_fir_s32_t filter1_fast;

int32_t fast0_history[(US_COEFFICIENTS+1)/2];
int32_t fast1_history[(US_COEFFICIENTS+1)/2];

void us_fast_init() {
    filter_fir_s32_init(&filter0_fast, fast0_history, (US_COEFFICIENTS+1)/2, us_coefficients_phase0, 0);
    filter_fir_s32_init(&filter1_fast, fast1_history, (US_COEFFICIENTS+1)/2, us_coefficients_phase1, 0);
}

void us_fast(int out[2], int in_sample) {
    out[0] = filter_fir_s32(&filter0_fast, in_sample);
    out[1] = filter_fir_s32(&filter1_fast, in_sample);
}
// end:lib_xcore_math

// start:coefficients-interleaved
int32_t us_coefficients_interleaved[US_COEFFICIENTS+1] = {
    CONVERT_FP(0.0001144368956127061),
    CONVERT_FP(0.0015503056579715265),
    CONVERT_FP(0.004441310178041441),
    CONVERT_FP(-0.004624415198892449),
    CONVERT_FP(-0.028890699902555182),
    CONVERT_FP(0.005899831447759155),
    CONVERT_FP(0.14890438350333784),
    CONVERT_FP(0.24348744648423895),
    CONVERT_FP(0.000012655763070268956),   // phase 1
    CONVERT_FP(0.0005222575167540901),   // phase 1
    CONVERT_FP(0.003206743194403911),   // phase 1
    CONVERT_FP(0.0027511574458392177),   // phase 1
    CONVERT_FP(-0.0174382058234831),   // phase 1
    CONVERT_FP(-0.025632812530463855),   // phase 1
    CONVERT_FP(0.06892545234870025),   // phase 1
    CONVERT_FP(0.2167916423815637),   // phase 1
    CONVERT_FP(0.14890438350333784),
    CONVERT_FP(0.005899831447759155),
    CONVERT_FP(-0.028890699902555182),
    CONVERT_FP(-0.004624415198892449),
    CONVERT_FP(0.004441310178041441),
    CONVERT_FP(0.0015503056579715265),
    CONVERT_FP(0.0001144368956127061),
    0,
    CONVERT_FP(0.2167916423815637),   // phase 1
    CONVERT_FP(0.06892545234870025),   // phase 1
    CONVERT_FP(-0.025632812530463855),   // phase 1
    CONVERT_FP(-0.0174382058234831),   // phase 1
    CONVERT_FP(0.0027511574458392177),   // phase 1
    CONVERT_FP(0.003206743194403911),   // phase 1
    CONVERT_FP(0.0005222575167540901),   // phase 1
    CONVERT_FP(0.000012655763070268956),   // phase 1
};
// end:coefficients-interleaved

// start:vpu
int32_t vpu0_history[(US_COEFFICIENTS+1)/2+1];

extern int us_vpu_asm(int *o, int32_t *d, int32_t *c);

void us_vpu(int out[2], int in_sample) {
    vpu0_history[US_COEFFICIENTS/2] = in_sample;
    us_vpu_asm(out, vpu0_history, us_coefficients_interleaved);
}
// end:vpu
