"""
The FRJ Pipeline definition.
"""

from audio_dsp.design.pipeline import Pipeline, generate_dsp_main
from audio_dsp.stages import *
from audio_dsp.tuning import send_config_to_device
from pathlib import Path
from tuning_utility.params import Params


def pipeline():
    # 0, 1 USB music
    # 2, 3 Mic
    p, in_edges = Pipeline.begin(4, fs=48000)

    mics = p.stage(Adder, in_edges[2, 3])
    mics = p.stage(VolumeControl, mics, "mics_vc")
    music = p.stage(VolumeControl, in_edges[0, 1], "music_vc")

    peq, peq_bypass = p.stage(Fork, mics).forks
    peq = p.stage(CascadedBiquads, peq, "peq")
    peq_sw = p.stage(Switch, peq_bypass + peq, "peq_sw")

    ns, ns_bypass = p.stage(Fork, peq_sw).forks
    ns = p.stage(NoiseSuppressorExpander, ns, "denoise")
    ns = p.stage(Switch, ns_bypass + ns, "denoise_sw")

    p.next_thread()
    rv_bp0, rv_bp1, rv0, rv1 = p.stage(Fork, ns, count=4).forks

    rv = p.stage(ReverbPlateStereo, rv0 + rv1, "reverb", predelay=20)
    rv_sw = p.stage(SwitchStereo, rv_bp0 + rv_bp1 + rv, "reverb_sw")

    duck_bypass, duck = p.stage(Fork, music).forks

    p.next_thread()
    processed_mic = p.stage(Fork, rv_sw, count=3).forks

    duck0 = p.stage(CompressorSidechain, duck[0] + processed_mic[0][0], "duck0")
    duck1 = p.stage(CompressorSidechain, duck[1] + processed_mic[0][1], "duck1")
    duck = p.stage(SwitchStereo, duck_bypass + duck0 + duck1, "duck_sw")
    duck = p.stage(Fork, duck).forks

    loopback0 = p.stage(Adder, duck[0][0] + processed_mic[1][0])
    loopback1 = p.stage(Adder, duck[0][1] + processed_mic[1][1])

    p.next_thread()

    gl_sw = p.stage(
        SwitchStereo, processed_mic[2][:] + loopback0 + loopback1, "loopback_sw"
    )
    gl_sw = p.stage(Fork, gl_sw).forks

    output = p.stage(VolumeControl, gl_sw[0][:], "output_vc")

    headphone = p.stage(SwitchStereo, duck[1][:] + gl_sw[1][:], "monitor_sw")
    headphone = p.stage(VolumeControl, headphone, "headphone_vc")

    # 4 outputs
    # 0, 1 USB
    # 2, 3 Headphone
    p.set_outputs(output + headphone)
    # p.draw()

    p["peq"].make_parametric_eq(
        [
            ["highpass", 100, 0.707],
            ["lowpass", 14000, 0.707],
        ]
    )

    p["reverb"].set_damping(0.2)
    p["reverb"].set_decay(0.6)
    p["reverb"].set_predelay(14)
    p["reverb"].set_wet_dry_mix(0.7)
    p["reverb"].set_width(1.0)

    p["denoise"].make_noise_suppressor_expander(
        ratio=3, threshold_db=-45, attack_t=0.005, release_t=0.12
    )

    p["duck0"].make_compressor_sidechain(
        ratio=5, threshold_db=-40, attack_t=0.01, release_t=0.5
    )
    p["duck1"].make_compressor_sidechain(
        ratio=5, threshold_db=-40, attack_t=0.01, release_t=0.5
    )
    return p


def update_from_tuning_params(p: Pipeline, params: Params):
    params.reverb.update_stage(p["reverb"])
    params.headphone_volume.update_stage(p["headphone_vc"])
    params.output_volume.update_stage(p["output_vc"])
    params.mic_volume.update_stage(p["mics_vc"])
    params.music_volume.update_stage(p["music_vc"])
    params.peq.update_stage(p["peq"])
    params.duck_switch.update_stage(p["duck_sw"])
    params.monitor_switch.update_stage(p["monitor_sw"])
    params.loopback_switch.update_stage(p["loopback_sw"])
    params.reverb_switch.update_stage(p["reverb_sw"])
    params.peq_switch.update_stage(p["peq_sw"])
    params.denoise_switch.update_stage(p["denoise_sw"])
    params.denoise.update_stage(p["denoise"])
    params.ducking.update_stage(p["duck0"])
    params.ducking.update_stage(p["duck1"])

    params.peq.update_stage(p["peq"])
    params.reverb.update_stage(p["reverb"])



def generate_code(p, dire):

    generate_dsp_main(p, out_dir=dire)
