from dataclasses import dataclass
from enum import Enum
from .params import Params
from .pipeline import update_from_tuning_params, generate_code
from audio_dsp.tuning import ConfigSender
from typing import Callable
from queue import Empty
from .usb import USBTransport


@dataclass
class StateUpdated:
    params: Params


@dataclass
class GetPEQGraph:
    return_cb: Callable


@dataclass
class SendTuning:
    return_cb: Callable


@dataclass
class Exit:
    pass


@dataclass
class CodeGen:
    pass


class _Device:
    def __init__(self, pipeline, params, logger, code_gen_dir):
        self._last_params = None
        self.pipeline = pipeline
        self.running = True
        self.params = params
        self.code_gen_dir = code_gen_dir
        self._log = logger
        self._send_peq = True
        self.transport = ConfigSender(USBTransport(host_app_path="host/dsp_host.exe"), "once")

    def run(self, queue):
        while self.running:
            try:
                command = queue.get(timeout=1)
            except Empty:
                continue

            if isinstance(command, StateUpdated):
                self.state_updated(command)
            elif isinstance(command, GetPEQGraph):
                self.get_peq(command)
            elif isinstance(command, SendTuning):
                self.send_tuning(command)
            elif isinstance(command, Exit):
                self.running = False
            elif isinstance(command, CodeGen):
                self.code_gen()
            else:
                self._log.log(f"Unknown command {type(command)}")

    def send_tuning(self, command: SendTuning):
        self._log.log("Sending Tuning")
        try:
            self.transport(self.pipeline)
        except Exception as e:
            self._log.log("Failed!")
            self._log.log(f"{type(e)} {e}")
        else:
            self._log.log("Success!")
        command.return_cb(True)

    def get_peq(self, command: GetPEQGraph):
        if self._send_peq:
            f, h = self.pipeline["peq"].get_frequency_response(1024*16)
            command.return_cb(f, h)
            # dont send again until there has been an update
            self._send_peq = False

    def state_updated(self, command: StateUpdated):
        self._log.log("State updated")
        self.params = command.params
        try:
            update_from_tuning_params(self.pipeline, self.params)
        except (ValueError, TypeError, ZeroDivisionError) as e:
            self._log.log("Tuning error: " + str(e))
        else:
            # new and valid PEQ data is available
            self._send_peq = True

    def code_gen(self):
        generate_code(self.pipeline, self.code_gen_dir)
        self._log.log("Code generation complete!")


def device_thread(queue, pipeline, init_params, logger, code_gen_dir):
    logger.log("On device thread")
    _Device(pipeline, init_params, logger, code_gen_dir).run(queue)
