from enum import IntEnum
from functools import partial
import queue
import threading
import sys
from PySide6.QtWidgets import (
    QApplication,
    QGroupBox,
    QMainWindow,
    QSizePolicy,
    QTabWidget,
    QWidget,
    QVBoxLayout,
    QHBoxLayout,
    QSlider,
    QPushButton,
    QLabel,
    QGridLayout,
    QDial,
    QFileDialog,
    QScrollArea,
    QSplitter,
    QComboBox,
    QDoubleSpinBox,
    QMessageBox,
)
from PySide6.QtCore import Qt, Signal, QObject, Slot, QTimer
import numpy as np
from PySide6.QtGui import QPixmap, QIcon
from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg
from matplotlib.figure import Figure
import matplotlib.pyplot as plt
from pydantic.fields import FieldInfo
from tuning_utility import params
from itertools import count
from pprint import pp
from pathlib import Path
import annotated_types
from argparse import ArgumentParser
import json
from audio_dsp.dsp import utils
from tuning_utility import pipeline, device
from copy import deepcopy
from pydantic import ValidationError
import warnings


import ctypes
ctypes.windll.shell32.SetCurrentProcessExplicitAppUserModelID(u'xmos.demo.app')


VOLUMES = ["mic_volume", "music_volume", "headphone_volume", "output_volume"]
SWITCHES = ["denoise_switch", "duck_switch", "monitor_switch", "loopback_switch"]


class ChecksumError(Exception):
    """Raised when the checksum does not match."""
    pass


class bq_param_box(QDoubleSpinBox):
    """
    A double spin box that sends a signal upwards when the value is changed.
    """
    on_changed = Signal(object)

    def __init__(self, param_name, param_props):
        super().__init__()
        if "maximum" in param_props:
            self.setMaximum(param_props["maximum"])
        elif "exclusiveMaximum" in param_props:
            self.setMaximum(param_props["exclusiveMaximum"])
        
        if "minimum" in param_props:
            self.setMinimum(param_props["minimum"], )
        elif "exclusiveMinimum" in param_props:
            self.setMinimum(param_props["exclusiveMinimum"], )

        self.valueChanged.connect(self.on_changed.emit)

    def wheelEvent(self, event):
        # don't spin with scroll wheel
        event.ignore()

class biquad_group(QWidget):
    """
    A widget consisting of a drop down bosx of biquad types, that updates
    the available control parameters when changed. Emits a signal upwards
    when a parameter is changed.
    """
    on_changed = Signal(object)

    def __init__(self, schema, filt):
        super().__init__()
        self.combobox = QComboBox()
        self.combobox.currentIndexChanged.connect(self.select_type)

        self.current_type = filt["type"]

        self.state = {}

        lay = QVBoxLayout()
        self.items_group = QGroupBox()
        self.items_group.setLayout(lay)

        layout = QVBoxLayout(self)
        layout.addWidget(self.combobox)
        layout.addWidget(self.items_group)

        y = schema["$defs"]
        keys = []

        # for each biquad type, make a widget of labelled scrollboxes
        for bq_type_key, bq_type_props in y.items():
            keys.append(bq_type_key)

            group = QWidget()
            layout = QGridLayout()
            group.setLayout(layout)

            # for each parameter, make a name and scrollbox in a table row
            for param_name, param_props in bq_type_props["properties"].items():

                if param_name =="type":
                    continue

                _lbl = QLabel(param_name, parent=group)

                widget = bq_param_box(param_name, param_props)

                # if value is defined, use it
                if bq_type_key[7:] == filt["type"] and param_name in filt:
                   widget.setValue(filt[param_name])
                else:
                    widget.setValue(param_props["default"])

                widget.on_changed.connect(partial(self.widget_on_changed, param_name))

                # add to table
                _row_index = layout.rowCount()
                layout.addWidget(_lbl, _row_index, 0)
                layout.addWidget(widget, _row_index, 1)

            # hide all parameters, reveal using self.select_type
            group.setVisible(False)
            self.items_group.layout().addWidget(group)

        self.combobox.addItems(keys)
        self.combobox.setCurrentText(f"biquad_{self.current_type}")

    def select_type(self, idx: int):
        # for each biquad type, hide it unless it matches the selected index
        for i in range(self.items_group.layout().count()):
            widget = self.items_group.layout().itemAt(i).widget()
            widget.setVisible(i == idx)
            if i == idx:
                # if selected index, update state with the parameter values
                for x in widget.children():
                    if type(x) is QLabel:
                        param_name = x.text()
                    if type(x) is bq_param_box:
                        self.state[param_name] = x.value()
        self.state["type"] = self.combobox.currentText()[7:]
        self.on_changed.emit(self.state)

    def widget_on_changed(self, name: str, value):
        self.state[name] = value
        self.on_changed.emit(self.state)


class MplCanvas(FigureCanvasQTAgg):
    def __init__(self):
        fig = Figure()
        self.fig = fig
        self.axes = fig.subplots(2, 1, sharex=True)
        self.is_drawn = False
        self.mag_line = None
        self.phase_line = None

        fig.suptitle(f"PEQ frequency response".title())

        super().__init__(fig)
        self.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

    def update_plot(self, f, h):
        h_db = utils.db(h)

        ticks = [20, 50, 100, 200, 500, 1000, 2000, 5000, 10000, 20000]
        ticks_labels = [20, 50, 100, 200, 500, 1000, 2000, "500", "10k", "20k"]

        # self.axes[0].clear()
        # self.axes[1].clear()
        if not self.is_drawn:
            self.mag_line = self.axes[1].semilogx(f, np.angle(h))
            self.phase_line = self.axes[0].semilogx(f, h_db)
            self.axes[0].set_ylim([-20, 20])
            self.axes[0].set_yticks([-20, -10, 0, 10, 20])
            self.axes[0].set_xlim([20, 20000])
            self.axes[0].set_xticks(ticks)
            self.axes[0].set_ylabel("Magnitude (dB)")
            self.axes[0].grid()
            self.axes[1].set_ylim([-3.2, 3.2])
            self.axes[1].set_yticks([-np.pi, -np.pi / 2, 0, np.pi / 2, np.pi])
            self.axes[1].set_yticklabels(["-π", "", "0", "", "π"])
            self.axes[1].set_ylabel("Phase (rad)")
            self.axes[1].set_xticks(ticks)
            self.axes[1].set_xticklabels(ticks_labels)

            self.axes[1].grid()
            self.axes[1].set_xlabel("Frequency (Hz)")
            self.fig.tight_layout()
            self.is_drawn = True
        else:
            self.mag_line[0].set_data(f, np.angle(h))
            self.phase_line[0].set_data(f, h_db)
        self.draw()


def _get_field_range(f: FieldInfo):
    meta = f.metadata
    min = [g.ge for g in meta if isinstance(g, annotated_types.Ge)] or [
        g.gt for g in meta if isinstance(g, annotated_types.Gt)
    ]
    max = [g.le for g in meta if isinstance(g, annotated_types.Le)] or [
        g.lt for g in meta if isinstance(g, annotated_types.Lt)
    ]
    try:
        return min[0], max[0]
    except IndexError as e:
        raise ValueError(f"Field {f} must have a range constraint") from e


class DSPState(QObject):
    STATE_CHANGED = Signal(params.Params)

    def __init__(self, params):
        super().__init__()
        self._state = params

    @property
    def state(self):
        return self._state.model_copy(deep=True)

    def update(self, update_dict):
        state = self._state.model_copy(update=update_dict)
        if state != self._state:
            self.STATE_CHANGED.emit(self.state)
        self._state = state

    def update_load(self, tuning):
        state = deepcopy(tuning)
        if state != self._state:
            self.STATE_CHANGED.emit(self.state)
        self._state = state

    def deep_update(self, label, field, value):
        data = getattr(self.state, label)
        data = data.model_copy(update={field: value})
        self.update({label: data})


class QtLogEmitter(QObject):
    log_signal = Signal(str)

    def log(self, s):
        self.log_signal.emit(s)


class Logger(QObject):
    @Slot()
    def log(self, str):
        print(str)


class DeviceEmitter(QObject):
    send_tuning_complete = Signal(bool)
    plot_updated = Signal(np.ndarray, np.ndarray)


class UpdateDeviceStateMachine(QObject):
    DEBOUNCE_MS = 0

    class _State(IntEnum):
        IDLE = 0
        TUNING = 1
        UPDATED_WHILE_TUNING = 2
        DEBOUNCE = 3

    @Slot()
    def _timeout(self):
        if self._state == self._State.IDLE:
            self._state = self._state.TUNING
            self._do_tuning_cb()

    def __init__(self, do_tuning_cb):
        self._state = self._State.IDLE
        self._do_tuning_cb = do_tuning_cb
        self._timer = QTimer()
        self._timer.setSingleShot(True)
        self._timer.timeout.connect(self._timeout)

    def tuning_complete(self):
        if self._state == self._State.TUNING:
            self._state = self._State.IDLE
        elif self._state == self._State.UPDATED_WHILE_TUNING:
            self._state = self._State.TUNING
            self._do_tuning_cb()
        else:
            raise ValueError(f"Invalid state transition: {self._state}")

    def state_updated(self):
        if self._state == self._State.TUNING:
            self._state = self._State.UPDATED_WHILE_TUNING
        elif self._state == self._State.IDLE:
            # State updated, assume more updates to come so
            # stay in idle until the debounce timer elapses.
            self._timer.stop()
            self._timer.start(self.DEBOUNCE_MS)
        else:
            # stay in this state
            pass


class DSPWindow(QMainWindow):
    peq_signal = Signal()

    def _send_state(self):
        self.q.put(device.StateUpdated(self.state.state))
        self.q.put(
            device.SendTuning(
                lambda s: self.device_emitter.send_tuning_complete.emit(s)
            )
        )
        self.q.put(device.GetPEQGraph(self.device_emitter.plot_updated.emit))

    def __init__(self, params, q):
        super().__init__()
        self.setWindowTitle("DSP Controller")
        self.q = q
        self.device_emitter = DeviceEmitter()

        logo = QIcon()
        logo_path = Path(__file__).parent / "tuning_utility" / "assets" / "XMOS_logo__X_transparent.png"
        logo.addFile(str(logo_path))
        self.setWindowIcon(logo)

        self.plot = MplCanvas()
        # Initialize the DSP state
        self.state = DSPState(params)

        device_state = UpdateDeviceStateMachine(self._send_state)
        self.device_emitter.send_tuning_complete.connect(
            lambda _: device_state.tuning_complete()
        )
        self.state.STATE_CHANGED.connect(lambda _: device_state.state_updated())

        self.device_emitter.plot_updated.connect(self.plot.update_plot)

        # Main widget
        self.main_widget = QWidget()
        self.main_layout = QVBoxLayout()
        self.main_widget.setLayout(self.main_layout)
        self.setCentralWidget(self.main_widget)

        # tab widget
        self.tabs = QTabWidget()
        self.main_layout.addWidget(self.tabs)

        # Add tabs
        self.make_tabs()

        # Buttons at the bottom
        button_layout = QHBoxLayout()
        self.load_button = QPushButton("Load")
        self.load_button.clicked.connect(self.on_load_button)

        self.save_button = QPushButton("Save")
        self.save_button.clicked.connect(self.on_save_button)

        self.gen_button = QPushButton("Generate Code")
        self.gen_button.clicked.connect(self.on_generate_button)

        button_layout.addWidget(self.load_button)
        button_layout.addWidget(self.save_button)
        button_layout.addWidget(self.gen_button)

        self.main_layout.addLayout(button_layout)

        device_state.state_updated()

    def make_tabs(self):
        self.tabs.addTab(self.create_tab1(), "Control")
        self.tabs.addTab(self.create_tab2(), "Configuration")
        # self.tabs.addTab(self.create_tab_geq(self.plot), "Equaliser")
        self.tabs.addTab(self.create_tab_peq(self.plot), "Equaliser")
        self.tabs.addTab(self.create_tab4(), "Pipeline")

    def make_slide_groupbox(self, label):
        box_label = self.state.state.model_fields[label].title or label
        groupbox = QGroupBox(box_label)
        grid = QGridLayout()

        state = getattr(self.state.state, label)
        fields = state.model_fields

        def setup(parent, field, field_spec, dial, label_widget):
            """define functions in a way that stops them from getting clobbered"""
            try:
                fmin, fmax = _get_field_range(field_spec)
            except ValueError as e:
                raise ValueError(f"Field {field} must have a range constraint") from e
            # dial works on ints, so have to scale the dial value to the state value
            # and vice versa
            multiplier = 1000 / (fmax - fmin)
            dial.setRange(0, 1000)
            title = field_spec.title or field

            def set_label(value):
                label_widget.setText(f"{title}: {value:.2f}")

            def dial_to_state(dial):
                return fmin + (dial / multiplier)

            def state_to_dial(state):
                return (state - fmin) * multiplier

            def on_dial(value):
                self.state.deep_update(label, field, dial_to_state(value))

            def on_state(state):
                value = getattr(getattr(state, label), field)
                dial.setValue(state_to_dial(value))
                set_label(value)

            value = getattr(state, field)
            dial.setValue(state_to_dial(value))
            set_label(value)
            dial.valueChanged.connect(on_dial)
            self.state.STATE_CHANGED.connect(on_state)

        for i, (field, field_spec) in enumerate(fields.items()):
            dial = QDial()
            label_widget = QLabel()
            setup(self, field, field_spec, dial, label_widget)
            n_cols = 5
            i_row = (i // n_cols) * 2
            i_col = i % n_cols
            grid.addWidget(label_widget, i_row + 1, i_col, Qt.AlignCenter)
            grid.addWidget(dial, i_row, i_col, Qt.AlignCenter)
        groupbox.setLayout(grid)
        return groupbox

    def create_tab1(self):
        tab = QWidget()
        layout = QHBoxLayout()

        # Volume sliders and buttons
        grid = QGridLayout()
        for i, label in enumerate(VOLUMES):
            slider = QSlider(Qt.Vertical)
            slider.setRange(-46, 0)
            slider.setValue(getattr(self.state.state, label).gain_db)
            slider.valueChanged.connect(
                lambda value, k=label: self.state.deep_update(k, "gain_db", value)
            )
            title = self.state.state.model_fields[label].title
            grid.addWidget(QLabel(title, alignment=Qt.AlignHCenter), 0, i, Qt.AlignHCenter)
            grid.addWidget(slider, 1, i, Qt.AlignHCenter)
            button = QPushButton("Mute")
            button.setCheckable(True)
            button.setChecked(getattr(self.state.state, label).mute)
            button.clicked.connect(
                lambda state, k=label: self.state.deep_update(k, "mute", state)
            )
            grid.addWidget(button, 2, i, Qt.AlignHCenter)

        # Reverb
        i = len(VOLUMES)
        slider = QSlider(Qt.Vertical)
        slider.setRange(0, 100)
        slider.setValue(self.state.state.reverb.wet_dry * 100)
        slider.valueChanged.connect(
            lambda value: self.state.deep_update("reverb", "wet_dry", value / 100)
        )
        self.state.STATE_CHANGED.connect(
            lambda state: slider.setValue(state.reverb.wet_dry * 100)
        )
        grid.addWidget(QLabel("Reverb\nWet/Dry", alignment=Qt.AlignHCenter), 0, i, Qt.AlignHCenter)
        grid.addWidget(slider, 1, i, Qt.AlignHCenter)
        button = QPushButton("Reverb")
        button.setCheckable(True)
        button.setChecked(self.state.state.reverb_switch.position)
        button.clicked.connect(
            lambda state: self.state.deep_update(
                "reverb_switch", "position", int(state)
            )
        )
        grid.addWidget(button, 2, i, Qt.AlignHCenter)

        # Additional buttons
        for i, label in enumerate(SWITCHES):
            button = QPushButton(self.state.state.model_fields[label].title)
            button.setCheckable(True)
            button.setChecked(getattr(self.state.state, label).position)
            button.clicked.connect(
                lambda state, key=label: self.state.deep_update(
                    key, "position", int(state)
                )
            )
            grid.addWidget(button, 3, i, Qt.AlignHCenter)

        i += 1
        logo = QLabel()
        # logo.setSizePolicy(QSizePolicy.Ignored, QSizePolicy.Ignored)
        icon = QIcon(
            str(Path(__file__).parent
            / "tuning_utility"
            / "assets"
            / "xmos_logo_small-103x30.png")
        )  # Replace with your image path
        logo.setPixmap(icon.pixmap(68, 30*103/68))
        grid.addWidget(logo, 3, i, Qt.AlignHCenter)

        layout.addStretch(1)
        layout.addLayout(grid)
        layout.addStretch(1)
        tab.setLayout(layout)
        return tab

    def create_tab2(self):
        tab = QWidget()
        layout = QVBoxLayout()

        for label in ("reverb", "ducking", "denoise"):
            layout.addWidget(self.make_slide_groupbox(label))

        tab.setLayout(layout)
        return tab

    def _update_geq_i(self, i, value):
        peq = self.state.state.peq
        peq.root[i].boost_db = value
        self.state.update({"peq": peq})

    def create_tab_geq(self, plot):
        tab = QWidget()
        layout = QVBoxLayout()

        # Graphical EQ with sliders
        eq_layout = QHBoxLayout()
        eq_layout.addStretch(1)
        for i in range(len(self.state.state.peq.root)):
            biquad = self.state.state.peq.root[i]
            freq = biquad.filter_freq
            slider = QSlider(Qt.Vertical)
            slider.setRange(-12, 12)
            slider.setValue(0)
            slider.valueChanged.connect(partial(self._update_geq_i, i))
            eq_layout.addWidget(slider)
            label = QLabel(str(int(freq)))
            label.setFixedHeight(200)
            eq_layout.addWidget(label)
        eq_layout.addStretch(1)

        layout.addLayout(eq_layout)
        layout.addWidget(plot)
        tab.setLayout(layout)
        return tab

    def _update_peq_i(self, i, value):
        # update the i-th filter in a parametric eq
        peq = self.state.state.peq
        handle = getattr(params, "biquad_" + value["type"])

        # only pass valid dict keys through to the biquad function
        valid_keys = handle.model_fields.keys()
        for field in list(value.keys()):
            if field not in valid_keys:
                value.pop(field)

        peq.root[i] = handle(**value)

        self.state.update({"peq": peq})

    def peq_biquad_panel(self, schema):
        # make a panel containing N sets of biquad parameter widgets
        container = QWidget()

        widget = QScrollArea()
        widget.setWidget(container)
        widget.setWidgetResizable(True)

        layout = QVBoxLayout(container)

        peq_values = self.state.state.peq.model_dump()

        self.peq = []
        for i, filt in enumerate(peq_values):
            # pass each biquad filter in state to make a new widget, using
            # the state parameters to set the values.
            this_bq = biquad_group(schema, filt)
            this_bq.on_changed.connect(partial(self._update_peq_i, i))

            self.peq.append(this_bq)
            layout.addWidget(self.peq[-1])

        # set width to avoid horizontal scroll
        min_width = widget.sizeHint().width() + 2 * widget.frameWidth() + widget.verticalScrollBar().sizeHint().width()
        widget.setMinimumWidth(max(min_width, 221))

        return widget

    def create_tab_peq(self, plot):
        tab = QWidget()
        layout = QVBoxLayout()

        # Horizontal layout
        # QSplitter can be adusted with the mouse
        horizontal_layout = QSplitter()

        schema = self.state.state.peq.model_json_schema()
        combo = self.peq_biquad_panel(schema)

        horizontal_layout.addWidget(combo)
        horizontal_layout.addWidget(plot)

        layout.addWidget(horizontal_layout)
        tab.setLayout(layout)
        return tab

    def create_tab4(self):
        tab = QWidget()
        layout = QVBoxLayout()

        # Load and display DSP Graph image
        label = QLabel()
        pipeline = Path(__file__).parent / "tuning_utility" / "assets" / "pipeline.png"
        icon = QIcon(str(pipeline))
        label.setPixmap(
            icon.pixmap(600, 400)
        )
        layout.addWidget(label, alignment=Qt.AlignCenter)

        tab.setLayout(layout)
        return tab

    def on_save_button(self):
        filename, _ = QFileDialog.getSaveFileName(self, str("Save xDSP"), ".", str("xDSP Files (*.json)"))
        if not filename:
            return
        filepath = Path(filename)
        filepath.write_text(json.dumps(self.state.state.model_dump(), indent=2))

    def on_load_button(self):
        filename, _ = QFileDialog.getOpenFileName(self, str("Open xDSP"), ".", str("xDSP Files (*.json)"))
        if not filename:
            return
        filepath = Path(filename)

        try:        
            tuning = params.Params.model_validate_json(filepath.read_text())
            validate_checksum(tuning.checksum, self.state.state.checksum)
        except ChecksumError as e:
            print(e)
            self.load_invalid_checksum_messagebox()
            # use actual pipeline checksum so we save the correct one
            tuning.checksum = self.state.state.checksum

        except ValidationError as e:
            print(e)
            msgBox = QMessageBox.critical(self, "Pipeline error", "The loaded .json pipeline is not "
            "compatible with this the pipeline.py and/or parameters.py in this version of the GUI.\n\n"
            "Please try loading a different file.")
            self.on_load_button()
            return

        self.state.update_load(tuning)
        current = self.tabs.currentIndex()
        self.tabs.clear()
        self.make_tabs()
        self.tabs.setCurrentIndex(current)

    def on_generate_button(self):
        this_dir = Path(__file__).parent
        code_gen_dir = this_dir / "app_an02031/src/generated_dsp"
        code_gen_dir = QFileDialog.getExistingDirectory(self, "Gen folder", str(code_gen_dir))
        p = pipeline.pipeline()
        pipeline.update_from_tuning_params(p, self.state.state)
        pipeline.generate_code(p, code_gen_dir)

    def load_invalid_checksum_messagebox(self):
        msgBox = QMessageBox.warning(self, "Checksum mismatch", "The checksum of the loaded pipeline does "
        "not match. This .json file may or may not work with the current pipeline.py file.\n\n"
        "If the pipeline is compatible, the checksum in the .json will be updated during saving.")


def validate_checksum(actual, desired):
    equal = np.array_equal(
    np.array(actual),
    np.array(desired),
    )

    if equal is False:
        raise ChecksumError(
            (
                "Pipeline mismatch; the pipeline defined in the GUI does not match "
                "the pipeline defined in the saved JSON file.\n"
                f"\n\tExpected checksum: {desired}\n\tGot {actual}"
            )
        )

def parse_args(tuning_path):
    parser = ArgumentParser()
    parser.add_argument(
        "--code-gen",
        "-c",
        type=Path,
        nargs='?',
        const=tuning_path,
        help=("If set, generate the pipeline code from the given tuning path and exit. "
              "Optionally, the path to a specific .json file can be supplied. "
              f"By default uses the file {tuning_path.name}"),
    )
    parser.add_argument(
        "--draw",
        "-d",
        type=Path,
        help="if set, draw the pipeline and exit",
    )
    return parser.parse_args()


if __name__ == "__main__":
    p = pipeline.pipeline()
    checksum = p.pipeline_stage["checksum"]

    this_dir = Path(__file__).parent
    tuning_path = this_dir / "tuning.json"
    code_gen_dir = this_dir / "app_an02031/src/generated_dsp"
    args = parse_args(tuning_path)

    if args.code_gen:
        tuning = params.Params.model_validate_json(args.code_gen.read_text())
        pipeline.update_from_tuning_params(p, tuning)
        pipeline.generate_code(p, code_gen_dir)
        exit(0)

    if args.draw:
        p.draw(args.draw)
        exit(0)

    if not tuning_path.exists():
        tuning = params.Params(checksum=checksum)
        tuning_path.write_text(json.dumps(tuning.model_dump(), indent=2))
    else:
        try:
            tuning = params.Params.model_validate_json(tuning_path.read_text())
            validate_checksum(tuning.checksum, checksum)
        except ChecksumError as e:
            print(e)
            warnings.warn(f"The checksum of the default .json file ({tuning_path.name}) does not match. "
                          "The pipeline may not be compatible.")
            # use actual pipeline checksum so we save the correct one
            tuning.checksum = checksum

        except ValidationError as e:
            print(e)
            print()
            print(f"Error: the default pipeline .json file ({tuning_path.name}) has been modified in an incompatible way.")
            while True:
                userin = input(f"Overwrite {tuning_path.name} with correct format? This will "
                "overwrite any modified tuning parameters with default values. Yes(y) / No(n): ")
                if userin.lower() in ["yes", "y"]:
                    tuning = params.Params(checksum=checksum)
                    tuning_path.write_text(json.dumps(tuning.model_dump(), indent=2))
                    break
                elif userin.lower() in ["no", "n"]:
                    exit()
                else:
                    print("Invalid input, please enter yes/no")

    pipeline.update_from_tuning_params(p, tuning)
    q = queue.Queue()

    app = QApplication(sys.argv)
    device_logger = QtLogEmitter()
    log_outputter = Logger()
    device_logger.log_signal.connect(log_outputter.log)
    device_thread = threading.Thread(
        target=device.device_thread, args=(q, p, tuning, device_logger, code_gen_dir)
    )
    device_thread.start()

    try:
        window = DSPWindow(tuning, q)
        window.show()
        sys.exit(app.exec())
    finally:
        q.put(device.Exit())
        device_thread.join()
