// Copyright 2024-2025 XMOS LIMITED.
// This Software is subject to the terms of the XMOS Public Licence: Version 1.

#pragma once

extern "C"
#include "device_control_shared.h"
#include <memory>
#include <iostream>
#include <vector>

/**
 * @brief Class for interfacing device_contol
 */
class Device_USB
{
    private:

        /** @brief Information to intialise the device */
        int * device_info;

    protected:

        /**
         * @brief State of the device.
         *
         * Used to prevent multiple initialisations / cleanups.
         */
        bool device_initialised = false;

    public:

        /**
         * @brief Construct a new Device object
         *
         * @param device_info   Pointer to the information to initialise a device
         */
        Device_USB(int * device_info);

        /** @brief Initialise a host (master) interface */
        control_ret_t device_init();

        /**
         * @brief Request to read from a controllable resource inside the device
         *
         * @param res_id        Resource ID
         * @param cmd_id        Command ID
         * @param payload       Array of bytes which constitutes the data payload
         * @param payload_len   Size of the payload in bytes
         */
        control_ret_t device_get(control_resid_t res_id, control_cmd_t cmd_id, uint8_t payload[], size_t payload_len);

        /**
         * @brief Request to write to a controllable resource inside the device
         *
         * @param res_id        Resource ID
         * @param cmd_id        Command ID
         * @param payload       Array of bytes which constitutes the data payload
         * @param payload_len   Size of the payload in bytes
         */
        control_ret_t device_set(control_resid_t res_id, control_cmd_t cmd_id, const uint8_t payload[], size_t payload_len);

        /**
         * @brief Destroy the Device object.
         *
         * This will shut down the host interface connection.
         */
        ~Device_USB();
};
