function(find_libusb LIBUSB_LIBRARIES_OUT LIBUSB_LIBRARY_PATH_OUT LIBUSB_TARGET_NAME_OUT)
    # Optional: user-specified fallback path
    set(LIBUSB_LIBRARY_PATH "" CACHE PATH "Path to prebuilt dynamically-linkable libusb library file, if XTC tools not available.")
    set(LIBUSB_SOURCE_PATH "" CACHE PATH "Path to libusb source directory, if XTC tools not available and prebuilt library not available")
    set(LIBUSB_DOWNLOAD_VERSION "" CACHE STRING "Version of libusb to download, if all other methods unavailable")

    # Check for the XMOS_TOOL_PATH environment variable
    if(DEFINED ENV{XMOS_TOOL_PATH})
        set(LIBUSB_XMOS_PATH $ENV{XMOS_TOOL_PATH})
    endif()

    set(LIBUSB_FOUND FALSE)
    add_custom_target(find_libusb_target ALL)
    set(${LIBUSB_TARGET_NAME_OUT} find_libusb_target PARENT_SCOPE)

    if(${CMAKE_SYSTEM_NAME} MATCHES "Linux")
        if(DEFINED LIBUSB_XMOS_PATH)
            set(LIBUSB_XMOS_LIB "${LIBUSB_XMOS_PATH}/lib/libusb-1.0.so")
            if(NOT EXISTS "${LIBUSB_XMOS_LIB}")
                message(FATAL_ERROR "libusb-1.0.so not found in XMOS tools path: ${LIBUSB_XMOS_PATH}/lib/")
            endif()
            set(LIBUSB_LIBRARIES "${LIBUSB_XMOS_LIB}")
            set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_XMOS_LIB}" PARENT_SCOPE)
            set(LIBUSB_FOUND TRUE)
        endif()

        if(NOT LIBUSB_FOUND)
            find_package(PkgConfig)
            if(PKG_CONFIG_FOUND)
                pkg_check_modules(LIBUSB_PC libusb-1.0)
                if(LIBUSB_PC_FOUND)
                    set(LIBUSB_LIBRARIES ${LIBUSB_PC_LIBRARIES})
                    set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_PC_LIBRARIES}" PARENT_SCOPE)
                    set(LIBUSB_FOUND TRUE)
                endif()
            endif()
        endif()

        if(NOT LIBUSB_FOUND AND LIBUSB_LIBRARY_PATH)
            if(EXISTS "${LIBUSB_LIBRARY_PATH}")
                message(STATUS "Using user-specified LIBUSB_LIBRARY_PATH: ${LIBUSB_LIBRARY_PATH}")
            else()
                message(FATAL_ERROR "libusb-1.0 library not found at LIBUSB_LIBRARY_PATH: ${LIBUSB_LIBRARY_PATH}")
            endif()
            set(LIBUSB_LIBRARIES ${LIBUSB_LIBRARY_PATH})
            set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_LIBRARY_PATH}" PARENT_SCOPE)
            set(LIBUSB_FOUND TRUE)
        endif()

        # Try building from source if not found
        if(NOT LIBUSB_FOUND AND LIBUSB_SOURCE_PATH)
            set(LIBUSB_BUILD_DIR "${CMAKE_BINARY_DIR}/libusb_build")
            set(LIBUSB_SO "${LIBUSB_SOURCE_PATH}/libusb/.libs/libusb-1.0.so")

            # Custom command to build libusb using autogen.sh and make
            add_custom_command(
                OUTPUT "${LIBUSB_SO}"
                COMMAND ./autogen.sh
                COMMAND make
                WORKING_DIRECTORY "${LIBUSB_SOURCE_PATH}"
                COMMENT "Building libusb from source using autogen.sh and make"
                VERBATIM
            )

            # Custom target to ensure build happens
            add_custom_target(libusb_build_from_source ALL
                DEPENDS "${LIBUSB_SO}"
            )

            set(LIBUSB_LIBRARIES "${LIBUSB_SO}")
            set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_SO}" PARENT_SCOPE)
            set(LIBUSB_FOUND TRUE)
            add_dependencies(find_libusb_target libusb_build_from_source)

            message(STATUS "libusb will be built at build time in ${LIBUSB_SOURCE_PATH}")
        endif()

        # Try downloading and building from source if not found and no valid LIBUSB_SOURCE_PATH
        if(NOT LIBUSB_FOUND AND LIBUSB_DOWNLOAD_VERSION)
            set(LIBUSB_URL "https://github.com/libusb/libusb/archive/refs/tags/v${LIBUSB_DOWNLOAD_VERSION}.tar.gz")
            set(LIBUSB_BUILD_DIR "${CMAKE_BINARY_DIR}/libusb_build")
            set(DOWNLOAD_DIR "${LIBUSB_BUILD_DIR}/libusb_download")
            set(EXTRACT_DIR "${LIBUSB_BUILD_DIR}/libusb-${LIBUSB_DOWNLOAD_VERSION}")
            set(TARBALL "${DOWNLOAD_DIR}/libusb-${LIBUSB_DOWNLOAD_VERSION}.tar.gz")
            set(SOURCE_DIR "${EXTRACT_DIR}/libusb-${LIBUSB_DOWNLOAD_VERSION}")

            file(MAKE_DIRECTORY "${DOWNLOAD_DIR}")
            file(MAKE_DIRECTORY "${EXTRACT_DIR}")

            # Download at configure time
            if(NOT EXISTS "${TARBALL}")
                message(STATUS "Downloading libusb v${LIBUSB_DOWNLOAD_VERSION}...")
                file(DOWNLOAD
                    "${LIBUSB_URL}"
                    "${TARBALL}"
                    SHOW_PROGRESS
                    STATUS DOWNLOAD_STATUS
                )
                list(GET DOWNLOAD_STATUS 0 STATUS_CODE)
                if(NOT STATUS_CODE EQUAL 0)
                    message(FATAL_ERROR "Failed to download libusb source code")
                endif()
            endif()

            # Extract at configure time
            if(NOT EXISTS "${SOURCE_DIR}")
                message(STATUS "Extracting libusb source...")
                file(ARCHIVE_EXTRACT
                    INPUT "${TARBALL}"
                    DESTINATION "${EXTRACT_DIR}"
                )
            endif()

            set(LIBUSB_SO "${SOURCE_DIR}/libusb/.libs/libusb-1.0.so")

            # Custom command to build libusb using autogen.sh and make
            add_custom_command(
                OUTPUT "${LIBUSB_SO}"
                COMMAND ./autogen.sh
                COMMAND make
                WORKING_DIRECTORY "${SOURCE_DIR}"
                COMMENT "Building libusb from downloaded source using autogen.sh and make"
                VERBATIM
            )

            add_custom_target(libusb_download_build_from_source ALL
                DEPENDS "${LIBUSB_SO}"
            )

            set(LIBUSB_LIBRARIES "${LIBUSB_SO}")
            set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_SO}" PARENT_SCOPE)
            set(LIBUSB_FOUND TRUE)
            add_dependencies(find_libusb_target libusb_download_build_from_source)

            message(STATUS "libusb has been downloaded and will be built at build time in ${SOURCE_DIR}")
        endif()

        if(NOT LIBUSB_FOUND)
            message(FATAL_ERROR "libusb-1.0 library not found in XMOS tools, via pkg-config, at LIBUSB_LIBRARY_PATH, built from LIBUSB_SOURCE_PATH, or by download.")
        endif()
        set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_XMOS_LIB}" PARENT_SCOPE)

    elseif(${CMAKE_SYSTEM_NAME} MATCHES "Windows")
        set(LIBUSB_BUILD_DIR "${CMAKE_BINARY_DIR}/libusb_build")
        if(DEFINED LIBUSB_XMOS_PATH)
            set(LIBUSB_XMOS_DLL "${LIBUSB_XMOS_PATH}/lib/libusb-1.0.dll")
            if(EXISTS "${LIBUSB_XMOS_DLL}")
                dll_to_lib("${LIBUSB_XMOS_DLL}" LIBUSB_LIB_FILE "${LIBUSB_BUILD_DIR}" DLL_TO_LIB_TARGET_NAME)
                add_dependencies(find_libusb_target ${DLL_TO_LIB_TARGET_NAME})
            else()
                message(FATAL_ERROR "libusb-1.0.dll not found in XMOS tools path: ${LIBUSB_XMOS_PATH}")
            endif()
            set(LIBUSB_LIBRARIES "${LIBUSB_LIB_FILE}")
            set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_XMOS_DLL}" PARENT_SCOPE)
            set(LIBUSB_FOUND TRUE)
        endif()

        if(NOT LIBUSB_FOUND AND LIBUSB_LIBRARY_PATH)
            if(EXISTS "${LIBUSB_LIBRARY_PATH}")
                message(STATUS "Using user-specified LIBUSB_LIBRARY_PATH: ${LIBUSB_LIBRARY_PATH}")
                get_filename_component(LIBUSB_LIBRARY_EXT "${LIBUSB_LIBRARY_PATH}" LAST_EXT)
                string(TOLOWER "${LIBUSB_LIBRARY_EXT}" LIBUSB_LIBRARY_EXT)
                if(LIBUSB_LIBRARY_EXT STREQUAL ".dll")
                    dll_to_lib("${LIBUSB_LIBRARY_PATH}" LIBUSB_LIB_FILE "${LIBUSB_BUILD_DIR}" DLL_TO_LIB_TARGET_NAME)
                    add_dependencies(find_libusb_target ${DLL_TO_LIB_TARGET_NAME})
                    set(LIBUSB_LIBRARIES "${LIBUSB_LIB_FILE}")
                else()
                    message(FATAL_ERROR "LIBUSB_LIBRARY_PATH must be a .dll file")
                endif()
            else()
                message(FATAL_ERROR "libusb-1.0 library not found at LIBUSB_LIBRARY_PATH: ${LIBUSB_LIBRARY_PATH}")
            endif()
            set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_LIBRARY_PATH}" PARENT_SCOPE)
            set(LIBUSB_FOUND TRUE)
        endif()

        # # Try building from source if not found
        if(NOT LIBUSB_FOUND AND LIBUSB_SOURCE_PATH)
            # Check if we're in an MSVC environment
            if(NOT MSVC)
                message(FATAL_ERROR "This function must be run in an MSVC environment")
            endif()
            
            # Build libusb using msbuild at build time
            set(MSVC_DIR "${LIBUSB_SOURCE_PATH}/msvc")
            set(BUILD_OUTPUT_DIR "${LIBUSB_SOURCE_PATH}/build/v143/Win32/Release/dll")
            set(OUTPUT_DIR "${LIBUSB_BUILD_DIR}")
            set(OUTPUT_DLL "${OUTPUT_DIR}/libusb-1.0.dll")
            
            # Create build directory
            file(MAKE_DIRECTORY ${OUTPUT_DIR})
            
            # Create custom command to build libusb at build time
            add_custom_command(
                OUTPUT "${BUILD_OUTPUT_DIR}/libusb-1.0.dll"
                COMMAND msbuild -p:PlatformToolset=v143,Platform=win32,Configuration=Release libusb.sln
                WORKING_DIRECTORY ${MSVC_DIR}
                COMMENT "Building libusb from source..."
                VERBATIM
            )
            
            # Create custom command to copy the DLL
            add_custom_command(
                OUTPUT "${OUTPUT_DLL}"
                COMMAND ${CMAKE_COMMAND} -E copy "${BUILD_OUTPUT_DIR}/libusb-1.0.dll" "${OUTPUT_DLL}"
                DEPENDS "${BUILD_OUTPUT_DIR}/libusb-1.0.dll"
                COMMENT "Copying libusb-1.0.dll to ${OUTPUT_DIR}"
                VERBATIM
            )
            
            # Create target to ensure build happens
            add_custom_target(libusb_build_from_source ALL
                DEPENDS "${OUTPUT_DLL}"
            )
            
            # Set up dll-to-lib conversion (dll_to_lib already uses add_custom_command)
            set(LIBUSB_LIBRARY_PATH "${OUTPUT_DLL}")
            dll_to_lib("${LIBUSB_LIBRARY_PATH}" LIBUSB_LIB_FILE "${LIBUSB_BUILD_DIR}" DLL_TO_LIB_TARGET_NAME)
            set(LIBUSB_LIBRARIES "${LIBUSB_LIB_FILE}")
            
            # Make sure our target also depends on the lib file
            add_dependencies(libusb_build_from_source ${DLL_TO_LIB_TARGET_NAME})
            add_dependencies(find_libusb_target libusb_build_from_source)

            message(STATUS "libusb will be built at build time in ${OUTPUT_DIR}")
            set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_LIBRARY_PATH}" PARENT_SCOPE)
            set(LIBUSB_FOUND TRUE)
        endif()

        if(NOT LIBUSB_FOUND AND LIBUSB_DOWNLOAD_VERSION)
            # Check if we're in an MSVC environment
            if(NOT MSVC)
                message(FATAL_ERROR "This function must be run in an MSVC environment")
            endif()
            
            # Set up paths and URLs
            set(LIBUSB_URL "https://github.com/libusb/libusb/archive/refs/tags/v${LIBUSB_DOWNLOAD_VERSION}.zip")
            set(DOWNLOAD_DIR "${LIBUSB_BUILD_DIR}/libusb_download")
            set(EXTRACT_DIR "${LIBUSB_BUILD_DIR}/libusb-${LIBUSB_DOWNLOAD_VERSION}")
            set(ZIP_FILE "${DOWNLOAD_DIR}/libusb-${LIBUSB_DOWNLOAD_VERSION}.zip")
            set(OUTPUT_DIR "${LIBUSB_BUILD_DIR}")
            set(OUTPUT_DLL "${OUTPUT_DIR}/libusb-1.0.dll")
            set(MSVC_DIR "${EXTRACT_DIR}/libusb-${LIBUSB_DOWNLOAD_VERSION}/msvc")
            set(BUILD_OUTPUT_DIR "${EXTRACT_DIR}/libusb-${LIBUSB_DOWNLOAD_VERSION}/build/v143/Win32/Release/dll")
            
            # Create directories
            file(MAKE_DIRECTORY ${DOWNLOAD_DIR})
            file(MAKE_DIRECTORY ${EXTRACT_DIR})
            file(MAKE_DIRECTORY ${OUTPUT_DIR})
            
            # Download at configure time (could be moved to build time with ExternalProject)
            if(NOT EXISTS "${ZIP_FILE}")
                message(STATUS "Downloading libusb v${LIBUSB_DOWNLOAD_VERSION}...")
                file(DOWNLOAD 
                    ${LIBUSB_URL} 
                    ${ZIP_FILE}
                    SHOW_PROGRESS
                    STATUS DOWNLOAD_STATUS
                )
                list(GET DOWNLOAD_STATUS 0 STATUS_CODE)
                if(NOT STATUS_CODE EQUAL 0)
                    message(FATAL_ERROR "Failed to download libusb source code")
                endif()
            endif()
            
            # Extract at configure time (could be moved to build time with ExternalProject)
            if(NOT EXISTS "${EXTRACT_DIR}/libusb-${LIBUSB_DOWNLOAD_VERSION}")
                message(STATUS "Extracting libusb source...")
                file(ARCHIVE_EXTRACT 
                    INPUT ${ZIP_FILE}
                    DESTINATION ${EXTRACT_DIR}
                )
            endif()
            
            # Build at build time
            add_custom_command(
                OUTPUT "${BUILD_OUTPUT_DIR}/libusb-1.0.dll"
                COMMAND msbuild -p:PlatformToolset=v143,Platform=win32,Configuration=Release libusb.sln
                WORKING_DIRECTORY ${MSVC_DIR}
                COMMENT "Building libusb from downloaded source..."
                VERBATIM
            )
            
            # Copy at build time
            add_custom_command(
                OUTPUT "${OUTPUT_DLL}"
                COMMAND ${CMAKE_COMMAND} -E copy "${BUILD_OUTPUT_DIR}/libusb-1.0.dll" "${OUTPUT_DLL}"
                DEPENDS "${BUILD_OUTPUT_DIR}/libusb-1.0.dll"
                COMMENT "Copying libusb-1.0.dll to ${OUTPUT_DIR}"
                VERBATIM
            )
            
            # Create target to ensure build happens
            add_custom_target(libusb_download_build ALL
                DEPENDS "${OUTPUT_DLL}"
            )
            
            # Set up dll-to-lib conversion
            set(LIBUSB_LIBRARY_PATH "${OUTPUT_DLL}")
            dll_to_lib("${LIBUSB_LIBRARY_PATH}" LIBUSB_LIB_FILE "${LIBUSB_BUILD_DIR}" DLL_TO_LIB_TARGET_NAME)
            set(LIBUSB_LIBRARIES "${LIBUSB_LIB_FILE}")
            
            # Make sure our target also depends on the lib file
            add_dependencies(libusb_download_build ${DLL_TO_LIB_TARGET_NAME})
            add_dependencies(find_libusb_target libusb_download_build)

            message(STATUS "libusb will be downloaded and built at build time in ${OUTPUT_DIR}")
            set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_LIBRARY_PATH}" PARENT_SCOPE)
            set(LIBUSB_FOUND TRUE)
        endif()

        if(NOT LIBUSB_FOUND)
            message(FATAL_ERROR "libusb-1.0 library not found in XMOS tools, at LIBUSB_LIBRARY_PATH, built from LIBUSB_SOURCE_PATH, or by download.")
        endif()

    elseif(${CMAKE_SYSTEM_NAME} MATCHES "Darwin")
        set(LIBUSB_FOUND FALSE)
        # Check XMOS tools
        if(DEFINED LIBUSB_XMOS_PATH)
            set(LIBUSB_XMOS_DYLIB "${LIBUSB_XMOS_PATH}/lib/libusb-1.0.dylib")
            if(EXISTS "${LIBUSB_XMOS_DYLIB}")
                set(LIBUSB_LIBRARIES "${LIBUSB_XMOS_DYLIB}")
                set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_XMOS_DYLIB}" PARENT_SCOPE)
                set(LIBUSB_FOUND TRUE)
            else()
                message(FATAL_ERROR "libusb-1.0.dylib not found in XMOS tools path: ${LIBUSB_XMOS_PATH}/lib/")
            endif()
        endif()

        # Check user-specified path
        if(NOT LIBUSB_FOUND AND LIBUSB_LIBRARY_PATH)
            if(EXISTS "${LIBUSB_LIBRARY_PATH}")
                message(STATUS "Using user-specified LIBUSB_LIBRARY_PATH: ${LIBUSB_LIBRARY_PATH}")
                set(LIBUSB_LIBRARIES "${LIBUSB_LIBRARY_PATH}")
                set(${LIBUSB_LIBRARY_PATH_OUT} "${LIBUSB_LIBRARY_PATH}" PARENT_SCOPE)
                set(LIBUSB_FOUND TRUE)
            else()
                message(FATAL_ERROR "libusb-1.0.dylib not found at LIBUSB_LIBRARY_PATH: ${LIBUSB_LIBRARY_PATH}")
            endif()
        endif()

        if(NOT LIBUSB_FOUND)
            message(FATAL_ERROR "libusb-1.0.dylib not found in XMOS tools or at LIBUSB_LIBRARY_PATH.")
        endif()
    else()
        message(FATAL_ERROR "Unsupported operating system: ${CMAKE_SYSTEM_NAME}.")
    endif()

    set(${LIBUSB_LIBRARIES_OUT} "${LIBUSB_LIBRARIES}" PARENT_SCOPE)
endfunction()

function(dll_to_lib DLL_FILE OUT_LIB DUMP_DEF_LIB_DIR TARGET_NAME)
    get_filename_component(DLL_NAME_WE "${DLL_FILE}" NAME_WE)
    get_filename_component(DLL_DIR "${DLL_FILE}" DIRECTORY)
    set(DUMP_FILE "${DUMP_DEF_LIB_DIR}/${DLL_NAME_WE}.dump")
    set(DEF_FILE "${DUMP_DEF_LIB_DIR}/${DLL_NAME_WE}.def")
    set(LIB_FILE "${DUMP_DEF_LIB_DIR}/${DLL_NAME_WE}.lib")

    # 1. Run DUMPBIN
    add_custom_command(
        OUTPUT "${DUMP_FILE}"
        COMMAND DUMPBIN /EXPORTS /OUT:"${DUMP_FILE}" "${DLL_FILE}"
        DEPENDS "${DLL_FILE}"
        COMMENT "Running DUMPBIN on ${DLL_FILE}"
    )

    # 2. Parse with PowerShell to produce .def file
    add_custom_command(
        OUTPUT "${DEF_FILE}"
        COMMAND powershell -ExecutionPolicy Bypass -File "${CMAKE_CURRENT_FUNCTION_LIST_DIR}/parse_dumpbin.ps1" "${DUMP_FILE}" "${DEF_FILE}"
        DEPENDS "${DUMP_FILE}" 
        COMMENT "Parsing exports with PowerShell"
    )

    # 3. Generate .lib
    add_custom_command(
        OUTPUT "${LIB_FILE}"
        COMMAND LIB /DEF:"${DEF_FILE}" /MACHINE:X64 /OUT:"${LIB_FILE}"
        DEPENDS "${DEF_FILE}"
        COMMENT "Generating .lib file"
    )

    add_custom_target(genlibfromdll_${DLL_NAME_WE} ALL
        DEPENDS "${LIB_FILE}"
        COMMENT "DLL -> DUMPBIN -> DEF -> LIB"
    )

    set(${OUT_LIB} "${LIB_FILE}" PARENT_SCOPE)
    set(${TARGET_NAME} genlibfromdll_${DLL_NAME_WE} PARENT_SCOPE)
endfunction()
