// Copyright 2024-2025 XMOS LIMITED.
// This Software is subject to the terms of the XMOS Public Licence: Version 1.
#include <string.h>
#include <stdlib.h>
#include <xcore/assert.h>
#include <debug_print.h>
#include "cmds.h" // Autogenerated
#include "cmd_offsets.h" // Autogenerated
#include "stages/crossfader.h"
#include <stdio.h>
#include "print.h"
#include "control/signal_chain.h"

void crossfader_process(int32_t **input, int32_t **output, void *app_data_state)
{
    crossfader_state_t *state = app_data_state;

    int32_t *in1 = input[0];
    int32_t *in2 = input[1];
    int32_t *out = output[0];
    int j = 0;
    do
    {
        *out++ = adsp_crossfader_slew(&state->cfs, *in1++, *in2++);

    } while (++j < state->frame_size);
}


void crossfader_init(module_instance_t* instance, adsp_bump_allocator_t* allocator, uint8_t id, int n_inputs, int n_outputs, int frame_size)
{
    crossfader_state_t *state = instance->state;
    crossfader_config_t *config = instance->control.config;

    memset(state, 0, sizeof(crossfader_state_t));
    state->n_inputs = n_inputs;
    xassert(n_inputs == 2 && "crossfader must have 2 inputs");

    state->frame_size = frame_size;
    xassert(n_outputs == 1 && "crossfader should only have one outputs");
    state->n_outputs = n_outputs;

    state->cfs.gain_1 = adsp_slew_gain_init(config->gains[0], 7);
    state->cfs.gain_2 = adsp_slew_gain_init(config->gains[1], 7);
}

void crossfader_control(void *module_state, module_control_t *control)
{
    crossfader_state_t *state = module_state;
    crossfader_config_t *config = control->config;

    if(control->config_rw_state == config_write_pending)
    {
        // Finish the write by updating the working copy with the new config
        state->cfs.gain_1.target_gain = config->gains[0];
        state->cfs.gain_2.target_gain = config->gains[1];

        control->config_rw_state = config_none_pending;
    }
    else if(control->config_rw_state == config_read_pending)
    {
        config->gains[0] = state->cfs.gain_1.target_gain;
        config->gains[1] = state->cfs.gain_2.target_gain;
        control->config_rw_state = config_read_updated;
    }
    else
    {
        // nothing to do.
    }
}

