#include <xcore/assert.h>
#include <xua.h>
#include <xua_audiohub.h>
#include <stdint.h>
#include <stages/adsp_pipeline.h>
#include <adsp_generated_auto.h>

#include "app_dsp.h"

// This assumes that the application has not enabled MIDI, SPDIF, ADAT, or PDM mics.
#define N_CHANS_FROM_AUDIO (I2S_CHANS_ADC)
#define N_CHANS_TO_AUDIO (I2S_CHANS_DAC)
#define N_CHANS_TO_DSP (NUM_USB_CHAN_OUT + I2S_CHANS_ADC)
#define N_CHANS_FROM_DSP (NUM_USB_CHAN_IN + I2S_CHANS_DAC)

// Ensure the number of DSP channels matches the number of audio channels
#if(N_CHANS_TO_DSP != ADSP_AUTO_N_INPUTS)
#error N_CHANS_TO_DSP mismatch with ADSP_AUTO_N_INPUTS
#endif

#if(N_CHANS_FROM_DSP != ADSP_AUTO_N_OUTPUTS)
#error N_CHANS_FROM_DSP mismatch with ADSP_AUTO_N_OUTPUTS
#endif

static adsp_pipeline_t *m_dsp;

void UserBufferManagement(unsigned *sampsFromUsbToAudio, unsigned *sampsFromAudioToUsb)
{
    xassert(NULL != m_dsp);

    static int32_t input_buf[N_CHANS_TO_DSP][ADSP_AUTO_FRAME_SIZE];
    static int32_t output_buf[N_CHANS_FROM_DSP][ADSP_AUTO_FRAME_SIZE];
    static int32_t *dsp_input[N_CHANS_TO_DSP];
    static int32_t *dsp_output[N_CHANS_FROM_DSP];

    const unsigned start_frame_value = ADSP_AUTO_FRAME_SIZE - 1;
    static unsigned frame_counter = start_frame_value;

    // If first frame, assing the pointers to the input buffer
    if (frame_counter == start_frame_value){
        for (unsigned ch = 0; ch < N_CHANS_TO_DSP; ch++){
            dsp_input[ch] = &input_buf[ch][0];
        }
        for (unsigned ch = 0; ch < N_CHANS_FROM_DSP; ch++){
            dsp_output[ch] = &output_buf[ch][0];
        }
    }

    // Collect input samples into the channel buffers
    for(unsigned i = 0; i < NUM_USB_CHAN_OUT; i++) {
        dsp_input[i][frame_counter] = ((int32_t *)sampsFromUsbToAudio)[i];
    }
    for(unsigned i = 0; i < I2S_CHANS_ADC; i++) {
        dsp_input[i + NUM_USB_CHAN_OUT][frame_counter] = ((int32_t *)sampsFromAudioToUsb)[i];
    }

    // Increment frame counter
    frame_counter++;

    // If we have enough samples, hand over to the DSP thread.
    if (frame_counter >= ADSP_AUTO_FRAME_SIZE){
        adsp_pipeline_source(m_dsp, dsp_input);
        adsp_pipeline_sink(m_dsp, dsp_output);
        frame_counter = 0;
    }

    // Write the output buffers out over USB/I2S
    for(unsigned i = 0; i < NUM_USB_CHAN_IN; i++) {
        sampsFromAudioToUsb[i] = (unsigned)dsp_output[i][frame_counter];
    }
    for(unsigned i = 0; i < I2S_CHANS_DAC; i++) {
        sampsFromUsbToAudio[i] = (unsigned)dsp_output[i + NUM_USB_CHAN_IN][frame_counter];
    }
}

// start dsp_thread
void dsp_thread(void) {
    // Initialise the DSP instance and enter the generated DSP main function.
    // This will never return.
    m_dsp = adsp_auto_pipeline_init();
    adsp_auto_pipeline_main(m_dsp);
}
// end dsp_thread
