#include <xcore/assert.h>
#include <xua.h>
#include <xua_audiohub.h>
#include <stdint.h>
#include <stages/adsp_pipeline.h>
#include <adsp_generated_auto.h>

// This assumes that the application has not enabled MIDI, SPDIF, ADAT, or PDM mics.
#define N_CHANS_FROM_AUDIO (I2S_CHANS_ADC)
#define N_CHANS_TO_AUDIO (I2S_CHANS_DAC)
#define N_CHANS_TO_DSP (NUM_USB_CHAN_OUT + I2S_CHANS_ADC)
#define N_CHANS_FROM_DSP (NUM_USB_CHAN_IN + I2S_CHANS_DAC)

static adsp_pipeline_t *m_dsp;

/// Fills an array of points to point to the corresponding element
/// of an input array.
static void arr_to_pointer(int32_t** pointer, unsigned* arr, int n) {
    for(int i = 0; i < n; ++i) {
        pointer[i] = &((int32_t*)arr)[i];
    }
}

// This should match the DSP pipeline frame size
#define ADSP_AUTO_FRAME_SIZE (1)

#if ADSP_AUTO_FRAME_SIZE == 1

    /// Standard lib_xua callback function for DSP frame size of 1
    void UserBufferManagement(unsigned* sampsFromUsbToAudio, unsigned* sampsFromAudioToUsb) {
        xassert(NULL != m_dsp);

        // construct DSP input arrays and hand over to the DSP thread.
        int32_t* dsp_input[N_CHANS_TO_DSP];
        arr_to_pointer(&dsp_input[0], sampsFromUsbToAudio, NUM_USB_CHAN_OUT);
        arr_to_pointer(&dsp_input[NUM_USB_CHAN_OUT], sampsFromAudioToUsb, N_CHANS_FROM_AUDIO);
        adsp_pipeline_source(m_dsp, dsp_input);

        // construct the DSP output arrays and wait for output from the DSP
        // thread.
        int32_t* dsp_output[N_CHANS_FROM_DSP];
        arr_to_pointer(&dsp_output[0], sampsFromUsbToAudio, N_CHANS_TO_AUDIO);
        arr_to_pointer(&dsp_output[N_CHANS_TO_AUDIO], sampsFromAudioToUsb, NUM_USB_CHAN_IN);
        adsp_pipeline_sink(m_dsp, dsp_output);
    }

#else

    /// Alternative lib_xua callback function for DSP frame size of 2 or more
    void UserBufferManagement_(unsigned* sampsFromUsbToAudio, unsigned* sampsFromAudioToUsb) {
        xassert(NULL != m_dsp);

        // Set up buffers for each input channel
        // The number of required buffers is defined in xua_conf.h
        static int32_t input_buff_0[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_1[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_2[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_3[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_4[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_5[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_6[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_7[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_8[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_9[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_10[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_11[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_12[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_13[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_14[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t input_buff_15[ADSP_AUTO_FRAME_SIZE] = {0};

        // set up buffers for each output channel
        // The number of required buffers is defined in xua_conf.h
        static int32_t output_buff_0[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_1[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_2[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_3[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_4[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_5[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_6[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_7[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_8[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_9[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_10[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_11[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_12[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_13[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_14[ADSP_AUTO_FRAME_SIZE] = {0};
        static int32_t output_buff_15[ADSP_AUTO_FRAME_SIZE] = {0};

        // collect pointers to the channel buffers
        static int32_t* dsp_input[NUM_USB_CHAN_OUT + I2S_CHANS_ADC] = {input_buff_0, input_buff_1, input_buff_2,
                                                                       input_buff_3, input_buff_4, input_buff_5,
                                                                       input_buff_6, input_buff_7, input_buff_8,
                                                                       input_buff_9, input_buff_10, input_buff_11,
                                                                       input_buff_12, input_buff_13, input_buff_14,
                                                                       input_buff_15};
        static int32_t* dsp_output[NUM_USB_CHAN_IN + I2S_CHANS_DAC] = {output_buff_0, output_buff_1, output_buff_2,
                                                                       output_buff_3, output_buff_4, output_buff_5,
                                                                       output_buff_6, output_buff_7, output_buff_8,
                                                                       output_buff_9, output_buff_10, output_buff_11,
                                                                       output_buff_12, output_buff_13, output_buff_14,
                                                                       output_buff_15};
        
        static int32_t counter = ADSP_AUTO_FRAME_SIZE - 1;
        
        // Collect input samples into the channel buffers
        for(int i = 0; i < NUM_USB_CHAN_OUT; i++) {
            dsp_input[i][counter] = ((int32_t *)sampsFromUsbToAudio)[i];
        }
        for(int i = 0; i < I2S_CHANS_ADC; i++) {
            dsp_input[i + NUM_USB_CHAN_OUT][counter] = ((int32_t *)sampsFromAudioToUsb)[i];
        }
    
        counter += 1;
    
        if (counter >= ADSP_AUTO_FRAME_SIZE){
            // once the buffer is full, send/receive with the DSP thread
            adsp_pipeline_source(m_dsp, dsp_input);
            adsp_pipeline_sink(m_dsp, dsp_output);
            counter = 0;
        }
    
        // Write the output buffers out over USB/I2S
        for(int i = 0; i < NUM_USB_CHAN_IN; i++) {
            sampsFromAudioToUsb[i] = (unsigned)dsp_output[i][counter];
        }
        for(int i = 0; i < I2S_CHANS_DAC; i++) {
            sampsFromUsbToAudio[i] = (unsigned)dsp_output[i + NUM_USB_CHAN_IN][counter];
        }
    }
#endif

// start dsp_thread
void dsp_thread(void) {
    // Initialise the DSP instance and enter the generated DSP main function.
    // This will never return.
    m_dsp = adsp_auto_pipeline_init();
    adsp_auto_pipeline_main(m_dsp);
}
// end dsp_thread
