// Copyright 2024 XMOS LIMITED.
// This Software is subject to the terms of the XMOS Public Licence: Version 1.
#include <string.h>
#include <stdlib.h>
#include <xcore/assert.h>
#include <debug_print.h>
#include "cmds.h" // Autogenerated
#include "cmd_offsets.h" // Autogenerated
#include "stages/noise_gate.h"
#include "dsp/adsp.h"

static inline void ng_copy_config_to_state(noise_gate_t *ng_state, int n_inputs, const noise_gate_config_t *ng_config)
{
    // Same config for all channels
    for(int i=0; i<n_inputs; i++)
    {
        ng_state[i].env_det.attack_alpha = ng_config->attack_alpha;
        ng_state[i].env_det.release_alpha = ng_config->release_alpha;
        ng_state[i].threshold = ng_config->threshold;
    }
}

static inline void ng_copy_state_to_config(noise_gate_config_t *ng_config, const noise_gate_t *ng_state)
{
    // Copy from channel 0 state to the config
    ng_config->attack_alpha = ng_state[0].env_det.attack_alpha;
    ng_config->release_alpha = ng_state[0].env_det.release_alpha;
    ng_config->envelope = ng_state[0].env_det.envelope;
    ng_config->gain = ng_state[0].gain;
    ng_config->threshold = ng_state[0].threshold;
}

void noise_gate_process(int32_t **input, int32_t **output, void *app_data_state)
{
    xassert(app_data_state != NULL);
    noise_gate_state_t *state = app_data_state;

    // do while saves instructions for cases
    // where the loop will always execute at
    // least once
    int i = 0;
    do {
        int32_t *in = input[i];
        int32_t *out = output[i];

        int j = 0;
        do {
            *out++ = adsp_noise_gate(&state->ng[i], *in++);
        } while(++j < state->frame_size);
    } while(++i < state->n_outputs);
}

void noise_gate_init(module_instance_t* instance, adsp_bump_allocator_t* allocator, uint8_t id, int n_inputs, int n_outputs, int frame_size)
{
    xassert(n_inputs == n_outputs && "Noise gate should have the same number of inputs and outputs");
    noise_gate_state_t *state = instance->state;
    noise_gate_config_t *config = instance->control.config;

    memset(state, 0, sizeof(noise_gate_state_t));
    state->n_inputs = n_inputs;
    state->n_outputs = n_outputs;
    state->frame_size = frame_size;

    state->ng = adsp_bump_allocator_malloc(allocator, NOISE_GATE_STAGE_REQUIRED_MEMORY(state->n_inputs));
    memset(state->ng, 0, NOISE_GATE_STAGE_REQUIRED_MEMORY(state->n_inputs));

    for(int i=0; i<state->n_inputs; i++)
    {
        state->ng[i].gain = INT32_MAX;
        state->ng[i].env_det.envelope = 1 << (-SIG_EXP);
    }

    ng_copy_config_to_state(state->ng, state->n_inputs, config);
}

void noise_gate_control(void *module_state, module_control_t *control)
{
    xassert(module_state != NULL);
    noise_gate_state_t *state = module_state;
    xassert(control != NULL);
    noise_gate_config_t *config = control->config;

    if(control->config_rw_state == config_write_pending)
    {
        // Finish the write by updating the working copy with the new config
        // TODO update only the fields written by the host
        ng_copy_config_to_state(state->ng, state->n_inputs, config);
        control->config_rw_state = config_none_pending;
    }
    else if(control->config_rw_state == config_read_pending)
    {
        ng_copy_state_to_config(config, state->ng);
        control->config_rw_state = config_read_updated;
    }
    else
    {
        // nothing to do
    }
}
