// Copyright 2024 XMOS LIMITED.
// This Software is subject to the terms of the XMOS Public Licence: Version 1.
#include <string.h>
#include <stdlib.h>
#include <xcore/assert.h>
#include <debug_print.h>
#include "cmds.h" // Autogenerated
#include "cmd_offsets.h" // Autogenerated
#include "stages/clipper.h"

void clipper_process(int32_t **input, int32_t **output, void *app_data_state)
{
    xassert(app_data_state != NULL);
    clipper_state_t *state = app_data_state;

    // do while saves instructions for cases
    // where the loop will always execute at
    // least once
    int i = 0;
    do {
        int32_t *in = input[i];
        int32_t *out = output[i];

        int j = 0;
        do {
            *out++ = adsp_clipper(state->clip[i], *in++);
        } while(++j < state->frame_size);
    } while(++i < state->n_outputs);
}

void clipper_init(module_instance_t* instance, adsp_bump_allocator_t* allocator, uint8_t id, int n_inputs, int n_outputs, int frame_size)
{
    xassert(n_inputs == n_outputs && "Clipper should have the same number of inputs and outputs");
    clipper_state_t *state = instance->state;
    clipper_config_t *config = instance->control.config;

    memset(state, 0, sizeof(clipper_state_t));
    state->n_inputs = n_inputs;
    state->n_outputs = n_outputs;
    state->frame_size = frame_size;

    state->clip = adsp_bump_allocator_malloc(allocator, CLIPPER_STAGE_REQUIRED_MEMORY(state->n_inputs));
    memset(state->clip, 0, CLIPPER_STAGE_REQUIRED_MEMORY(state->n_inputs));

    for (int i = 0; i < state->n_inputs; i++) {
        state->clip[i] = config->threshold;
    }
}

void clipper_control(void *module_state, module_control_t *control)
{
    xassert(module_state != NULL);
    clipper_state_t *state = module_state;
    xassert(control != NULL);
    clipper_config_t *config = control->config;

    if(control->config_rw_state == config_write_pending)
    {
        // Finish the write by updating the working copy with the new config
        // TODO update only the fields written by the host
        for (int i = 0; i < state->n_inputs; i++) {
            state->clip[i] = config->threshold;
        }
        control->config_rw_state = config_none_pending;
    }
    else if(control->config_rw_state == config_read_pending)
    {
        config->threshold = state->clip[0];
        control->config_rw_state = config_read_updated;
    }
    else
    {
        // nothing to do
    }
}
