#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>

static const uint32_t n_bit_tests[][2] = {
  {1, 1}, //CRC-1
  {3, 0x6}, //CRC-3-GSM
  {4, 0xc}, //CRC-4-ITU
  {5, 0x12}, //CRC-5-EPC
  {5, 0x15}, //CRC-5-USB
  {5, 0x14}, //CRC-5-ITU
  {6, 0x39}, //CRC-6-CDMA2000-A
  {6, 0x38}, //CRC-6-CDMA2000-B
  {6, 0x26}, //CRC-6-DARC
  {6, 0x3d}, //CRC-6-GSM
  {6, 0x30}, //CRC-6-ITU
  {7, 0x48}, //CRC-7
  {7, 0x53}, //CRC-7-MVB
  {8, 0xab}, //CRC-8
  {8, 0xf4}, //CRC-8-AUTOSAR
  {8, 0xe5}, //CRC-8-Bluetooth
  {8, 0xe0}, //CRC-8-CCITT
  {8, 0x8c}, //CRC-8-Dallas/Maxim	
  {8, 0x9c}, //CRC-8-DARC
  {8, 0x92}, //CRC-8-GSM-B
  {8, 0xb8}, //CRC-8-SAE J1850
  {8, 0xd9}, //CRC-8-WCDMA
  {10, 0x331}, //CRC-10
  {10, 0x26f}, //CRC-10-CDMA2000
  {10, 0x2ba}, //CRC-10-GSM
  {11, 0x50e}, //CRC-11
  {12, 0xf01} //CRC-12
};

uint32_t bitrev(uint32_t x) {
    x = (((x & 0xaaaaaaaa) >> 1) | ((x & 0x55555555) << 1));
    x = (((x & 0xcccccccc) >> 2) | ((x & 0x33333333) << 2));
    x = (((x & 0xf0f0f0f0) >> 4) | ((x & 0x0f0f0f0f) << 4));
    x = (((x & 0xff00ff00) >> 8) | ((x & 0x00ff00ff) << 8));
    return((x >> 16) | (x << 16));
}

uint32_t crc32_ref(uint32_t len, uint8_t *buf, uint32_t init, 
                    uint32_t final_xor, uint32_t poly){
    uint32_t val, crc = init;
    while(len--){
        val = (crc^*buf++)&0xff;
        for(int i=0; i < 8; i++)
            val = val & 1 ? (val>>1)^poly : val>>1;
        crc = val^(crc>>8);
    }
    return crc^final_xor;
}

uint32_t crc_xcore(uint32_t crc, uint32_t data, uint32_t poly, unsigned bit_count){
    for(int i=0;i<bit_count;i++){
      if(crc&1)
        crc = (((data&1)<<31)|(crc>>1))^poly;
      else 
        crc =  ((data&1)<<31)|(crc>>1);
      data >>= 1;
    }
    return crc;
}

uint32_t crc32_asm(uint32_t crc, uint32_t data, uint32_t poly){
  return crc_xcore(crc, data, poly, 32);
}

uint32_t crc8_asm(uint32_t crc, uint32_t data, uint32_t poly){
  return crc_xcore(crc, data, poly, 8);
}

uint32_t crcn_asm(uint32_t crc, uint32_t data, uint32_t poly, unsigned n)
{
  return crc_xcore(crc, data, poly, n);
}

uint32_t crcn(uint32_t crc, uint32_t data, uint32_t poly, unsigned n)
{
  if (n < 32){ //branch due to shift right of 32 undefined in C
    uint32_t shifted_crc = crc << (32 - n);
    uint32_t shifted_data = crc >> n | (data << (32 - n));
    return crc32_asm(shifted_crc, shifted_data, poly);
  } else {
    return crc32_asm(crc, data, poly);
  }
}

#define ARRAY_SIZE(x) (sizeof(x) / sizeof(x[0]))

uint32_t crc32_rev(uint32_t data, uint32_t crc, uint32_t poly)
{
  assert(poly >> 31); // Only works if the most significant bit of poly is set.
  crc = bitrev(crc^data);;
  crc = crc32_asm(crc, 0, bitrev((poly << 1) | 1));
  return bitrev(crc);
}

uint32_t make_magic_number(uint32_t poly)
{
  uint32_t data[2] = {0, ~0};
  uint32_t crc = 0;
  for (int i = ARRAY_SIZE(data) - 1; i > 0; i--)
    crc = crc32_rev(data[i], crc, poly);
  return crc;
}

void check_magic_number(uint32_t poly, uint32_t magic)
{
  // Check that CRC of the array is zero as expected.
  uint32_t data[2] = {magic, ~0};
  uint32_t crc = 0;
  for (int i = 0; i < ARRAY_SIZE(data); i++)
    crc = crc32_asm(crc, data[i], poly);
  assert(crc == 0);
}

#define ETHERNET_POLY 0xEDB88320
#define ETHERNET_INITIAL_VALUE 0xffffffff
#define ETHERNET_FINAL_XOR 0xffffffff

#define USB_POLY 0x14
#define USB_INITIAL_VALUE 0x1f
#define USB_FINAL_XOR 0x1f

void test_ethernet_rx(){

  const unsigned int length = 20;
  uint8_t data[length];
  for (int i=0;i<length;i++)
    data[i] = rand();

  //Generate the CRC of 00 00 00 00 from the spec.
  uint8_t zero[4] = {0, 0, 0, 0};
  uint32_t zero_crc = crc32_ref(sizeof(zero), zero, 
    ETHERNET_INITIAL_VALUE, ETHERNET_FINAL_XOR, ETHERNET_POLY);

  // Ingest all the data
  uint32_t crc = crc32_ref(sizeof(data), data, 
    ETHERNET_INITIAL_VALUE, ETHERNET_FINAL_XOR, ETHERNET_POLY); 
  // This CRC must be appended to the frame. 

  //Create a new frame as it would be recieved
  uint8_t data_with_crc[sizeof(data)+sizeof(crc)];
  memcpy(data_with_crc, data, sizeof(data));
  memcpy(data_with_crc+sizeof(data), (uint8_t*)&crc, sizeof(crc));

  // Verify that the crc we computed is correct
  uint32_t crc_spec = crc32_ref(sizeof(data_with_crc), data_with_crc, 
    ETHERNET_INITIAL_VALUE, ETHERNET_FINAL_XOR, ETHERNET_POLY);   
  // To be correct the CRC over all the data plus the appened CRC should 
  // be equal to the CRC of 4 bytes of zeros.
  assert(crc_spec == zero_crc);

  // The CRC32 instruction
  uint32_t residual = make_magic_number(ETHERNET_POLY);
  check_magic_number(ETHERNET_POLY, residual);
  for(int i=0;i<sizeof(data_with_crc)/sizeof(uint32_t);i++)
      residual = crc32_asm(residual, ((uint32_t*)data_with_crc)[i], ETHERNET_POLY);
  // If all is correct then the residual should be all 0xff's.
  assert(residual == ~0);
}

void test_ethernet_tx(){

  const unsigned int length = 4*5;
  uint8_t data[length];
  for (int i=0;i<length;i++)
    data[i] = rand();

  uint32_t init = ETHERNET_INITIAL_VALUE;
  uint32_t final_xor = ETHERNET_FINAL_XOR;

  // As per the spec the CRC should be initialised with 0xffffffff
  // and should end with an xor with 0xffffffff.
  uint32_t specification_crc = crc32_ref(sizeof(data), data, 
    init, final_xor, ETHERNET_POLY);

  //The XMOS CRC32 instruction
  uint32_t xmos_crc = init^(((uint32_t*)data)[0]);
  for(int i=1;i<sizeof(data)/sizeof(uint32_t);i++)
      xmos_crc = crc32_asm(xmos_crc, ((uint32_t*)data)[i], ETHERNET_POLY);
  xmos_crc = crc32_asm(xmos_crc, final_xor, ETHERNET_POLY);

  assert(specification_crc == xmos_crc);

}

void test_crcn(){
  for(int i=1;i<16;i++){
    uint32_t data = rand(), crc = rand();
    uint32_t a = crcn(data, crc, ETHERNET_POLY, i%32+1);
    uint32_t b = crcn_asm(data, crc, ETHERNET_POLY, i%32+1);
    assert(a == b);
  }
}

void test_n_bit_poly(){

  for(int idx=0;idx< sizeof(n_bit_tests) / sizeof(n_bit_tests[0]); idx++){

    uint32_t poly = n_bit_tests[idx][0];
    uint32_t poly_len = n_bit_tests[idx][0];

    const unsigned int length = 4*12;
    uint8_t data[length];
    for (int i=0;i<length;i++)
      data[i] = rand();
    
    uint32_t mask = (1<<(poly_len)) - 1;
    uint32_t init = rand()&mask;
    uint32_t final_xor = rand()&mask;

    uint32_t spec_crc = crc32_ref(sizeof(data), data,  init, final_xor, poly);

    //Using the CRC32 instruction

    //First XOR in the initial CRC into the data 
    for(int len=0;len < poly_len; len +=8){
      data[len/8] = (init ^ data[len/8]);
      init >>= 8;
    }

    //Then perform the data ingestion as normal
    uint32_t xmos_crc = 0;
    for(int i=0;i<sizeof(data);i++){
      //crc8_asm consumes one byte at a time
      xmos_crc = crc8_asm(xmos_crc, data[i], poly); 
    }

    //Finally, ingest 32 zeros with the final_xor included.
    xmos_crc = crc32_asm(xmos_crc, final_xor, poly);

    assert(spec_crc == xmos_crc);
  }
}




int main(){
  srand ( time(NULL) );
  for(int i=0;i<16;i++){
    test_crcn();
    test_n_bit_poly();
    test_ethernet_rx();
    test_ethernet_tx();
  }
  return 0;
}