import contextlib
import os
from pathlib import Path
import sys
from time import sleep
import xtagctl.xtagctl_lib as xtagctl_lib
from typing import List

import logging

log = logging.getLogger(__name__)

# Import all exceptions
from xtagctl.exceptions import *


def reset_adapter(adapter_id: str):
    """ Reset a connected adapter """
    _reset_adapter(adapter_id)


def _reset_adapter(adapter_id: str, lock_dir: Path = None, config_dir: Path = None):
    with xtagctl_lib.device_lock(lock_dir):
        dev_controller = xtagctl_lib.DeviceController(lock_dir, config_dir)
        dev_controller.reset_adapter(adapter_id)


def reset_all_adapters(*targets):
    """ Reset all connected adapters matching any of targets """
    _reset_all_adapters(targets)


def _reset_all_adapters(targets: list, lock_dir: Path = None, config_dir: Path = None):
    with xtagctl_lib.device_lock(lock_dir):
        dev_controller = xtagctl_lib.DeviceController(lock_dir, config_dir, verbose=True)
        dev_controller.reset_all_adapters(targets)


@contextlib.contextmanager
def acquire(*args, timeout: int = 600):
    """ Contextlib interface to acquire and safely release targets """

    with _acquire(*args, timeout=timeout) as adapter_id:
        yield adapter_id


@contextlib.contextmanager
def _acquire(
    *targets: List[str],
    timeout: int = 600,
    lock_dir: Path = None,
    config_dir: Path = None,
    xrun_bin: str = "xrun",
):
    """Contextlib interface to acquire and safely release targets
    This function is written such that a KeyboardInterrupt does not leave acquired xtags
    """
    count = 0
    adapter_ids = []

    def _release_adapters_top_level():
        # Do not timeout here! We must release what we have acquired
        with xtagctl_lib.GracefulExit():
            with xtagctl_lib.device_lock(lock_dir, timeout=-1):
                dev_controller = xtagctl_lib.DeviceController(
                    lock_dir, config_dir, xrun_bin
                )
                for adapter in adapter_ids:
                    try:
                        dev_controller.release_adapter(adapter)
                    except XtagctlException:
                        pass

    try:
        while len(adapter_ids) < len(targets) and (count < timeout or timeout < 0):
            with xtagctl_lib.GracefulExit():
                with xtagctl_lib.device_lock(lock_dir):
                    dev_controller = xtagctl_lib.DeviceController(
                        lock_dir, config_dir, xrun_bin
                    )
                    adapters_before = dev_controller.dev.acquired.copy()

                    def _release_adapters_during_acquire():
                        # Release all targets acquired so far
                        acquired = set(dev_controller.dev.acquired) - set(adapters_before)
                        for adapter in acquired:
                            dev_controller.release_adapter(adapter)

                    try:
                        try:
                            for target_name in targets:
                                adapter_ids.append(
                                    dev_controller.acquire_target(target_name)
                                )
                        except xtagctl_lib.XtagctlDeviceInUse:
                            _release_adapters_during_acquire()
                            adapter_ids = []
                    except:
                        _release_adapters_during_acquire()
                        raise

            if len(adapter_ids) < len(targets):
                # Couldn't acquire all targets, retry in 1s
                sleep(1)
                logging.info("Waiting for device(s) to free...")
                count += 1

        if timeout > 0 and count >= timeout:
            raise xtagctl_lib.XtagctlException("Timeout during target acquire")

        if len(adapter_ids) == 1:
            yield adapter_ids[0]
        else:
            yield adapter_ids
    except:
        _release_adapters_top_level()
        raise
    finally:
        try:
            _release_adapters_top_level()
        except:
            log.critical("Exception caught, releasing adapters first...")
            _release_adapters_top_level()
            raise


def error(msg: str):
    print(msg, file=sys.stderr)
    sys.exit(1)


def main():
    try:
        xtagctl_lib.App()
    except xtagctl_lib.XtagctlException as e:
        error(str(e))
