import os, sys, stat
import tempfile
from threading import Thread
import time
import shutil
from typing import List, Tuple
from pathlib import Path
import multiprocessing

import pytest
from xtagctl.xtagctl_lib import DeviceController, XtagctlException
import xtagctl

import logging
logging.getLogger("xtagctl").propagate = True
logging.getLogger("xtagctl").setLevel(logging.DEBUG)


@pytest.fixture
def lock_dir():
    tmp_dir = Path(tempfile.mkdtemp())
    try:
        yield tmp_dir
    finally:
        shutil.rmtree(tmp_dir)


@pytest.fixture
def config_dir():
    tmp_dir = Path(tempfile.mkdtemp())
    try:
        yield tmp_dir
    finally:
        shutil.rmtree(tmp_dir)


def create_printer(bin_path: Path, message: str):
    """ Creates a binary that prints the desired lines when executed """

    with open(bin_path, "w") as f:
        f.write("#!/bin/bash\ncat << HERE\n" + message + "\nHERE\n")
    os.chmod(bin_path, stat.S_IRWXU)

    return bin_path


def fake_xrun(tmp_dir: Path, entries: List[Tuple[str, str, int]]):
    """Creates an xrun binary on the PATH which returns artificial entries

    Format:
    (adapterID: str, [0 (disconnected), 1 (in-use), 2 (ready)]
    """

    header = """
Available XMOS Devices
----------------------

"""
    table_header = """  ID	Name			Adapter ID	Devices
  --	----			----------	-------
"""
    none_entry = "  No Available Devices Found"
    row_fmt = "  {:<2}\t{:<20}\t{}\t{}\n"

    if not entries:
        xrun_out = header + none_entry
    else:
        xrun_out = header + table_header
        for i, e in enumerate(entries):
            if e[-1] == 0:
                status = "None"
            elif e[-1] == 1:
                status = "In Use"
            elif e[-1] == 2:
                status = "O[0]"
            xrun_out += row_fmt.format(i, "XMOS XTAG-2", e[0], status)
    xrun_out += "\n"

    xrun_path = create_printer(Path(tmp_dir) / "fake_xrun", xrun_out)

    return xrun_path


def test_acquire_ok(lock_dir, config_dir):
    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target
    dev_controller = DeviceController(lock_dir, config_dir, Path(xrun_path))
    dev_controller.acquire_target("123")

def test_acquire_regex_ok(lock_dir, config_dir):
    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target
    dev_controller = DeviceController(lock_dir, config_dir, Path(xrun_path))
    dev_controller.acquire_target("/12.*/")

def test_acquire_no_device_map(lock_dir, config_dir):
    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    dev_controller = DeviceController(lock_dir, config_dir, Path(xrun_path))
    try:
        dev_controller.acquire_target("abcdefgh")
        raise Exception("Acquire shouldn't have worked")
    except XtagctlException:
        pass

def test_acquire_missing(lock_dir, config_dir):
    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("bbcdefgh 123\n")  # AdapterID Target
        f.write("cbcdefgh 123\n")  # AdapterID Target
        f.write("ebcdefgh 123\n")  # AdapterID Target
    dev_controller = DeviceController(lock_dir, config_dir, Path(xrun_path))
    try:
        dev_controller.acquire_target("abcdefgh")
        raise Exception("Acquire shouldn't have worked")
    except XtagctlException:
        pass


def test_acquire_acquired(lock_dir, config_dir):
    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target
    dev_controller = DeviceController(lock_dir, config_dir, Path(xrun_path))
    dev_controller.acquire_target("123")
    try:
        dev_controller.acquire_target("123")
        raise Exception("Acquire shouldn't have worked")
    except XtagctlException:
        pass


def test_acquire_inuse(lock_dir, config_dir):
    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 1)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target
    dev_controller = DeviceController(lock_dir, config_dir, Path(xrun_path))
    try:
        dev_controller.acquire_target("123")
        raise Exception("Acquire shouldn't have worked")
    except XtagctlException:
        pass


def test_release_ok(lock_dir, config_dir):
    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target
    with open(lock_dir / "acquired", "w") as f:
        f.write("abcdefgh\n")  # AdapterID Target
    dev_controller = DeviceController(lock_dir, config_dir, Path(xrun_path))
    dev_controller.release_adapter("abcdefgh")


def test_release_missing(lock_dir, config_dir):
    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    with open(lock_dir / "acquired", "w") as f:
        f.write("abcdefgh\n")  # AdapterID Target
    dev_controller = DeviceController(lock_dir, config_dir, Path(xrun_path))
    try:
        dev_controller.release_adapter("abcdefgh")
        raise Exception("Release shouldn't have worked")
    except:
        pass


def test_cleanup_acquired(lock_dir, config_dir):
    """If an adapter appears in acquired file but not in device_map,
    it should be removed."""

    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    with open(lock_dir / "acquired", "w") as f:
        f.write("abcdefgh\n")  # AdapterID Target
    dev_controller = DeviceController(lock_dir, config_dir, Path(xrun_path))
    with open(lock_dir / "acquired", "r") as f:
        assert not "abcdefgh" in f.read()


def test_contextlib_api(lock_dir, config_dir):
    """ Tests use of xtagctl in a with statement """

    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target

    with xtagctl._acquire(
        "123", timeout=5, lock_dir=lock_dir, config_dir=config_dir, xrun_bin=xrun_path
    ) as f:
        print(f)

    print("Attempt to require it again (checks if release worked)")

    with xtagctl._acquire(
        "123", timeout=5, lock_dir=lock_dir, config_dir=config_dir, xrun_bin=xrun_path
    ) as f:
        print(f)


def test_multiple_acquire(lock_dir, config_dir):
    """ Tests use of xtagctl in a with statement (multiple devices)"""

    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2), ("bbcdefgh", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\nbbcdefgh 123\n")  # AdapterID Target

    with xtagctl._acquire(
        "123",
        "123",
        timeout=5,
        lock_dir=lock_dir,
        config_dir=config_dir,
        xrun_bin=xrun_path,
    ) as (f1, f2):
        print(f1, f2)


def test_cleanup_on_fail(lock_dir, config_dir):
    """ Tests use of xtagctl in a with statement when a "test" fails """

    xrun_path = fake_xrun(lock_dir, [("abcdefgh", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target

    try:
        with xtagctl._acquire(
            "123",
            timeout=5,
            lock_dir=lock_dir,
            config_dir=config_dir,
            xrun_bin=xrun_path,
        ) as f1:
            raise Exception("Nothing")
    except:
        # Catch exception
        pass

    # The with statement should've cleared everything up, so we can acquire again here
    with xtagctl._acquire(
        "123", timeout=5, lock_dir=lock_dir, config_dir=config_dir, xrun_bin=xrun_path
    ) as f1:
        print(f1)


def test_xrun_timeout(lock_dir, config_dir):

    # Create a fake xrun that sleeps for 30s
    xrun_path = Path(lock_dir) / "fake_xrun"
    with open(xrun_path, "w") as f:
        f.write("#!/bin/bash\nsleep 30")
    os.chmod(xrun_path, stat.S_IRWXU)

    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target

    test_pass = False
    try:
        with xtagctl._acquire(
            "123",
            timeout=5,
            lock_dir=lock_dir,
            config_dir=config_dir,
            xrun_bin=xrun_path,
        ) as f1:
            print("hi")
    except xtagctl.xtagctl_lib.XtagctlXrunTimeout:
        # Expect the call to timeout
        test_pass = True

    assert test_pass


def test_error_on_disconnected_xtag(lock_dir, config_dir):
    """ Test that xtag doesn't retry when a device is disconnected """

    # Create an xrun where the device is not connected
    xrun_path = fake_xrun(lock_dir, [("something", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target

    exception_found = False
    start = time.time()
    try:
        with xtagctl._acquire(
            "123",
            timeout=5,
            lock_dir=lock_dir,
            config_dir=config_dir,
            xrun_bin=xrun_path,
        ) as f1:
            print("hello")
    except xtagctl.xtagctl_lib.XtagctlDeviceNotConnected:
        # Expect an exception
        exception_found = True
    end = time.time()

    # Shouldn't have looped for ages before raising an exception
    assert end - start < 15
    # Should have seen an exception
    assert exception_found


def test_atomic_acquire(lock_dir, config_dir):
    """Test that while one thread is attempting to acquire multiple targets,
    another thread can acquire those targets one at a time
    """

    xrun_path = fake_xrun(lock_dir, [("absdddd", 2), ("defdddd", 2)])
    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("absdddd dut\n")  # AdapterID Target
        f.write("defdddd harness\n")  # AdapterID Target

    def acquire_both():
        with xtagctl._acquire(
            "dut",
            "harness",
            timeout=5,
            lock_dir=lock_dir,
            config_dir=config_dir,
            xrun_bin=xrun_path,
        ) as (dut, harness):
            print("Got them both")

    with xtagctl._acquire(
        "dut", timeout=5, lock_dir=lock_dir, config_dir=config_dir, xrun_bin=xrun_path
    ) as dut:
        # Start another thread and attempt to acquire both the DUT and Harness
        other_thread = Thread(target=acquire_both)
        other_thread.start()
        with xtagctl._acquire(
            "harness",
            timeout=5,
            lock_dir=lock_dir,
            config_dir=config_dir,
            xrun_bin=xrun_path,
        ) as harness:
            print("Use the harness for something...")
            time.sleep(1)
            # The other thread should still be attempting to acquire both targets
            assert other_thread.is_alive()

        # The other thread should still be attempting to acquire both targets
        assert other_thread.is_alive()

    time.sleep(2)
    # The other thread should've been able to acquire both targets by now
    assert not other_thread.is_alive()


def test_xrun_too_short(lock_dir, config_dir):
    """ Tests xtagctl error handling when xrun output is too short """

    # Create an xrun that prints only one line
    xrun_path = create_printer(lock_dir / "short", "Something not right")

    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("absdddd dut\n")  # AdapterID Target

    try:
        with xtagctl._acquire(
            "dut", timeout=5, lock_dir=lock_dir, config_dir=config_dir, xrun_bin=xrun_path
        ) as dut:
            print("Hello")
        # If we get here, no exception was raised!
        assert False
    except XtagctlException:
        # Found exception
        pass


def test_sigterm(lock_dir, config_dir):
    # Create a fake xrun that sleeps for 30s
    xrun_path = Path(lock_dir) / "fake_xrun"
    with open(xrun_path, "w") as f:
        f.write("#!/bin/bash\nsleep 30")
    os.chmod(xrun_path, stat.S_IRWXU)

    # Create a device_map with other entries in
    with open(config_dir / "device_map", "w") as f:
        f.write("abcdefgh 123\n")  # AdapterID Target

    def new_proc(lock_dir, config_dir, xrun_path):
        with xtagctl._acquire(
            "123",
            timeout=5,
            lock_dir=lock_dir,
            config_dir=config_dir,
            xrun_bin=xrun_path,
        ) as f1:
            print("hi")

    # Spawn a new process
    p = multiprocessing.Process(target=new_proc, args=(lock_dir, config_dir, xrun_path))
    p.start()

    # Kill the process with SIGTERM
    p.terminate()

    # Assert that the process has died
    time.sleep(2)
    assert not p.is_alive()

    # Make sure we can lock OK
    with xtagctl.xtagctl_lib.device_lock(lock_dir, timeout=1):
        print("Got lock")
