xtagctl
=======

Software for managing multiple xCore devices on a single host.

Hardware setup:

 - Each xtag is connected to a single known xCore device. 
 - The mapping between xCore devices and xTag adapter IDs is stored in a `device_map` file.
 - Optionally, specify either a reset pin wired to a FT232H board, OR the keyword 'uhubctl' if the xtag is connected to a compatible smart usb hub.
 - To maintain backwards compatibility, we put any 'uhubctl' mappings in a new `uhubctl_map` mapping file.
 - If the smart hub and board configuration requires it, we can use a comma separated, zero space list of uhubctl command arguments.

   **Driver and permissions setup for the board are detailed [here](https://learn.adafruit.com/circuitpython-on-any-computer-with-ft232h/setup).**

   **See uhubctl on github [here](https://github.com/mvp/uhubctl)**

The format for the device_map is:
 ```
 # Example comment

 <adapter-id> <target-name> [reset-pin/uhubctl]
 <adapter-id> <target-name> [reset-pin/uhubctl]
 ...
 ```

 Specifying a reset pin is optional.
 
 The target names can be used multiple times e.g.
 ```
 Gw1JBssi xvf3510 C0
 OifU63gd xvf3510 C1
 ```
 
 To maintain backwards compatibility, we separate all uhubctl commands into a new mapping file called `uhubctl_map`, both globally and locally.
 e.g. 
 ```
 Gw1JBssi xvf3510 uhubctl
 OifU63gd xvf3510 uhubctl[-a2,-l3-1,-p2]
 DqaAw43c xvf3510 uhubctl[-a2,-l3-1.3,-p4,-r100]
```


## Installation

Requires Uhubctl 2.4+ for automatic power control.
Requires Python 3.7+ so you may need to use pyenv on older systems.

`pip install -e xtagctl`

This will put the xtagctl command on your PATH.

## Configuration

By default, xtagctl will store all it's program files in `~/.xtag`.

The default behavior can be changed using these environment variables:

- `XTAGCTL_CONFIG_DIR`: The directory that contains the local `device_map` and the `uhubctl_map` file.
- `XTAGCTL_LOCK_DIR`: The directory that contains the `lock` and `acquired` files.

**The global `device_map` and `uhubctl_map` for the organisation are stored in a git repo here:
[xtagctl_config](https://github0.xmos.com/xmos-int/xtagctl_config/)**

This global mapping should be kept up-to-date with all adapter mappings
connected to the hardware test infrastructure. The latest mapping will be
downloaded on each invocation of xtagctl. All uhubctl mappings must be
stored only in the `uhubctl_map` file.

If the global mapping can't be fetched, xtagctl will fallback to using only the
local mapping. In the normal case, local mappings are appended to the global
mapping, and local maps take priority, with `uhubctl_map` being the highest priority.
So priority order from first to last is:

  - local uhubctl_map
  - local device_map
  - global uhubctl_map
  - global device_map

An example of a local uhubctl_map:
```
more ~/.xtag/uhubctl_map
# sw-hw-xcai-aug0
TU7F9RC1 XCORE-AI-EXPLORER-V2 uhubctl[-a2,-l4-1,-p3]
```

## Python API

```python
import sh
import time
import xtagctl

def test_usb_audio():
    # Using the with statement handles the acquire/release process for you
    with xtagctl.acquire("usb_audio_xs2_mc_dut") as adapter_id:
        # Reset adapter before running test
        xtagctl.reset_adapter(adapter_id)
        time.sleep(2) # Wait for adapter to enumerate
        # xrun firmware
        sh.xrun("--adapter-id", adapter_id, "usb_audio.xe")
        # Run some tests
        raise Exception()
        # The device will still be released even if exceptions are raised 

def test_target_regex():
    # Passing a slash-wrapped string will match targets against a regex pattern
    with xtagctl.acquire("/usb_audio_.*_dut/") as adapter_id:
        # Reset adapter before running test
        xtagctl.reset_adapter(adapter_id)
        time.sleep(2) # Wait for adapter to enumerate
        # xrun firmware
        sh.xrun("--adapter-id", adapter_id, "usb_audio.xe")
        # Run some tests
        raise Exception()
        # The device will still be released even if exceptions are raised 

def test_multiple_targets():
    # Multiple targets should be acquired atomically in a single with statement
    # to avoid the Dining Philosophers problem
    with xtagctl.acquire("xvf3510_dut", "xvf3510_harness") as (dut_id, harness_id):
        # Reset adapters before running test
        xtagctl.reset_adapter(dut_id)
        xtagctl.reset_adapter(harness_id)
        time.sleep(2) # Wait for adapters to enumerate
        # xrun firmware
        sh.xrun("--adapter-id", dut_id, "usb_audio.xe")
        sh.xrun("--adapter-id", harness_id, "test_harness.xe")
        # Run some tests
        raise Exception()
        # The device will still be released even if exceptions are raised 

```

## Wiring example

![Wiring Example](/images/xtag_reset_wiring.jpeg)

This setup is described by the following `device_map` file:
```
Y5QQDRlP    usb_audio_mc_xs2_dut        C0
0WlQ6AZf    usb_audio_mc_xs2_harness    C1
```

## Commands

- `xtagctl acquire <TARGET>`

Attempts to acquire the specified target.

Accepts a regex pattern if wrapped in forward slashes `/TARGET_PATTERN/`.

If the device is missing from the `device_map`, the command will exit with a
non-zero return code.

If the device is already acquired, the command will exit with a non-zero return
code.

If the device is in-use, the command will exit with a non-zero return code.


- `xtagctl release <adapterID>`

Frees the xtag with the specified adapter ID.

If the xtag specified doesn't exist in the device map, the command will exit
with a non-zero return code.


- `xtagctl reset <adapterID>`

Resets the xtag with the specified `adapterID`:
```
xtagctl reset olD44Kiw
```

If the xtag specified doesn't have a reset pin mapped in the device map, the 
command will print a warning but continue to return 0.


- `xtagctl reset_all <Targets>`

Resets all xtags with one of the matching board `Targets`. Can be
passed any positive number of space separated targets:
```
xtagctl reset_all xcore-ai-explorer xcore-ai-explorer-v2
```

Also accepts a regex pattern if wrapped in forward slashes `/TARGET_PATTERN/`:
```
xtagctl reset_all /xcore.*/
```

If a specified xtag doesn't have a reset method mapped in the device map, the 
command will print a warning but continue to return 0.


- `xtagctl status`

Display a representation of the device map, including which devices are
connected, disconnected, or in-use. This command is read-only.

Will also show connected devices which are not yet mapped.


## Architecture

The `device_map` file is never written to by xtagctl.

The list of acquired adapters is stored in `~/.xtag/acquired`. This should not
be edited manually.

Modifying the acquired file is controlled by a status.lock file.

## Troubleshooting

### Logging

You can enable logging for xtagctl - logging is handled via python's built-in logging
library. See the official docs, or
[this youtube video](https://www.youtube.com/watch?v=DxZ5WEo4hvU).

To enable logging in tests, put the following snippet at the top of your pytest file:

```python
import logging
logging.getLogger("xtagctl").propagate = True
logging.getLogger("xtagctl").setLevel(logging.DEBUG)
```

Then enable live logging in your pytest.ini:
```ini
[pytest]
log_cli = 1
log_cli_level = INFO
```

### Hanging

xtagctl will clean up after a failure if terminated via SIGINT (Ctrl-C) or SIGTERM.

If xtagctl is killed by another signal, it may leave the lock file and acquired file in
a messy state. To fix this, remove the status.lock file and the acquired file in the
xtagctl config directory.

**Note:** If using Jenkins, ensure that you're removing the files for the jenkins user!
