#include <set>
#include <volcontrol.h>
#import <AudioToolbox/AudioServices.h>


#include <string.h>

const char *gClockNameString[3] =
{
  "Internal",
  "S/PDIF",
  "ADAT",
};

UInt32 getXMOSDeviceID(void)
{
  AudioObjectPropertyAddress propertyAddress = {
    kAudioHardwarePropertyDevices,
    kAudioObjectPropertyScopeGlobal,
    kAudioObjectPropertyElementMain
  };

  UInt32 dataSize = 0;
  OSStatus status = AudioObjectGetPropertyDataSize(kAudioObjectSystemObject, &propertyAddress, 0, NULL, &dataSize);
  if(kAudioHardwareNoError != status) {
    fprintf(stderr, "AudioObjectGetPropertyDataSize (kAudioHardwarePropertyDevices) failed: %i\n", status);
    return kAudioObjectUnknown;
  }
  UInt32 deviceCount = (dataSize / sizeof(AudioDeviceID));
  AudioDeviceID *audioDevices = (AudioDeviceID *) malloc(dataSize);
  if(NULL == audioDevices) {
    fputs("Unable to allocate memory", stderr);
    return kAudioObjectUnknown;
  }

  status = AudioObjectGetPropertyData(kAudioObjectSystemObject, &propertyAddress, 0, NULL, &dataSize, audioDevices);
  if(kAudioHardwareNoError != status) {
    fprintf(stderr, "AudioObjectGetPropertyData (kAudioHardwarePropertyDevices) failed: %i\n", status);
    free(audioDevices), audioDevices = NULL;
    return kAudioObjectUnknown;
  }

  // Iterate through all the devices and determine which are input-capable
  propertyAddress.mScope = kAudioDevicePropertyScopeInput;
  for(UInt32 i = 0; i < deviceCount; ++i) {
    // Query device UID
    CFStringRef deviceUID = NULL;
    dataSize = sizeof(deviceUID);
    propertyAddress.mSelector = kAudioDevicePropertyDeviceUID;
    status = AudioObjectGetPropertyData(audioDevices[i], &propertyAddress, 0, NULL, &dataSize, &deviceUID);
    if(kAudioHardwareNoError != status) {
      fprintf(stderr, "AudioObjectGetPropertyData (kAudioDevicePropertyDeviceUID) failed: %i\n", status);
      continue;
    }

    // Query device manufacturer
    CFStringRef deviceManufacturer = NULL;
    dataSize = sizeof(deviceManufacturer);
    propertyAddress.mSelector = kAudioDevicePropertyDeviceManufacturerCFString;
    status = AudioObjectGetPropertyData(audioDevices[i], &propertyAddress, 0, NULL, &dataSize, &deviceManufacturer);
    if(kAudioHardwareNoError != status) {
      fprintf(stderr, "AudioObjectGetPropertyData (kAudioDevicePropertyDeviceManufacturerCFString) failed: %i\n", status);
      continue;
    }

    if (CFStringFind(deviceManufacturer,
      CFStringCreateWithCString(NULL,
      "XMOS",
      kCFStringEncodingMacRoman),
      0).location == kCFNotFound)
      continue;

    return audioDevices[i];
  }
  printf("Cannot find XMOS device\n");
  exit(1);
}


AudioDeviceID getDefaultOutputDeviceID()
{
  AudioDeviceID outputDeviceID = kAudioObjectUnknown;

  // get output device device
  OSStatus status = noErr;
  AudioObjectPropertyAddress propertyAOPA;
  propertyAOPA.mScope = kAudioObjectPropertyScopeGlobal;
  propertyAOPA.mElement = kAudioObjectPropertyElementMain;
  propertyAOPA.mSelector = kAudioHardwarePropertyDefaultOutputDevice;

  if (!AudioObjectHasProperty(kAudioObjectSystemObject, &propertyAOPA))
    {
      printf("Cannot find default output device!");
      return outputDeviceID;
    }

  status = AudioObjectGetPropertyData(kAudioObjectSystemObject, &propertyAOPA, 0, NULL, (UInt32[]){sizeof(AudioDeviceID)}, &outputDeviceID);

  if (status != 0)
    {
      printf("Cannot find default output device!");
    }
  return outputDeviceID;
}

 float getVolume (AudioDeviceID deviceID, UInt32 scope, UInt32 channel)
{
  AudioObjectPropertyAddress prop = {
    kAudioDevicePropertyVolumeScalar,
    scope == ScopeOutput ? kAudioDevicePropertyScopeOutput : kAudioDevicePropertyScopeInput,
    channel
  };

  if(!AudioObjectHasProperty(deviceID, &prop)) {
      printf("Audio device has no volume property\n");
      exit(1);
  }
  Float32 volume;
  UInt32 dataSize = sizeof(volume);
  OSStatus result = AudioObjectGetPropertyData(deviceID, &prop, 0, NULL, &dataSize, &volume);

  if(kAudioHardwareNoError != result) {
      printf("Error getting volume\n");
      exit(1);
    }

  return volume;
}

 void setVolume (AudioDeviceID deviceID, UInt32 scope,
                 UInt32 channel, Float32 volume)
{
  AudioObjectPropertyAddress prop = {
    kAudioDevicePropertyVolumeScalar,
    scope == ScopeOutput ? kAudioDevicePropertyScopeOutput : kAudioDevicePropertyScopeInput,
    channel
  };

  if(!AudioObjectHasProperty(deviceID, &prop)) {
      printf("Audio device has no volume property\n");
      exit(1);
  }
  UInt32 dataSize = sizeof(volume);
  OSStatus result = AudioObjectSetPropertyData(deviceID, &prop, 0, NULL, dataSize, &volume);

  if(kAudioHardwareNoError != result) {
      printf("Error setting volume\n");
      exit(1);
    }

}

void setClock(AudioDeviceHandle deviceID, uint32_t clockId)
{
  AudioObjectPropertyAddress prop = {
    kAudioDevicePropertyClockSources,
    kAudioDevicePropertyScopeInput,
    kAudioObjectPropertyElementMain
  };

  if (!AudioObjectHasProperty(deviceID, &prop)) {
      printf("Audio device has no clock property\n");
      exit(1);
  }

  UInt32 numClockSources = 0;
  UInt32 dataSize = 0;

  OSStatus result = AudioObjectGetPropertyDataSize(deviceID, &prop, 0, NULL, &dataSize);
  if(kAudioHardwareNoError != result) {
    printf("Error getting number of clocks\n");\
    exit(1);
  }
  numClockSources = dataSize / (sizeof(UInt32));
  UInt32 clockSources[numClockSources];

  if(numClockSources == 0) {
    printf("Error: No clock sources available\n");
    exit(1);
  }

  result = AudioObjectGetPropertyData(deviceID, &prop, 0, NULL, &dataSize, clockSources);
  if(kAudioHardwareNoError != result) {
    printf("Error getting the available clock sources\n");
    exit(1);
  }

  // Get the names of the clock sources
  printf("Found %d clock sources\n", numClockSources);

  for(unsigned i = 0; i < numClockSources; i++) {
    printf("Clock %d = %u; ", i, clockSources[i]);
    prop.mSelector = kAudioDevicePropertyClockSourceNameForIDCFString;
    prop.mScope = kAudioDevicePropertyScopeInput;
    CFStringRef clockName = NULL;
    UInt32 dataSize = sizeof(clockName);

    AudioValueTranslation avt;
    avt.mInputData = &clockSources[i];
    avt.mInputDataSize = sizeof(UInt32);
    avt.mOutputData = &clockName;
    avt.mOutputDataSize = dataSize;
    dataSize = sizeof(avt);

    result = AudioObjectGetPropertyData(deviceID, &prop, 0, NULL, &dataSize, &avt);
    if(kAudioHardwareNoError != result) {
      printf("Error getting clock name, %d\n", result);
      exit(1);
    }
    printf("%s\n", CFStringGetCStringPtr(clockName, kCFStringEncodingMacRoman));

    if (CFStringFind(clockName,
      CFStringCreateWithCString(NULL,
      gClockNameString[clockId - 1],
      kCFStringEncodingMacRoman),
      0).location == kCFNotFound)
    {
      // Not the right clock
      continue;
    }
    // Found the required clock
    // Select the clock

    prop.mSelector = kAudioDevicePropertyClockSource;

    UInt32 currentClockSource = 0;
    unsigned set_retries = 0;
    unsigned set_retry_count = 5;

    while(set_retries++ < set_retry_count)
    {
      result = AudioObjectSetPropertyData(deviceID, &prop, 0, NULL, sizeof(UInt32), &clockSources[i]);
      if(kAudioHardwareNoError != result) {
        printf("Error setting clock to %s. Error %d\n", gClockNameString[clockId - 1], result);
        exit(1);
      }
      else
      {
        sleep(1);

        unsigned check_retries = 0;
        unsigned check_retry_count = 3;
        // Check if the clock stick
        do {
          result = AudioObjectGetPropertyData(deviceID, &prop, 0, NULL, &dataSize, &currentClockSource);
          if(kAudioHardwareNoError != result) {
            currentClockSource = 0; // Set to an invalid value so we remain in the check retry loop
            printf("AudioObjectGetPropertyData() returned error %d\n", result);
          }
          printf("Current clock ID: %u, retry count %d\n", currentClockSource, check_retries);
          sleep(1);
          check_retries += 1;
        }while((currentClockSource != clockSources[i]) && (check_retries < check_retry_count));

        if(currentClockSource == clockSources[i]) {
          printf("Clock source set to %s\n", gClockNameString[clockId - 1]);
          return;
        }
      }
    }
    // Clock didn't stick
    printf("Error Unable to change clock to '%s'\n", gClockNameString[clockId - 1]);
    exit(1);

  }
  // Looks like clock not found
  printf("Clock source '%s' not found\n", gClockNameString[clockId - 1]);
  exit(1);
}

void printSupportedStreamFormats(AudioStreamBasicDescription *formats, unsigned num_formats) {
  std::set<unsigned> samp_freqs;
  for (unsigned i = 0; i < num_formats; ++i)
  {
    samp_freqs.insert(formats[i].mSampleRate);
  }

  std::set<unsigned>::iterator it;
  for(it=samp_freqs.begin(); it != samp_freqs.end(); ++it)
  {
    unsigned sr = *it;
    unsigned found_first = 0;

    for (unsigned i = 0; i < num_formats; ++i) {
      if(formats[i].mSampleRate == sr)
      {
        if(!found_first)
        {
          printf("Sampling rate %6u:\n", sr);
          found_first = 1;
        }
        printf("channels: %2u, bit-depth: %2u\n", formats[i].mChannelsPerFrame, formats[i].mBitsPerChannel);
      }
    }
    if(found_first)
    {
      printf("\n");
    }
  }

}

void showStreamFormats(AudioDeviceHandle deviceID) {
  AudioObjectPropertyAddress propertyAddressInput = {
    kAudioDevicePropertyStreamFormats,
    kAudioObjectPropertyScopeInput,
    kAudioObjectPropertyElementMain
  };
  AudioObjectPropertyAddress propertyAddressOutput = {
    kAudioDevicePropertyStreamFormats,
    kAudioObjectPropertyScopeOutput,
    kAudioObjectPropertyElementMain
  };
  UInt32 size;
  int err = AudioObjectGetPropertyDataSize(deviceID, &propertyAddressInput, 0, NULL, &size);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to get supported input stream formats, error %d\n", err);
    exit(1);
  }

  int num_formats = size / sizeof(AudioStreamBasicDescription);
  auto formats = (AudioStreamBasicDescription *)calloc(num_formats, sizeof(AudioStreamBasicDescription));
  if (!formats) {
    printf("Error: failed to allocate memory for supported input stream formats\n");
    exit(1);
  }

  err = AudioObjectGetPropertyData(deviceID, &propertyAddressInput, 0, NULL, &size, formats);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to populate supported input stream format data, error %d\n", err);
    goto err_free;
  }

  printf("Input stream formats:\n");
  printSupportedStreamFormats(formats, num_formats);

  free(formats);

  err = AudioObjectGetPropertyDataSize(deviceID, &propertyAddressOutput, 0, NULL, &size);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to get supported output stream formats, error %d\n", err);
    exit(1);
  }

  num_formats = size / sizeof(AudioStreamBasicDescription);
  formats = (AudioStreamBasicDescription *)calloc(num_formats, sizeof(AudioStreamBasicDescription));
  if (!formats) {
    printf("Error: failed to allocate memory for supported output stream formats\n");
    exit(1);
  }

  err = AudioObjectGetPropertyData(deviceID, &propertyAddressOutput, 0, NULL, &size, formats);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to populate supported output stream format data, error %d\n", err);
    goto err_free;
  }

  printf("\nOutput stream formats:\n");
  printSupportedStreamFormats(formats, num_formats);

  free(formats);
  return;

err_free:
  free(formats);
  exit(1);
}

void setStreamFormat(AudioDeviceHandle deviceID, uint32_t scope, unsigned sample_rate, unsigned num_chans, unsigned bit_depth) {
  AudioObjectPropertyAddress propertyAddressFormats = {
    kAudioDevicePropertyStreamFormats,
    scope == ScopeInput ? kAudioObjectPropertyScopeInput : kAudioObjectPropertyScopeOutput,
    kAudioObjectPropertyElementMain
  };
  AudioObjectPropertyAddress propertyAddressFormat = {
    kAudioDevicePropertyStreamFormat,
    scope == ScopeInput ? kAudioObjectPropertyScopeInput : kAudioObjectPropertyScopeOutput,
    kAudioObjectPropertyElementMain
  };
  UInt32 size;
  int err = AudioObjectGetPropertyDataSize(deviceID, &propertyAddressFormats, 0, NULL, &size);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to get supported stream formats, error %d\n", err);
    exit(1);
  }

  int num_formats = size / sizeof(AudioStreamBasicDescription);
  auto formats = (AudioStreamBasicDescription *)calloc(num_formats, sizeof(AudioStreamBasicDescription));
  if (!formats) {
    printf("Error: failed to allocate memory for supported stream formats\n");
    exit(1);
  }

  err = AudioObjectGetPropertyData(deviceID, &propertyAddressFormats, 0, NULL, &size, formats);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to populate supported stream format data, error %d\n", err);
    goto err_free;
  }

  int i;
  for (i = 0; i < num_formats; ++i) {
    if (((unsigned)formats[i].mSampleRate == sample_rate) && (formats[i].mChannelsPerFrame == num_chans) && (formats[i].mBitsPerChannel == bit_depth)) {
      break;
    }
  }
  if (i == num_formats) {
    printf("Error: no supported format matching %u channels, %u bit resolution at sample rate %u\n", num_chans, bit_depth, sample_rate);
    printf("Supported %s stream formats:\n", scope == ScopeInput ? "input" : "output");
    printSupportedStreamFormats(formats, num_formats);
    goto err_free;
  }

  err = AudioObjectSetPropertyData(deviceID, &propertyAddressFormat, 0, NULL, sizeof(AudioStreamBasicDescription), &formats[i]);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to set stream format, error %d\n", err);
    goto err_free;
  }

  free(formats);
  return;

err_free:
  free(formats);
  exit(1);
}

void finish(void) {

}

void showCurrentStreamFormat(AudioDeviceHandle deviceID)
{
  AudioObjectPropertyAddress propertyAddressInput = {
    kAudioDevicePropertyStreamFormat,
    kAudioObjectPropertyScopeInput,
    kAudioObjectPropertyElementMain
  };
  AudioObjectPropertyAddress propertyAddressOutput = {
    kAudioDevicePropertyStreamFormat,
    kAudioObjectPropertyScopeOutput,
    kAudioObjectPropertyElementMain
  };
  UInt32 size;

  int err = AudioObjectGetPropertyDataSize(deviceID, &propertyAddressInput, 0, NULL, &size);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to get property size for input stream format, error %d\n", err);
    exit(1);
  }
  if(size != sizeof(AudioStreamBasicDescription))
  {
    printf("Unexpected size (%u) of input stream format property. Expected %lu\n", size, sizeof(AudioStreamBasicDescription));
    exit(1);
  }

  err = AudioObjectGetPropertyDataSize(deviceID, &propertyAddressOutput, 0, NULL, &size);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to get property size for output stream format, error %d\n", err);
    exit(1);
  }
  if(size != sizeof(AudioStreamBasicDescription))
  {
    printf("Unexpected size (%u) of output stream format property. Expected %lu\n", size, sizeof(AudioStreamBasicDescription));
    exit(1);
  }

  AudioStreamBasicDescription in_fmt, out_fmt;

  err = AudioObjectGetPropertyData(deviceID, &propertyAddressInput, 0, NULL, &size, &in_fmt);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to get input stream format, error %d\n", err);
    exit(1);
  }
  err = AudioObjectGetPropertyData(deviceID, &propertyAddressOutput, 0, NULL, &size, &out_fmt);
  if (kAudioHardwareNoError != err) {
    printf("Error: failed to get output stream format, error %d\n", err);
    exit(1);
  }
  if((unsigned)in_fmt.mSampleRate != (unsigned)out_fmt.mSampleRate)
  {
    printf("Error: Sample rate different on input (%6u) and output (%6u) interfaces.\n", (unsigned)in_fmt.mSampleRate, (unsigned)out_fmt.mSampleRate);
    exit(1);
  }
  printf("\nCurrent stream format:\nSampling rate: %6u\n", (unsigned)in_fmt.mSampleRate);
  printf("Input number of channels: %2u\n", (unsigned)in_fmt.mChannelsPerFrame);
  printf("Input bit depth: %2u\n", (unsigned)in_fmt.mBitsPerChannel);
  printf("Output number of channels: %2u\n", (unsigned)out_fmt.mChannelsPerFrame);
  printf("Output bit depth: %2u\n", (unsigned)out_fmt.mBitsPerChannel);
}

void setFullStreamFormat(AudioDeviceHandle deviceID, unsigned sample_rate,
                         unsigned in_num_chans, unsigned in_bit_depth,
                         unsigned out_num_chans, unsigned out_bit_depth)
{
  setStreamFormat(deviceID, ScopeInput, sample_rate, in_num_chans, in_bit_depth);
  setStreamFormat(deviceID, ScopeOutput, sample_rate, out_num_chans, out_bit_depth);
}
