//*****************************************//
//  xmiditest.h
//
//  Re-create xmiditest using RtMidi
//
//*****************************************//

#ifndef XMIDITEST_H
#define XMIDITEST_H

#include <iostream>
#include <cstdlib>
#include <typeinfo>
#include <assert.h>
#include "RtMidi.h"
#include "types64bit.h"

// Platform-dependent sleep routines.
#if defined(__WINDOWS_MM__)
  #include <windows.h>
  #define SLEEP_FN( milliseconds ) Sleep( (DWORD) milliseconds ) 
#else // Unix variants
  #include <unistd.h>
  #define SLEEP_FN( milliseconds ) usleep( (unsigned long) (milliseconds * 1000.0) )
#endif

#define USB_MAX_PKT_LEN 256

#define DEF_TESTS 180 // Default Number of tests
#define DEF_SYS_MSG false // By Default omit System message tests
#define DEF_SEED 1 // Default random number seed

#define NUM_CHANS 16  // Number of Midi channels (instruments)
#define NUM_BUFS (NUM_CHANS + 1)  // Number of Display buffers required (NUM_CHANS + System-message) 

#define MIDI_VAL1_BITS 7 // No. of bits in single-parameter Midi data value (7) 
#define MIDI_VAL2_BITS (MIDI_VAL1_BITS << 1) // No. of bits in double-parameter Midi data value (14)

#define NUM_MIDI_1VALS (1 << MIDI_VAL1_BITS) // No. of possible values for a single Midi data parameter (128)
#define NUM_MIDI_2VALS (1 << MIDI_VAL2_BITS) // No. of possible values for a double Midi data parameter (16384)

#define MSK_7BIT (NUM_MIDI_1VALS - 1) // Mask for 7-bit Midi data value (127)

#define NAP_TIME 500 // Sleep time in milli-seconds 

#define UNDEFINED (-1) // Undefined Midi value
#define UNTESTED (-2) // Un-tested Midi value
#define STR_LEN 32 // String length

#define MANUFACTURER_ID 0x02 // "Big Briar" manufacturer's ID (doesn't really matter whose, I hope).

// Channel-message Status bytes
#define NUM_1PARAM_MSGS 2 // No of 1-Parameter channel messages.
#define NUM_2PARAM_MSGS 5 // No of 2-Parameter channel messages.

#define NOTE_OFF_STS				0x80
#define NOTE_ON_STS				0x90
#define KEY_PRESS_STS			0xA0
#define CONTROL_CHANGE_STS	0xB0
#define PROG_CHANGE_STS		0xC0
#define CHAN_PRESS_STS			0xD0
#define PITCH_WHEEL_STS		0xE0

// System-message Status bytes
#define NUM_SYS_MSGS 8 // Max. possible No of System messages. WARNING. Some are Undefined
#define TSTED_SYS_MSGS 5 // No. of tested System messages.

#define STRT_SYSEX_STS	0xF0
#define MIDI_TIME_STS		0xF1 // Currently untested
#define SONG_PTR_STS		0xF2
#define SONG_SEL_STS		0xF3
#define UNDEF_SM_F4_STS	0xF4 // Undefined
#define UNDEF_SM_F5_STS	0xF5 // Undefined
#define TUNE_REQ_STS		0xF6
#define END_SYSEX_STS		0xF7

// Real-Time message Status bytes
#define NUM_RT_MSGS 8 // Max. possible No of Real-Time messages. WARNING. Some are Undefined
#define TSTED_RT_MSGS 4 // No of tested Real-Time messages.

#define TIME_CLK_STS		0xF8
#define UNDEF_RT_F9_STS	0xF9 // Undefined
#define START_SONG_STS	0xFA
#define CONT_SONG_STS		0xFB
#define STOP_SONG_STS		0xFC
#define UNDEF_RT_FD_STS	0xFD // Undefined
#define ACTIV_SENSE_STS	0xFE // Currently untested
#define RESET_STS				0xFF // Currently untested

#define REAL_TIME_SEQ 0							// Real time
#define TUNE_REQ_SEQ (REAL_TIME_SEQ + TSTED_RT_MSGS) // Tuning tests start after Real-Time Messages 
#define SONG_SEL_SEQ (TUNE_REQ_SEQ + 1)
#define SONG_PTR_SEQ (SONG_SEL_SEQ + NUM_MIDI_1VALS)
#define SYS_EX_SEQ (SONG_PTR_SEQ + NUM_MIDI_2VALS)

// Sequence No. breakpoints for 1-parameter channel messages (128 apart)
#define PROG_CHANGE_SEQ 0
#define CHAN_PRESS_SEQ (PROG_CHANGE_SEQ + NUM_MIDI_1VALS)
#define TWO_PARAM_SEQ (CHAN_PRESS_SEQ + NUM_MIDI_1VALS)

// Sequence No. breakpoints for 2-parameter channel messages (16384 apart)
#define NOTE_ON_SEQ (TWO_PARAM_SEQ)
#define NOTE_OFF_SEQ (NOTE_ON_SEQ + NUM_MIDI_2VALS)
#define KEY_PRESS_SEQ (NOTE_OFF_SEQ + NUM_MIDI_2VALS)
#define PITCH_WHEEL_SEQ (KEY_PRESS_SEQ + NUM_MIDI_2VALS)
#define CONTROL_CHANGE_SEQ (PITCH_WHEEL_SEQ + NUM_MIDI_2VALS)

#define SYSEX_TSTS 6 // Number of tests of System-Exclusive messages (per cycle)

#define RND_I_BITS 16 // Used to generate internal random numbers
#define SHIFT_I_BITS (32 - RND_I_BITS) // Bit-shift used  to get Most-Significant RND_I_BITS bits. NB Assumes 32-bit computation
#define NUM_I_RNDS (1 << RND_I_BITS) // No. of Unique internal random numbers
#define MAX_I_RND (NUM_I_RNDS - 1) // Maximum value internal random numbers (all 1's used as a Mask)

/* All Midi Messages under test */
typedef enum MIDI_MSG_ETAG
{
	SYS_EX_MSG = 0, // System-Exclusive message

	NOTE_OFF_MSG, // Channel-messages
	NOTE_ON_MSG,
	KEY_PRESS_MSG,
	CONTROL_CHANGE_MSG,
	PROG_CHANGE_MSG,
	CHAN_PRESS_MSG,
	PITCH_WHEEL_MSG,

	SONG_PTR_MSG, // Tested System messages
	SONG_SEL_MSG,
	TUNE_REQ_MSG,

	TIME_CLK_MSG, // Tested Real-Time messages
	START_SONG_MSG,
	CONT_SONG_MSG,
	STOP_SONG_MSG,

  NUM_MIDI_MSGS	// Handy Value!-)
} MIDI_MSG_ENUM;

/* All Midi States */
typedef enum MIDI_STATE_ETAG
{
  START = 0, // Waiting for a Midi status byte
  CHAN_MESS, // Processing channel message
  TIME_MESS, // Processing Real-Time message
  SYST_MESS, // Processing system message (NOT system exculsive)
  SYST_EXCL, // Processing system exculsive message
  NUM_MIDI_STATES	// Handy Value!-)
} MIDI_STATE_ENUM;

// Structure containing display data
typedef struct BUF_TAG
{
  std::vector<U8_T> * msg_p; // Pointer to array of bytes to hold message
  R64_T diff_time; // Difference time (since previous time-stamp)
} BUF_TYP;

// Structure containing received midi data
typedef struct MIDI_RX_TAG
{
	BUF_TYP res_bufs[NUM_BUFS]; // results buffers

  RtMidiIn * midi_in_p; // Pointer to RtMidiIn Object

	volatile S32_T read_cnt; // Midi messages read from Input port
} MIDI_RX_TYP;

// Structure containing midi data to transmit
typedef struct MIDI_TX_TAG
{
  std::vector<U8_T> msg_s; // Array of bytes to hold message

  RtMidiOut * midi_out_p; // Pointer to RtMidiOut Object

	U8_T midi_data[USB_MAX_PKT_LEN]; // Array of Midi data bytes
	U8_T midi_status; // Midi Status byte

	S32_T write_cnt; // Midi messages written to Output port
} MIDI_TX_TYP;

// Structure containing MIDI check data
typedef struct CHECK_TAG
{
	S32_T real_time_check[NUM_RT_MSGS]; // Array of real-time check values
	MIDI_STATE_ENUM state; // Current Midi State
	U8_T status; // Midi Status byte
	S32_T data_len; // length of Midi message data (ignoring status byte)
	S32_T byte_id; // Identifier for position of byte in Midi data
	U32_T	param; // check parameter extracted from received data-bytes
	S32_T	shift; // No of bits to shift check parameter before check
	S32_T tst_res; // test result check value
	S32_T fail_cnt; // Counts failed test vectors
	S32_T good_cnt; // Counts successful test vectors
} CHECK_TYP;

// Structure containing MIDI-message configuration data
typedef struct CONFIG_TAG
{
  char name[STR_LEN]; // Text description of Midi-message
	U8_T status; // Midi Status byte
	S32_T len;	// No. of Midi data bytes
	S32_T num_tsts; // No. of times this message is tested (per cycle)
} CONFIG_TYP;

// Structure containing global data
typedef struct GLOBAL_TAG
{
	MIDI_RX_TYP rx_data_s; // Structure of received Midi data
	MIDI_TX_TYP tx_data_s; // Structure of Midi data to transmit

	CHECK_TYP check_s; // Structure of Midi check data

  std::bitset<8> bits; // Array of 8 bits (used for printing binary)
 
	CONFIG_TYP config_data[NUM_MIDI_MSGS]; // Array of structures of Midi-message configuration data
	MIDI_MSG_ENUM * rnd2msg_p; // Pointer to a dynamically allocated look-up table that converts random numbers to Midi-message
	U8_T real_time_data[TSTED_RT_MSGS];		// Array of Real-Time message status bytes to test
	U8_T param1_data[NUM_1PARAM_MSGS]; // Array of 'single parameter' channel-message status bytes
	U8_T param2_data[NUM_2PARAM_MSGS]; // Array of 'double parameter' channel-message status bytes

	bool do_sys_msg; // Flag set if system message tests required
	S32_T test_cnt; // counts number of test vectors executed
	S32_T num_tests; // Number of test requested (per channel)
	S32_T rnd_size; // Size of random-number to Midi-message look-up table (dynamically allocated)
	U32_T seed; // Random number seed
} GLOBAL_TYP;

#endif /* XMIDITEST_H */
