//*****************************************//
//  xmiditest.cpp
//
//  Simple program to test MIDI sysex sending and receiving.
//
//*****************************************//

#include "xmiditest.h"


/*****************************************************************************/
void usage( // Issue usage instructions
	void 
) 
{
  std::cout << "\nUsage: xmiditest [-x] [N]\n";
  std::cout << "\n   N : Number of tests (per channel).   N > 0 \n";
	std::cout << "\n  -x : Include System Message tests (one per test).\n";

  exit( 0 );
} // usage
/*****************************************************************************/
void read_command_line( // Initialise global data with command-line options
	GLOBAL_TYP * glob_p, // Pointer to global data structure
	int argc, // Number of command-line arguments
	char * argv[] // Array of command-line arguments pointers
)
{
	S32_T arg_cnt; // command-line argument counter
	char * arg_ptr; // pointer to command-line argument
	char arg_opt; // argument option
	bool num_found = false; // Clear Numerical argument
	bool bad_args = false; // Clear bad arguments flag


  // Minimal command-line check.
  if ( argc < 2 )
	{
		printf("ERROR: Too few arguments\n" );
		bad_args = true; // Set bad arguments flag
  } // if ( argc < 2 )
	else
	{
	  if ( argc > 3 )
		{
			printf("ERROR: Too many arguments\n" );
			bad_args = true; // Set bad arguments flag
	  } // if ( argc < 2 )
  } // else !( argc < 2 )

	// Initialise configuration data to default values
	for (arg_cnt=1; arg_cnt<argc; arg_cnt++)
	{
		arg_ptr = argv[arg_cnt];

		// Check if command-option
		if ('-' == arg_ptr[0])
		{
			arg_opt = arg_ptr[1]; // Get command option

			switch(arg_opt)
			{
				case 'x':
					glob_p->do_sys_msg = true;
				break;

				default:
					printf("ERROR: Unknown command option: -%c\n" ,arg_opt );
					bad_args = true; // Set bad arguments flag
			} // switch(arg_opt)
		} // if (arg_ptr[0] == '-')
		else
		{ // Assume Test number argument
			// Check inumerical argument already found
			if (num_found)
			{
				printf("ERROR: Too many 'No-of-tests' fields: %s\n" ,arg_ptr );
				bad_args = true; // Set bad arguments flag
			} // if (num_found)
			else
			{
			  glob_p->num_tests = (U32_T)atoi( arg_ptr ); // Assign message length
	
				if (glob_p->num_tests < 1)
				{ 
					printf("ERROR: Too few tests: %d\n" ,glob_p->num_tests );
					bad_args = true; // Set bad arguments flag
				} // if (glob_p->num_tests < 1)

				num_found = true; // Set flag indication Numerical field found
			} // if (num_found)
		} // else !(arg_ptr[0] == '-')
	} // for arg_cnt

	// Check for bad command-line arguments
	if (bad_args)
	{
		usage();
	} // if (bad_args)
	else
	{
		printf("No Tests to run = %d\n" ,glob_p->num_tests );
		std::cout << "Run System Message Tests = " << std::boolalpha << glob_p->do_sys_msg << std::endl;
	} // else !(bad_args)

} // read_command_line_options
/*****************************************************************************/
U32_T internal_rand( // Generates number in the range 0..MAX_I_RND
	GLOBAL_TYP * glob_p // Pointer to global data structure
)
{
	glob_p->seed = (glob_p->seed * (U32_T)1103515245) + (U32_T)12345;

	// Return number in the range 0..MAX_I_RND; 
	return((U32_T)(glob_p->seed >> SHIFT_I_BITS) & MAX_I_RND); 
} // rnd15 
/*****************************************************************************/
U32_T rand_range( // Generates random number in the range 0..max_rand 
	GLOBAL_TYP * glob_p, // Pointer to global data structure
	U32_T num_rands // No. of different possible random numbers to return
)
{
	U32_T rand_val = internal_rand( glob_p );


	rand_val *= num_rands;
	rand_val >>= RND_I_BITS;

	return rand_val;
} // rand_bits 
/*****************************************************************************/
void init_receive_data( // Initialise structure for received midi data
	GLOBAL_TYP * glob_p, // Pointer to global data structure
	MIDI_RX_TYP * midi_rx_p // Pointer to structure of received Midi data 
)
{
  midi_rx_p->midi_in_p = 0; // Clear pointer to RtMidi-In Object
  midi_rx_p->read_cnt = 0;	// Clear Midi read counter
} // init_receive_data
/*****************************************************************************/
void init_transmit_data( // Initialise structure for transmitted midi data
	GLOBAL_TYP * glob_p, // Pointer to global data structure
	MIDI_TX_TYP * midi_tx_p // Pointer to structure of Midi data to transmit 
)
{
  midi_tx_p->midi_out_p = 0; // Clear pointer to RtMidi-Out Object
  midi_tx_p->write_cnt = 0;	// Clear Midi write counter
} // init_transmit_data
/*****************************************************************************/
void init_check_data( // Initialise structure for midi check data
	GLOBAL_TYP * glob_p, // Pointer to global data structure
	CHECK_TYP * check_p // Pointer to structure of Midi check data
)
{
  check_p->state = START; // Initialise Midi state
	check_p->status = 0; // Midi Status byte
	check_p->data_len = 0; // length of Midi message data (ignoring status byte)
	check_p->byte_id= 0; // identifier for position of byte in midi-message
	check_p->param = 0; // check parameter based on value of received data-bytes
	check_p->shift = 0; // No of bits to shift check parameter before check
	check_p->tst_res = 0; // test result check value
	check_p->fail_cnt = 0; // Clear count of failed test vectors
	check_p->good_cnt = 0; // Clear count of successful test vectors


  check_p->real_time_check[0] = 0; // Timing-Clock
  check_p->real_time_check[1] = UNDEFINED;
  check_p->real_time_check[2] = 1; // Start-Song
  check_p->real_time_check[3] = 2; // Continue-Song
  check_p->real_time_check[4] = 3; // Stop-Song
  check_p->real_time_check[5] = UNDEFINED;
  check_p->real_time_check[6] = UNTESTED; // Active-Sensing: Currently un-tested
  check_p->real_time_check[7] = UNTESTED; // Reset: Currently un-tested
} // init_check_data
/*****************************************************************************/
void configure_one_midi_message( // Configure array of all Midi message data
	GLOBAL_TYP * glob_p, // Pointer to global data structure
	const char * str_p, // Pointer to input string
	MIDI_MSG_ENUM msg_id, // Midi-message identifier
	U8_T inp_status, // input status value
	S32_T num_bytes // Number of occurances of input status value
)
{
	CONFIG_TYP config_data_s; // Structure of status data


	// Check string length
	if (STR_LEN <= strlen( str_p ))
	{
		printf("ERROR: Message name '%s' too large, Reduce name, or increase STR_LEN\n" ,str_p );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // else !(0 < glob_p->sysex_num)

	strcpy( config_data_s.name ,str_p );

	config_data_s.status = inp_status; // Assign status value

	// Check for special-case of System-Exclusive
	if (SYS_EX_MSG == msg_id)
	{ // System-Exclusive
		config_data_s.len = UNDEFINED; // NB System-Exclusive messages are variable-length
		config_data_s.num_tsts = SYSEX_TSTS;
	} // if (SYS_EX_MSG == msg_id)
	else
	{
		config_data_s.len = num_bytes; // Assign number of data bytes
		config_data_s.num_tsts = (1 << num_bytes); // Calculate No. of occurances
	} // else !(SYS_EX_MSG == msg_id)

	glob_p->rnd_size += config_data_s.num_tsts; // Update total number of randoms tests required

	glob_p->config_data[msg_id] = config_data_s;
} // configure_one_midi_message
/*****************************************************************************/
void configure_all_midi_message_data( // Configure array of all Midi message data
	GLOBAL_TYP * glob_p // Pointer to global data structure
)
{
	// Configure System-Exclusive data (variable length)
	configure_one_midi_message( glob_p ,"System-Exclusive" ,SYS_EX_MSG ,UNDEFINED ,UNDEFINED );

	// Configure Channel-message data
	configure_one_midi_message( glob_p ,"Note-Off" ,NOTE_OFF_MSG ,NOTE_OFF_STS ,2 ); // 
	configure_one_midi_message( glob_p ,"Note-On" ,NOTE_ON_MSG ,NOTE_ON_STS ,2 );
	configure_one_midi_message( glob_p ,"Key-Pressure" ,KEY_PRESS_MSG ,KEY_PRESS_STS ,2 );
	configure_one_midi_message( glob_p ,"Control-Change" ,CONTROL_CHANGE_MSG ,CONTROL_CHANGE_STS ,2 );
	configure_one_midi_message( glob_p ,"Program-Change" ,PROG_CHANGE_MSG ,PROG_CHANGE_STS ,1 );
	configure_one_midi_message( glob_p ,"Channel-Pressure" ,CHAN_PRESS_MSG ,CHAN_PRESS_STS ,1 );
	configure_one_midi_message( glob_p ,"Pitch-Wheel" ,PITCH_WHEEL_MSG ,PITCH_WHEEL_STS ,2 );

	// Configure System-message data
	configure_one_midi_message( glob_p ,"Song-Position" ,SONG_PTR_MSG ,SONG_PTR_STS ,2 );
	configure_one_midi_message( glob_p ,"Song-Select" ,SONG_SEL_MSG ,SONG_SEL_STS ,1 );
	configure_one_midi_message( glob_p ,"Tune_Request" ,TUNE_REQ_MSG ,TUNE_REQ_STS ,0 );

	// Configure Real-Time message data
	configure_one_midi_message( glob_p ,"Timing-Clock" ,TIME_CLK_MSG ,TIME_CLK_STS ,0 );
	configure_one_midi_message( glob_p ,"Start-Song" ,START_SONG_MSG ,START_SONG_STS ,0 );
	configure_one_midi_message( glob_p ,"Continue-Song" ,CONT_SONG_MSG ,CONT_SONG_STS ,0 );
	configure_one_midi_message( glob_p ,"Stop-Song" ,STOP_SONG_MSG ,STOP_SONG_STS ,0 );

} // configure_all_midi_message_data
/*****************************************************************************/
void init_all_message_occurances(  // Initialise number of occurances for all Midi-messages
	GLOBAL_TYP * glob_p // Pointer to global data structure
)
/* 
 * The proportion of each Midi-message type is managed by populating glob_p->rnd2msg[]
 * The occurance frequency of a message type is dependent on the number of data-parameters
 * of that message
 * All occurances must sum to NUM_RND_STS_VALS
 */
{
	S32_T msg_cnt; // message-type counter
	S32_T tst_cnt; // test counter
	S32_T chk_size = 0; // Used to check number of random tests
	S32_T mem_size = 0; // Size of memory in bytes


	// Allocate look-up table
	mem_size = glob_p->rnd_size * sizeof(MIDI_MSG_ENUM);
	glob_p->rnd2msg_p = (MIDI_MSG_ENUM *)malloc(mem_size); 

	if (NULL == glob_p->rnd2msg_p)
	{
		printf("ERROR: Failed to malloc %d Bytes\n" ,mem_size );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (NULL == glob_p->rnd2msg_p)

	// Loop through all Midi-message types
	for(msg_cnt=0; msg_cnt<NUM_MIDI_MSGS; msg_cnt++)
	{ 
		// Loop through all tests for current Midi-message type
		for (tst_cnt=0; tst_cnt<glob_p->config_data[msg_cnt].num_tsts; tst_cnt++)
		{
			glob_p->rnd2msg_p[chk_size] = (MIDI_MSG_ENUM)msg_cnt; // Assign message-type to this test
			chk_size++; // Increment occurance sum
		} // for tst_cnt
	} // for msg_cnt

	// Check Test sum
	if (chk_size != glob_p->rnd_size)
	{
		printf("ERROR: Test-sum mis-match: Expected %d, Counted %d\n" ,glob_p->rnd_size ,chk_size );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (NUM_RND_STS_VALS != glob_p->rnd_size)

} // init_all_message_occurances
/*****************************************************************************/
void init_globals( // Initialise global data
	GLOBAL_TYP * glob_p, // Pointer to global data structure
	int argc, // Number of command-line arguments
	char * argv[] // Array of command-line arguments pointers
)
{
	// Initialise configuration data to default values
	glob_p->do_sys_msg = DEF_SYS_MSG;
	glob_p->test_cnt = 0;
	glob_p->num_tests = DEF_TESTS;
	glob_p->rnd_size = 0; // Size of random-number to Midi-message look-up table (dynamically allocated)
	glob_p->seed = DEF_SEED; // Default Random Number seed

	// Update configuration data with command-line options
	read_command_line( glob_p ,argc ,argv );

	// Initialise Non-configurable data ...

	// Real-Time messages. WARNING These are NOT is obvious slots
	glob_p->real_time_data[0] = TIME_CLK_STS;
	glob_p->real_time_data[1] = START_SONG_STS;
	glob_p->real_time_data[2] = CONT_SONG_STS;
	glob_p->real_time_data[3] = STOP_SONG_STS;

	// Single Parameter Channel messages
	glob_p->param1_data[0] = PROG_CHANGE_STS;
	glob_p->param1_data[1] = CHAN_PRESS_STS;

	// Double Parameter Channel messages
	glob_p->param2_data[0] = NOTE_ON_STS;
	glob_p->param2_data[1] = NOTE_OFF_STS;
	glob_p->param2_data[2] = KEY_PRESS_STS;
	glob_p->param2_data[3] = PITCH_WHEEL_STS;
	glob_p->param2_data[4] = CONTROL_CHANGE_STS;

	glob_p->rnd_size = 0; // Clear number of random tests. NB Incremented in configure_one_midi_message()
	configure_all_midi_message_data( glob_p );	// Configure array of all Midi message data

	// Initialise number of occurances for all Midi-messages
	init_all_message_occurances( glob_p );

	// Initialise data structure for transmitted/received Midi data
  init_receive_data( glob_p ,&(glob_p->rx_data_s) );
  init_transmit_data( glob_p ,&(glob_p->tx_data_s) );

	// Initialise Midi check data structure
  init_check_data( glob_p ,&(glob_p->check_s) );
} // init_globals
/*****************************************************************************/
void check_midi_message( // Calculate check result for Real-Time message
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p  // Pointer to structure of Midi check data 
)
{
	check_p->tst_res += (check_p->param << check_p->shift); // Build final test result value

	// Check if test passed
	if (check_p->tst_res == glob_p->test_cnt)
	{ // Success
		check_p->good_cnt++; 	
	} // if (check_p->tst_res == glob_p->test_cnt)
	else
	{ // Failed
		check_p->fail_cnt++; 	

printf("State=%d Status =0x%x Len=%d Byte_id=%d ChkVal=0x%x Test_id=0x%x  Test_res=0x%x \n"  
  ,check_p->state 
	,check_p->status
	,check_p->data_len
	,check_p->byte_id
	,check_p->param
	,glob_p->test_cnt
	,check_p->tst_res
); // MB~

	} // if (check_p->tst_res == glob_p->test_cnt)

} // check_midi_message
/*****************************************************************************/
void clear_check_data( // Clears Midi check data at start of new message 
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p  // Pointer to structure of Midi check data 
) // Returns size of channel message [0, 1 or 2]
{
	check_p->tst_res= 0; // Clear test result
	check_p->param = 0; // Clear check parameter
	check_p->shift = 0; // Clear check shift
	check_p->byte_id = 0; // Reset position of 1st byte in midi-message
} // clear_check_data
/*****************************************************************************/
MIDI_STATE_ENUM start_channel_message( // From status, evaluate No. of channel-message bytes (excluding status)
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T status // Received status byte
) // Returns new Midi-state. NB Currently always CHAN_MESS
{
  U8_T msg_typ = (status & 0xF0); // NB 4 MS bits are channel-message type


	clear_check_data( glob_p ,check_p );

	// Determine channel-message type
	switch(msg_typ)
	{
		case NOTE_OFF_STS: // Note-Off
			check_p->tst_res = NOTE_OFF_SEQ; // Initialise Note-OFF test-result value
			check_p->data_len = 2;
		break; // NOTE_OFF_STS

		case NOTE_ON_STS: // Note-On
			check_p->tst_res = NOTE_ON_SEQ; // Initialise Note-On test-result value
			check_p->data_len = 2;
		break; // NOTE_ON_STS

		case KEY_PRESS_STS: // Poly Key pressure
			check_p->tst_res = KEY_PRESS_SEQ; // Initialise Poly-Key-Pressure test-result value
			check_p->data_len = 2;
		break; // KEY_PRESS_STS

		case CONTROL_CHANGE_STS:	// Control Change
			check_p->tst_res = CONTROL_CHANGE_SEQ; // Initialise Control-Change test-result value
			check_p->data_len = 2;
		break; // CONTROL_CHANGE_STS

		case PROG_CHANGE_STS: // Program Change
			check_p->tst_res = PROG_CHANGE_SEQ; // Initialise Program-Change test-result value
			check_p->data_len = 1;
		break; // PROG_CHANGE_STS

		case CHAN_PRESS_STS:	// Channel pressure
			check_p->tst_res = CHAN_PRESS_SEQ; // Initialise Channel-Pressure test-result value
			check_p->data_len = 1;
		break; // CHAN_PRESS_STS

		case PITCH_WHEEL_STS: // Key Pitch Bend
			check_p->tst_res = PITCH_WHEEL_SEQ; // Initialise Pitch-Wheel test-result value
			check_p->data_len = 2;
		break; // PITCH_WHEEL_STS

		default:
			printf("ERROR: Unsupported  Midi channel-message %d\n" ,msg_typ );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(msg_typ)

	return CHAN_MESS;
} // start_channel_message
/*****************************************************************************/
MIDI_STATE_ENUM start_system_message( // From status, determines No. of system-message bytes (excluding status)
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T status // Received status byte
) // Returns new Midi-state
{
	clear_check_data( glob_p ,check_p );

	// Determine system-message type
	switch(status)
	{
		case MIDI_TIME_STS: // Midi-Time Code
			printf("ERROR: Midi-Time-Code currently unsupported %d\n" ,status );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);

			check_p->data_len = 1;
			return SYST_MESS;
		break; // MIDI_TIME_STS

		case SONG_PTR_STS: // Song Pointer
			check_p->tst_res = SONG_PTR_SEQ; // Initialise test-result value
			check_p->shift = 1; // Special-case shift for Song-Pointer

			check_p->data_len = 2;
			return SYST_MESS;
		break; // SONG_PTR_STS

		case SONG_SEL_STS:	// Song Select
			check_p->tst_res = SONG_SEL_SEQ; // Initialise test-result value

			check_p->data_len = 1;
			return SYST_MESS;
		break; // SONG_SEL_STS

		case TUNE_REQ_STS: // Tune Request
			check_p->tst_res = TUNE_REQ_SEQ; // Initialise test-result value

			check_p->data_len = 0; // No data bytes expected
			return START;
		break; // TUNE_REQ_STS

		default:
			printf("ERROR: Unsupported  Midi system-message %d\n" ,status );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(status)
} // start_system_message
/*****************************************************************************/
void parse_real_time_message( // Calculate check result for Real-Time message
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T status // Received status byte
)
{
	// NB There are only 8 possible RTM', therefore check result is only dependent on LS 3 bits
	check_p->tst_res = REAL_TIME_SEQ + check_p->real_time_check[status & 0x07];
} // parse_real_time_message
/*****************************************************************************/
MIDI_STATE_ENUM parse_message_data_bytes( // Expecting Midi-message data-byte
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T data_byt // Received input data byte
) // Returns Midi-state
{
	check_p->param <<= MIDI_VAL1_BITS;
	check_p->param += data_byt;

	check_p->byte_id++; // Increment byte identifier

	// Check for end of message
	if (check_p->byte_id == check_p->data_len)
	{ // End-of Midi message
		check_midi_message( glob_p ,check_p );

		check_p->state = START; // Switch to Midi start state
	} // if (check_p->byte_id == check_p->data_len)

	return check_p->state; // Returns (possible new) Midi-state
} // parse_message_data_bytes
/*****************************************************************************/
void parse_sysex_data( // Expecting system-exclusive data-byte
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T data_byt // Received input data byte
)
{
	// Check for Manufacturers ID
	if (0 < check_p->byte_id)
	{ // NOT Manufacturers ID
		check_p->param <<= MIDI_VAL1_BITS; // Shift left to make room for MIDI_VAL1_BITS LS bits
		check_p->param += data_byt; // Add-in MIDI_VAL1_BITS bits at LS end of param
	} // if (0 < check_p->byte_id)
	else
	{ // Manufacturers ID
		if (MANUFACTURER_ID != data_byt)
		{
			printf("ERROR: Expected Manufacturer's Id of %d,  Found %d\n" ,MANUFACTURER_ID ,data_byt );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		} // if (MANUFACTURER_ID != data_byt)
	} // if (0 < check_p->byte_id)

	check_p->byte_id++; // Increment byte identifier

} // parse_sysex_data
/*****************************************************************************/
MIDI_STATE_ENUM parse_system_exclusive_byte( // Expecting system-exclusive byte 
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T inp_byt // Received input byte
) // Returns Midi-state
{
	/*
	 * To Determine the type of System-Exclusive byte, the largest byte ID is tested 1st, 
   * and then in descending numerical order. This enables early returns to be used.
	 */

	// Check for Real-Time message
	if (END_SYSEX_STS < inp_byt)
	{
		parse_real_time_message( glob_p ,check_p ,inp_byt );

		return check_p->state; // Returns current Midi-state
	} // if (END_SYSEX_STS < inp_byt)

	// Check for End of System-Exclusive message
	if (END_SYSEX_STS == inp_byt) 
	{
		// Check if any SysEx data to check
		if (1 < check_p->byte_id)
		{ // SysEx data-bytes to check
			check_midi_message( glob_p ,check_p );
		} // if (1 < check_p->byte_id)

		return START; // Return next Midi-state
	} // if (END_SYSEX_STS == inp_byt) 

	// Check for System-Exclusive data
	if (inp_byt < NOTE_OFF_STS)
	{ // system exclusive data
		parse_sysex_data( glob_p ,check_p ,inp_byt );

		return check_p->state; // Returns current Midi-state
	} // if (inp_byt < NOTE_OFF_STS)

  // If we get this far, System-Exclusive byte has an Illegal value
	printf("ERROR: Expected System Exclusive Byte Found %d\n" ,inp_byt );
	printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
	exit(-1);

} // parse_system_exclusive_byte
/*****************************************************************************/
MIDI_STATE_ENUM parse_message_status_byte( // Inspects Midi status byte and assigns next Midi-state
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p, // Get pointer to structure of Midi check data 
  U8_T sts_byt // Received status byte
) // Returns next Midi-state
{
	MIDI_STATE_ENUM new_state; // new Midi-state


	check_p->status = sts_byt; // Update Midi-status

	/*
	 * To Determine the type of Midi-message, the largest status ID is tested 1st, 
   * and then in descending numerical order. This enables early returns to be used.
	 */

	// Check for Real-Time message
	if (END_SYSEX_STS < sts_byt)
	{ 
		parse_real_time_message( glob_p ,check_p ,sts_byt );
		return START;
	} // if (END_SYSEX_STS < sts_byt)

	// Check for System message
	if (STRT_SYSEX_STS < sts_byt) 
	{
		new_state = start_system_message( glob_p ,check_p ,sts_byt );
		return new_state;
	} // if (STRT_SYSEX_STS < sts_byt) 

	// Check for System-Exclusive message
	if (STRT_SYSEX_STS == sts_byt)  
	{
		clear_check_data( glob_p ,check_p );
		return SYST_EXCL;
	} // if (STRT_SYSEX_STS == sts_byt)  

	// Check for channel-message
	if (MSK_7BIT < sts_byt)
	{
		new_state = start_channel_message( glob_p ,check_p ,sts_byt );
		return new_state;
	} // if (MSK_7BIT < sts_byt)

  // If we get this far, status-byte has an Illegal value
	printf("ERROR: Expected Midi Status-Byte. Found %d\n" ,sts_byt );
	printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
	exit(-1);
} // parse_message_status_byte
/*****************************************************************************/
void update_midi_state( // Update Midi state based on value of input byte
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	CHECK_TYP * check_p, // Get pointer to structure of Midi check data 
  U8_T inp_byt // Received input byte
)
{
	// Determine current Midi state
	switch(check_p->state)
	{
		case START: // Start a new message
			check_p->state = parse_message_status_byte( glob_p ,check_p ,inp_byt ); // Get new Midi-state
		break; // START

		case CHAN_MESS: // Continue parsing channel-message
			check_p->state = parse_message_data_bytes( glob_p ,check_p ,inp_byt );
		break; // CHAN_MESS 

		case SYST_MESS: // Continue parsing System-message
			check_p->state = parse_message_data_bytes( glob_p ,check_p ,inp_byt );
		break; // SYST_MESS 

		case SYST_EXCL: // Continue parsing System-exclusive message
			check_p->state = parse_system_exclusive_byte( glob_p ,check_p ,inp_byt );
		break; // SYST_EXCL 

		default:
			printf("ERROR: Unsupported  Midi State %d\n" ,check_p->state );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(check_p->state)

} // update_midi_state
/*****************************************************************************/
void mycallback( // Callback function, executed when Midi-In port receives data
	R64_T deltatime, // difference between time stamps
	std::vector< U8_T > * msg_p, // Pointer to vector containing received data
	void * void_ptr // userData
)
{
	GLOBAL_TYP * glob_p = (GLOBAL_TYP *)void_ptr; // Retrieve pointer to global data structure
	MIDI_RX_TYP * midi_rx_p = &(glob_p->rx_data_s); // Get pointer to structure of received Midi data 
	CHECK_TYP * check_p = &(glob_p->check_s); // Get pointer to structure of Midi check data 
	BUF_TYP * buf_p = &(midi_rx_p->res_bufs[midi_rx_p->read_cnt]); // Get pointer to next results buffer
  U32_T nBytes = msg_p->size(); ; // Get number of message bytes received
  U8_T inp_byt; // Received input byte


	// Check received data message ...

	// Check for Empty message
  if ( nBytes > 0 )
	{ // message NOT empty

		// Loop through received bytes
	  for ( U32_T byte_cnt=0; byte_cnt<nBytes; byte_cnt++ )
		{
			inp_byt = msg_p->at(byte_cnt); // Get next input byte

			update_midi_state( glob_p ,check_p ,inp_byt ); // Check one byte
	  } // for byte_cnt
  } // if ( nBytes > 0 )
	else
	{
    std::cout << "ERROR: Empty message.";
  } // if ( nBytes > 0 )

	// Copy Callback data to buffer, so we can display later at end of test vector
	buf_p->diff_time = deltatime; 
	buf_p->msg_p = new std::vector<U8_T>( (const std::vector<U8_T>)*msg_p ); // Create message in buffer
	midi_rx_p->read_cnt++; // Increment received messages
} // mycallback
/*****************************************************************************/
void display_results( // Print all result in buffer
	GLOBAL_TYP * glob_p  // Pointer to global data structure
)
{
	MIDI_RX_TYP * midi_rx_p = &(glob_p->rx_data_s); // Get pointer to structure of received Midi data 
	BUF_TYP * buf_p = &(midi_rx_p->res_bufs[midi_rx_p->read_cnt]); // Get pointer to next results buffer
	S32_T buf_cnt; // Buffer counter
  U32_T nBytes; // Get number of message bytes received


	for(buf_cnt=0; buf_cnt<midi_rx_p->read_cnt; buf_cnt++)
	{
		buf_p = &(midi_rx_p->res_bufs[buf_cnt]); // Get pointer to next results buffer
		printf("                                    Diff_Time=%7.5f:" ,buf_p->diff_time );

		nBytes = buf_p->msg_p->size(); // Get number of bytes received

	  if ( nBytes > 0 )
		{
			// Loop through received bytes
		  for ( U32_T byte_cnt=0; byte_cnt<nBytes; byte_cnt++ )
			{
				glob_p->bits = (int)buf_p->msg_p->at(byte_cnt); // Convert to binary digits
		    std::cout << " " << glob_p->bits;
		  } // for byte_cnt
	  } // if ( nBytes > 0 )
		else
		{
	    std::cout << "ERROR: Empty message.";
	  } // if ( nBytes > 0 )
	
	  std::cout << std::endl;

		delete buf_p->msg_p; // Delete message buffer
	} // for buf_cnt

} // display_results
/*****************************************************************************/
void synchronise( // Wait for previous test results to be received before starting display
	GLOBAL_TYP * glob_p  // Pointer to global data structure
)
{
	MIDI_RX_TYP * midi_rx_p = &(glob_p->rx_data_s); // Get pointer to structure of received Midi data 
	MIDI_TX_TYP * midi_tx_p = &(glob_p->tx_data_s); // Get pointer to structure of Midi data to transmit


	// Wait for all written messages to be received
	while ( midi_rx_p->read_cnt < midi_tx_p->write_cnt );

//	check_results( glob_p ); // check results for previous test vector

	display_results( glob_p ); // display results for previous test vector

	// Clear Midi read/write counters
  midi_rx_p->read_cnt = 0;
  midi_tx_p->write_cnt = 0;

} // synchronise
/*****************************************************************************/
bool open_midi_port( // Opens a Midi port
	RtMidi * rtmidi_p, // Pointer to RtMidi object
	const char * port_str // String holding port direction (Input or Output)  
) // Returns true if exactly one port available.
{
	U32_T nPorts = rtmidi_p->getPortCount(); // Get number of ports
 	U32_T port_cnt = 0; // port counter


	// Switch on number of ports
	switch (nPorts)
	{
		case 0 : // No port available
	    std::cout << "ERROR: No " << port_str << " ports available!" << std::endl;
  	  return false;
		break; // case 0

		case 1 :  // Open port
	    std::cout << "Opening " << port_str << " port "<< rtmidi_p->getPortName() << std::endl;
		  rtmidi_p->openPort(0);
		break; // case 1

		default : // More than one port available
		  std::string portName;
	
	    std::cout << "ERROR: More than one " << port_str << " port available:-" << std::endl;

	    for ( port_cnt=0; port_cnt<nPorts; port_cnt++ ) 
			{
	      portName = rtmidi_p->getPortName(port_cnt);
	      std::cout << "     Port_" << port_cnt << ": " << portName << '\n';
	    } // for port_cnt

  	  return false;
		break; // default
	} // switch (nPorts)

  return true;
} // open_midi_port 
/*****************************************************************************/
bool configure_midi_ports( // Configure which Midi ports to use
	GLOBAL_TYP * glob_p  // Pointer to global data structure
) // Returns true if configuration successful
{
  // RtMidiOut & RtMidiIn constructors
  try 
	{
    glob_p->tx_data_s.midi_out_p = new RtMidiOut();
    glob_p->rx_data_s.midi_in_p = new RtMidiIn();
  } // try
  catch ( RtMidiError &error ) 
	{
		printf("\nERROR: Failed to successfully construct RtMidi Objects\n");
    error.printMessage();
    return false;
  } // catch

  // Don't ignore sysex, timing, or active sensing messages.
  glob_p->rx_data_s.midi_in_p->ignoreTypes( false, true, true );

  try 
	{
    if ( open_midi_port( glob_p->tx_data_s.midi_out_p ,"Output" ) == false ) return false;
    if ( open_midi_port( glob_p->rx_data_s.midi_in_p  ,"Input" ) == false )  return false;
  } // try
  catch ( RtMidiError &error ) 
	{
		printf("\nERROR: Failed to successfully open Midi Ports\n");
    error.printMessage();
    return false;
  } // catch

  glob_p->rx_data_s.midi_in_p->setCallback( &mycallback ,(void *)glob_p ); // NB Pass in pointer to global data structure

	std::cout << std::endl;
	return true;
} // 	configure_midi_ports
/*****************************************************************************/
bool send_message( // Wrapper for RtMidi sendmessge()
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	MIDI_TX_TYP * midi_tx_p // Pointer to structure of Midi data to transmit 
)
{
  std::vector<U8_T> * msg_p = &(midi_tx_p->msg_s); // Local pointer to Array of bytes to hold message


  try 
	{
		midi_tx_p->midi_out_p->sendMessage( msg_p ); // Call RtMidi function
  } // try
  catch ( RtMidiError &error ) 
	{
		printf("\nERROR: Failed to successfully send RtMidi message\n");
    error.printMessage();
    return false;
  } // catch

  midi_tx_p->write_cnt++; // Increment message write counter

  return true;
} // send_message
/*****************************************************************************/
bool packet_to_message( // Convert Midi packet to RtMidi message
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
	MIDI_TX_TYP * midi_tx_p, // Pointer to structure of Midi data to transmit 
	U8_T num_bytes // Number of Midi data bytes
) // Return status flag
{
  std::vector<U8_T> * msg_p = &(midi_tx_p->msg_s); // Local pointer to Array of bytes to hold message
	S32_T byte_cnt; // Midi data byte counter

	glob_p->bits = midi_tx_p->midi_status;
	printf("%1d: " ,num_bytes ); //MB~
	std::cout << glob_p->bits;

	msg_p->clear();
  msg_p->push_back( midi_tx_p->midi_status );

	for (byte_cnt=0; byte_cnt<num_bytes; byte_cnt++)
	{ 	
		glob_p->bits = midi_tx_p->midi_data[byte_cnt];
		std::cout << " " << glob_p->bits;
	  msg_p->push_back( midi_tx_p->midi_data[byte_cnt] );
	} // for byte_cnt

	printf("\n"); //MB~

	if (false == send_message( glob_p ,midi_tx_p ) ) return false; // Send message id 

	return true;
} // packet_to_message
/*****************************************************************************/
bool build_midi_packet(
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
//	const MIDITimeStamp	&timeStamp, //MB~ ToDo
  U8_T status, // Transmitted status byte
	U8_T num_bytes, // Number of Midi data bytes
	U32_T chk_val // Check value for test sequence
) // Return status flag
{
	MIDI_TX_TYP * midi_tx_p = &(glob_p->tx_data_s); // Get pointer to structure of Midi data to transmit
	S32_T byte_cnt; // Midi data byte counter
	U8_T ls_off = (num_bytes - 1); // Midi-message offset for LS byte of check-value


	midi_tx_p->midi_status = status;

	// NB tests configured to send MS byte of check value first
	for (byte_cnt=0; byte_cnt<num_bytes; byte_cnt++)
	{
		midi_tx_p->midi_data[ls_off - byte_cnt] = (chk_val & MSK_7BIT); // Store LS 7-bits
		chk_val >>= MIDI_VAL1_BITS; // Get next 7-bits
	} // for byte_cnt

	if (false == packet_to_message( glob_p ,midi_tx_p ,num_bytes )) return false;

	return true;
} // build_midi_packet
/*****************************************************************************/
bool build_chan_1param(
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
//	const MIDITimeStamp	&timeStamp, //MB~ ToDo
	U32_T chk_val // Check value for test sequence
) // Return status flag
{
  U8_T status; // Transmitted status byte
	S32_T chan_cnt; // Channel counter


	printf(" 1-Parameter\n" );

	// Build one message for each channel.
	for(chan_cnt = 0; chan_cnt < NUM_CHANS; chan_cnt++)
	{
		status = (glob_p->param1_data[chk_val >> MIDI_VAL1_BITS] + chan_cnt);
		if (false == build_midi_packet( glob_p ,status ,1 ,chk_val )) return false; // 1 data parameter
	} // for chan_cnt 

	return true;
} // build_chan_1param
/*****************************************************************************/
bool build_chan_2param(
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
//	const MIDITimeStamp	&timeStamp, //MB~ ToDo
	U32_T chk_val // Check value for test sequence
) // Return status flag
{
  U8_T status; // Transmitted status byte
	S32_T chan_cnt; // Channel counter


	printf(" 2-Parameter\n" );

	// Build one message for each channel.
	for(chan_cnt = 0; chan_cnt < NUM_CHANS; chan_cnt++)
	{
		status = (glob_p->param2_data[chk_val >> MIDI_VAL2_BITS] + chan_cnt);
		if (false == build_midi_packet( glob_p ,status ,2 ,chk_val )) return false; // 2 data parameters
	} // for chan_cnt 

	return true;
} // build_chan_2param
/*****************************************************************************/
bool build_systemexclusive(
	GLOBAL_TYP * glob_p,  // Pointer to global data structure
//	const MIDITimeStamp	&timeStamp, //MB~ ToDo
	U32_T chk_val // Check value for test sequence
) // Return status flag
{
	MIDI_TX_TYP * midi_tx_p = &(glob_p->tx_data_s); // Get pointer to structure of Midi data to transmit


	printf(" System-Exclusive\n" );

	midi_tx_p->midi_status = STRT_SYSEX_STS;
	midi_tx_p->midi_data[0] = MANUFACTURER_ID;
	midi_tx_p->midi_data[1] = ((chk_val >> (4 * MIDI_VAL1_BITS)) & MSK_7BIT);
	midi_tx_p->midi_data[2] = ((chk_val >> (3 * MIDI_VAL1_BITS)) & MSK_7BIT);
	midi_tx_p->midi_data[3] = ((chk_val >> (2 * MIDI_VAL1_BITS)) & MSK_7BIT);
	midi_tx_p->midi_data[4] = ((chk_val >> MIDI_VAL1_BITS) & MSK_7BIT);
	midi_tx_p->midi_data[5] = (chk_val & MSK_7BIT);
	midi_tx_p->midi_data[6] = END_SYSEX_STS;
	
 	if (false == packet_to_message( glob_p ,midi_tx_p ,7 )) return false;

	return true;
} // build_systemexclusive
/*****************************************************************************/
bool xmidi_test_vector( // Generates all xmidi test vectors
	GLOBAL_TYP * glob_p  // Pointer to global data structure
) // Returns flag indicating if send message was successful
/*
 * System Message Sequence number assignments:
 * 	0-4			Real Time Messages.
 *  5			Tune Request.
 *	6-133		Song Select.
 *	134-32901	1st of pair is Song Pointer, 2nd of pair is SysEx.
 *	32902...	SysEx (seq no is 5 bytes after manufactuer's ID) // WARNING may never be tested 
 *
 * Channel Message sequence number assignments:
 *	0-255		Single parameter messages
 *	256-65791	Two parameter messages. // WARNING may never be tested 
 *	65792-81407	Control Change (first param can only be 0 - 121).
 *
 *  WARNING as DEF_TESTS = 180, some of above categories are never tetesd 
 */
{
	U32_T chk_val; // Check value for test, (Should be recovered correctly in received messages)


	if (TWO_PARAM_SEQ <= glob_p->test_cnt)
	{	// Two-parameter channel messages plus Song-Pointer or System-Exclusive message
		chk_val = (glob_p->test_cnt - TWO_PARAM_SEQ);
		if (false == build_chan_2param( glob_p ,chk_val )) return false;

		chk_val = (glob_p->test_cnt - SONG_PTR_SEQ); // Initialise Song-Pointer check-value

		// Check if odd test number
		if (chk_val & 1)
		{ // Odd 
			if (false == build_systemexclusive( glob_p ,glob_p->test_cnt )) return false; // System Exclusive
		} // if (chk_val & 1)
		else
		{ // Even
			printf(" Song-Pointer\n" );
			if (false == build_midi_packet( glob_p ,SONG_PTR_STS ,2 ,(chk_val >> 1) )) return false; // Song Pointer
		} // if (chk_val & 1)

		return true;
	} // if (TWO_PARAM_SEQ <= glob_p->test_cnt)

	if (SONG_PTR_SEQ <= glob_p->test_cnt)
	{	// One-parameter channel message plus  Song-Pointer or System-Exclusive message
		if (false == build_chan_1param( glob_p ,glob_p->test_cnt )) return false;

		chk_val = (glob_p->test_cnt - SONG_PTR_SEQ); // Initialise Song-Pointer check-value

		// Check if odd test offset
		if (chk_val & 1)
		{ // Odd 
			if (false == build_systemexclusive( glob_p ,glob_p->test_cnt )) return false; // System Exclusive
		} // if (chk_val & 1)
		else
		{ // Even:
			/* 
			 * NB Song-Pointer check-values need to be even. This is because the 'check algorithm' 
       * for a Song-Pointer discards the LS-bit of chk_val. 
			 * So only even values can be re-created in the result checker.
			 */
			printf(" Song-Pointer\n" );
			if (false == build_midi_packet( glob_p ,SONG_PTR_STS ,2 ,(chk_val >> 1) )) return false; // Song Pointer
		} // if (chk_val & 1)

		return true;
	} // 	if (SONG_PTR_SEQ <= glob_p->test_cnt)

	if (SONG_SEL_SEQ <= glob_p->test_cnt)
	{	// Song-Select system message
		if (false == build_chan_1param( glob_p ,glob_p->test_cnt )) return false;

		printf(" Song-Select\n" );
		chk_val = (glob_p->test_cnt - SONG_SEL_SEQ);
		if (false == build_midi_packet( glob_p ,SONG_SEL_STS ,1 ,chk_val )) return false; // Song-Select

		return true;
	} // if (SONG_SEL_SEQ <= glob_p->test_cnt)

	if (TUNE_REQ_SEQ <= glob_p->test_cnt)
	{	// Tune-Request system message
		if (false == build_chan_1param( glob_p ,glob_p->test_cnt )) return false;

		printf(" Tune-Request\n" );
		if (false == build_midi_packet( glob_p ,TUNE_REQ_STS ,0 ,UNDEFINED )) return false; // Tune-Request

		return true;
	} // if (TUNE_REQ_SEQ <= glob_p->test_cnt)

	if (REAL_TIME_SEQ <= glob_p->test_cnt)
	{	// Real-time system message
		if (false == build_chan_1param( glob_p ,glob_p->test_cnt )) return false;

		printf(" Real-Time\n" );
		if (false == build_midi_packet( glob_p ,glob_p->real_time_data[glob_p->test_cnt] ,0 ,UNDEFINED )) return false; // Real-Time
	} // if (REAL_TIME_SEQ <= glob_p->test_cnt)

	return true;
} // xmidi_test_vector
/*****************************************************************************/
bool do_all_test_vectors( // Generates all xmidi test vectors
	GLOBAL_TYP * glob_p  // Pointer to global data structure
) // Returns flag indicating if send message was successful
{
	// loop through two-parameter channel messages, Song-Pointer and System-Exclusive messages
	for (glob_p->test_cnt=0; glob_p->test_cnt<glob_p->num_tests; glob_p->test_cnt++)
	{
		printf("%4u:" ,glob_p->test_cnt );

		if (false == xmidi_test_vector( glob_p )) return false;

		synchronise( glob_p ); // Wait for test results
	} // for glob_p->test_cnt

	return true;
} // do_all_test_vectors
/*****************************************************************************/
void print_test_results( // Print Test results
	GLOBAL_TYP * glob_p  // Pointer to global data structure
)
{
	CHECK_TYP * check_p = &(glob_p->check_s); // Get pointer to structure of Midi check data 

	std::cout << (check_p->good_cnt + check_p->fail_cnt) << " checks compeleted. ";
	std::cout << check_p->good_cnt << " Passed. ";
  std::cout << check_p->fail_cnt << " Failed." << std::endl; 
} // print_test_results
/*****************************************************************************/
void free_resources( // Free up dynamically created objects 
	GLOBAL_TYP * glob_p  // Pointer to global data structure
)
{
  delete glob_p->tx_data_s.midi_out_p;
  delete glob_p->rx_data_s.midi_in_p;
} // init_globals
/*****************************************************************************/
int main( int argc, char *argv[] )
{
	GLOBAL_TYP glob_s; // Structure of global data


	printf("\nProgram xmiditest Starts ...\n\n");

	init_globals( &glob_s ,argc ,argv );

	// Test Midi configuration
	if (true == configure_midi_ports(&glob_s))
	{ // Midi Configuration OK
		if (false == do_all_test_vectors( &glob_s ))
		{
			printf("\nERROR: Failed to send all test vectors\n");
		} //if (false == do_all_test_vectors( &glob_s ))

		print_test_results( &glob_s );
	} // if (true == configure_midi_ports(&glob_s) )

	free_resources( &glob_s );

	printf("\nProgram Ends\n");

  return 0;
} // main
/*****************************************************************************/
// xmiditest
