//*****************************************//
//  xmidi_transmit.cpp
//
//  Simple program to test MIDI sysex sending and receiving.
//
//*****************************************//

#include "RtMidi.h"
#include "Socket_lite.h"
#include "xmidi_common.h"
#include "xmidi_callback.h"
#include "xmidi_check.h"
#include "xmidi_generate.h"
#include "xmidi_globals.h"
#include "xmidi_transmit.h"

/*****************************************************************************/
static void init_transmit_globals( // Initialise global generation data
	GLOBAL_TYP * glob_p, // Pointer to structure of global data
	int argc, // Number of command-line arguments
	char * argv[] // Array of command-line arguments pointers
)
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	GENER_TYP * gener_p = &(glob_p->gener_s); // Pointer to structure of test-vector generation data
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 


	glob_p->config_id = TRANSMIT; // Configure code to transmit Midi-messages

	// Initialise command-option data
	init_command_options( options_p ,argc ,argv );

  init_common_data( glob_p );

  init_generate_data( glob_p->msg_cfg_arr ,options_p ,gener_p );

	gener_p->file_cfg_p = init_file_config_data( glob_p ); // Configure Generation file 
	check_p->file_cfg_p = NULL;	// NO Byte-Check file
	check_p->config_id = TRANSMIT;	// Configure code to transmit Midi-messages

	// Initialise structure of new verification data
  init_verification_data( &(check_p->cur_verify_s) ,&(check_p->chk_pkt_arr[1]) );
  init_verification_data( &(check_p->new_verify_s) ,&(check_p->chk_pkt_arr[0]) );

	// Initialise data structure for transmitted/received Midi data
  init_midi_tx_data( &(gener_p->tx_data_s) ); // Initialise Midi-transmit structure

	// Check if in loop-back test configuration
	if (options_p->loopback)
	{ // Loop-back mode
	  init_check_data( glob_p->msg_cfg_arr ,options_p ,check_p );

		check_p->chk_tst_vect = true; // Set flag if checking against test vector

	  init_midi_rx_data( &(check_p->rx_data_s) ); // Initialise Midi-receive structure
	} // if (options_p->loopback)

} // init_transmit_globals
/*****************************************************************************/
static void print_transmit_config( // Print Transmission COnfiguration details
	GLOBAL_TYP * glob_p // Pointer to structure of global data
)
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	FILE_CONFIG_TYP * file_cfg_p = &(glob_p->file_cfg_s); // Get pointer to structure of file configuration data


	printf("\nProgram xmidi_transmit Starts ...\n\n");

	// Determine mode of generated Midi-messages file)
	switch(file_cfg_p->mode)
	{
		case INP_FILE : // Use pre-generated file
			printf("Reading pre-generated test-vectors from Midi file %s\n" ,file_cfg_p->name );
		break; // case INP_FILE

		case OUT_FILE : // Generate new file
			printf("Writing newly generated test-vectors to Midi file %s\n" ,file_cfg_p->name );
		break; // case OUT_FILE

		default:
			printf("ERROR: Unexpected File-mode %d detected\n" ,file_cfg_p->mode );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(file_cfg_p->mode)

	printf("No. of tests to generate = %d\n" ,options_p->num_vects );

	if(options_p->seed)
	{
		printf("Transmitting Randomised Tests for Seed = %d\n" ,options_p->seed );
	} // if(options_p->seed)
	else
	{
		printf("Transmitting NON-randomised Tests \n" );
	} // if(options_p->seed)

	if (options_p->loopback)
	{ // Generate test-vectors
		printf("Testing in LoopBack mode\n");
	} // if (options_p->loopback)
	else
	{
		if (options_p->on_host)
		{ // Generate test-vectors
			printf("Transmitting to target IP Address %s\n" ,options_p->ip_addr );
		} // if (options_p->on_host)
		else
		{
			printf("Transmitting at target IP Address %s\n" ,options_p->ip_addr );
		} // else !(options_p->on_host)
	} // else !(options_p->loopback)

} // print_transmit_config
/*****************************************************************************/
static bool configure_transmit_midi_ports( // Configure which Midi output port to use
	GLOBAL_TYP * glob_p // Pointer to structure of global data
) // Returns true if configuration successful
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	GENER_TYP * gener_p = &(glob_p->gener_s); // Pointer to structure of test-vector generation data
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 

	if (false == configure_midi_output_port( gener_p ,glob_p->port_names )) return false;

	// Check if in loop-back test configuration
	if (options_p->loopback)
	{
		if (false == configure_midi_input_port( check_p ,glob_p->port_names )) return false;
	} // if (options_p->loopback)

	return true;
} // 	configure_transmit_midi_ports
/*****************************************************************************/
static void free_transmit_resources( // Free up dynamically created objects 
	GLOBAL_TYP * glob_p // Pointer to structure of global data
)
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	GENER_TYP * gener_p = &(glob_p->gener_s); // Pointer to structure of test-vector generation data
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 
	int error; // Error status (0 for NO errors)


	error =	fflush( NULL ); // Flush any open output files

	// Check file-close succeeded
	if (0 != error)
	{

		printf("ERROR: While flushing files returned %d \n" ,error );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (0 != error)

	error =	fclose( glob_p->file_cfg_s.ptr ); // Close midi test data file

	// Check file-close succeeded
	if (0 != error)
	{
		printf("ERROR: Failed to close file '%s'\n" ,glob_p->file_cfg_s.name );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (0 != error)

  delete gener_p->tx_data_s.midi_out_p;

	if (options_p->loopback)
	{
	  delete check_p->rx_data_s.midi_in_p;
	} // if (options_p->loopback)
} // free_transmit_resources
/*****************************************************************************/
int main( int argc, char *argv[] )
{
	GLOBAL_TYP glob_s; // Structure of global data
	GLOBAL_TYP * glob_p = &glob_s; // Pointer to structure of global data
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 
	U8_T tst_res; // test result returned to shell


	init_transmit_globals( glob_p ,argc ,argv );

	if (options_p->print)
	{
		print_transmit_config( glob_p );
	} // if (options_p->print)

	// Test Midi configuration
	if (true == configure_transmit_midi_ports( glob_p ))
	{ // Midi Configuration OK

		// Check if in loop-back test configuration
		if (!options_p->loopback)
		{ // Stand-alone mode
			init_socket( glob_p ); // Initialise socket
		} // if (!options_p->loopback)

		if (false == do_all_test_vectors( glob_p ))
		{
			printf("\nERROR: Failed to send all test vectors\n");
		} //if (false == do_all_test_vectors( glob_p ))
	} // if (true == configure_transmit_midi_ports

	free_transmit_resources( glob_p );

	// Check if in loop-back test configuration
	if (options_p->loopback)
	{ // Loop-back mode
		print_all_message_results( check_p );

		tst_res = gen_final_test_result( check_p );	// Calculate final 8-bit test result
	} // if (options_p->loopback)
	else
	{ // Dual-platform mode
		glob_p->sckt_p->socReceive( (char *)&tst_res ,1 ); // Wait for test result from Receiver
	} // if (options_p->loopback)

	printf("\nTotal Failures= %d \n" ,(int)tst_res );
	if (options_p->print) printf("Program Ends \n" );

	return tst_res; // Return test result to shell
} // main
/*****************************************************************************/
// xmidi_transmit
