//*****************************************//
//  xmidi_receive.cpp
//
//  Simple program to test MIDI sysex sending and receiving.
//
//*****************************************//

#include "RtMidi.h"
#include "Socket_lite.h"
#include "xmidi_common.h"
#include "xmidi_generate.h"
#include "xmidi_callback.h"
#include "xmidi_check.h"
#include "xmidi_globals.h"
#include "xmidi_receive.h"

/*****************************************************************************/
static void init_receive_globals( // Initialise global check data
	GLOBAL_TYP * glob_p, // Pointer to structure of global data
	int argc, // Number of command-line arguments
	char * argv[] // Array of command-line arguments pointers
)
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	GENER_TYP * gener_p = &(glob_p->gener_s); // Pointer to structure of test-vector generation data
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 


	glob_p->config_id = RECEIVE; // Configure code to receive Midi-messages

	// Initialise common command-options data
	init_command_options( options_p ,argc ,argv );

  init_common_data( glob_p );

  init_check_data( glob_p->msg_cfg_arr ,options_p ,check_p ); // Initialise check-data values

	check_p->file_cfg_p = init_file_config_data( glob_p ); // Configure Byte-Check file
	gener_p->file_cfg_p = NULL;	// NO Generation file
	check_p->config_id = RECEIVE;	// Configure code to receive Midi-messages

	check_p->chk_tst_vect = true; // Set flag if checking against test vector

	// Initialise structure of new verification data
  init_verification_data( &(check_p->new_verify_s) ,&(check_p->chk_pkt_arr[0]) );
  init_verification_data( &(check_p->cur_verify_s) ,&(check_p->chk_pkt_arr[1]) );

  init_midi_rx_data( &(check_p->rx_data_s) );	// MB~ ToDo. Initialise structure of received Midi data 

	// Check if in loop-back test configuration
	if (options_p->loopback)
	{ // Loop-back mode
		init_generate_data( glob_p->msg_cfg_arr ,options_p ,gener_p ); // Initialise test-vector generation data

	  init_midi_tx_data( &(gener_p->tx_data_s) );	// Initialise data structure for transmitted Midi data
	} // if (options_p->loopback)
} // init_receive_globals
/*****************************************************************************/
static void print_receive_config( // Print Reception Configuration details
	GLOBAL_TYP * glob_p // Pointer to structure of global data
)
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	FILE_CONFIG_TYP * file_cfg_p = &(glob_p->file_cfg_s); // Get pointer to structure of file configuration data


	printf("\nProgram xmidi_receive Starts ...\n\n");

	// Determine mode of check file
	switch(file_cfg_p->mode)
	{
		case INP_FILE : // Compare results against pre-generated input file
			printf("Reading check data from Midi file %s\n" ,file_cfg_p->name );
		break; // case INP_FILE

		case OUT_FILE : // Dump result to output file for off-line checking
			printf("Writing received results to Midi file %s\n" ,file_cfg_p->name );
		break; // case OUT_FILE

		default:
			printf("ERROR: Unexpected File-mode %d detected\n" ,file_cfg_p->mode );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(file_cfg_p->mode)

	printf("No. of tests to check = %d\n" ,options_p->num_vects );

	if(options_p->seed)
	{
		printf("Receiving Randomised Tests for Seed = %d\n" ,options_p->seed );
	} // if(options_p->seed)
	else
	{
		printf("Receiving NON-randomised Tests \n" );
	} // if(options_p->seed)

	if (options_p->loopback)
	{ // Generate test-vectors
		printf("Testing in LoopBack mode\n");
	} // if (options_p->loopback)
	else
	{
		if (options_p->on_host)
		{ // Generate test-vectors
			printf("Receiving from target IP Address %s\n" ,options_p->ip_addr );
		} // if (options_p->on_host)
		else
		{
			printf("Receiving at target IP Address %s\n" ,options_p->ip_addr );
		} // else !(options_p->on_host)
	} // else !(options_p->loopback)

} // print_receive_config
/*****************************************************************************/
static bool configure_receive_midi_ports( // Configure which Midi output port to use
	GLOBAL_TYP * glob_p // Pointer to structure of global data
) // Returns true if configuration successful
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	GENER_TYP * gener_p = &(glob_p->gener_s); // Pointer to structure of test-vector generation data
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 


	if (false == configure_midi_input_port( check_p ,glob_p->port_names )) return false;

	// Check if in loop-back test configuration
	if (options_p->loopback)
	{
		if (false == configure_midi_output_port( gener_p ,glob_p->port_names )) return false;
	} // if (options_p->loopback)

	return true;
} // 	configure_receive_midi_ports
/*****************************************************************************/
static void free_receive_resources( // Free up dynamically created objects 
	GLOBAL_TYP * glob_p // Pointer to structure of global data
)
	/*
	 * NB By design, the input file is NOT closed.
	 * This is because mycallback() may cause an uncontrolled input file read
	 * A read on an unclosed file can be handled, 
	 * however a read on a closed file causes undefined behaviour
	 */
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	GENER_TYP * gener_p = &(glob_p->gener_s); // Pointer to structure of test-vector generation data
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 
	int error; // Error status (0 for NO errors)


	error =	fflush( NULL ); // Flush any open output files

	// Check file-close succeeded
	if (0 != error)
	{

		printf("ERROR: While flushing files returned %d \n" ,error );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (0 != error)

	if (options_p->loopback)
	{
		delete gener_p->tx_data_s.midi_out_p;
	} // if (options_p->loopback)

  delete check_p->rx_data_s.midi_in_p;
} // free_receive_resources
/*****************************************************************************/
int main( int argc, char *argv[] )
{
	GLOBAL_TYP glob_s; // Structure of global data
	GLOBAL_TYP * glob_p = &glob_s; // Pointer to structure of global data
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 
	U8_T tst_res; // test result (sent to shell and transmitter)


	init_receive_globals( glob_p ,argc ,argv );

	if (options_p->print)
	{
		print_receive_config( glob_p );
	} // if (options_p->print)

	// Test Midi configuration
	if (true == configure_receive_midi_ports( glob_p ))
	{ // Midi Configuration OK

		// Check if in loop-back test configuration
		if (!options_p->loopback)
		{ // Stand-alone mode
			init_socket( glob_p ); // Initialise socket
		} // if (!options_p->loopback)

		if (false == do_all_test_vectors( glob_p ))
		{
			printf("\nERROR: Failed to send all test vectors\n");
		} //if (false == do_all_test_vectors( glob_p ))
	} // if (true == configure_receive_midi_ports

	free_receive_resources( glob_p );

	print_all_message_results( check_p );  // Print all results statistics for each Midi-message type

	tst_res = gen_final_test_result( check_p );	// Calculate final 8-bit test result

	// Check if in loop-back test configuration
	if (!options_p->loopback)
	{ // Dual-platform mode
		// Send result to transmitter
		glob_p->sckt_p->socSend( (const char *)&tst_res ,1 );
	} // if (options_p->loopback)

	printf("\nTotal Failures= %d \n" ,(int)tst_res );
	if (options_p->print) printf("Program Ends \n" );

	return tst_res; // Return result to shell
} // main
/*****************************************************************************/
// xmidi_receive
