//*****************************************//
//  xmidi_generate.cpp
//
//  Simple program to test MIDI sysex sending and receiving.
//
//*****************************************//

#include "RtMidi.h"
#include "xmidi_common.h"
#include "xmidi_generate.h"

/*****************************************************************************/
void init_generate_data( // Initialise test-vector generation data
	MSG_CONFIG_TYP msg_cfg_arr[], // Array of structures of Midi-message configuration data
	OPTIONS_TYP * options_p, // Pointer to command-option data structure
	GENER_TYP * gener_p // Pointer to structure of test-vector generation data
)
{
	// Initialise Non-configurable data ...

	// Real-Time messages. WARNING These are NOT is obvious slots
	gener_p->real_time_data[0] = TIME_CLK_STS;
	gener_p->real_time_data[1] = START_SONG_STS;
	gener_p->real_time_data[2] = CONT_SONG_STS;
	gener_p->real_time_data[3] = STOP_SONG_STS;

	// Single Parameter Channel messages
	gener_p->param1_data[0] = PROG_CHANGE_STS;
	gener_p->param1_data[1] = CHAN_PRESS_STS;

	// Double Parameter Channel messages
	gener_p->param2_data[0] = NOTE_ON_STS;
	gener_p->param2_data[1] = NOTE_OFF_STS;
	gener_p->param2_data[2] = KEY_PRESS_STS;
	gener_p->param2_data[3] = PITCH_WHEEL_STS;
	gener_p->param2_data[4] = CONTROL_CHANGE_STS;

	gener_p->msg_cfg_arr_p = msg_cfg_arr; // Set pointer to first element of Midi-message configuration data

	// Initialise test generation data (NB re-written for each Midi-message)
	init_test_generation_data( msg_cfg_arr ,&(gener_p->tst_gen_s) ,options_p->seed );

	gener_p->print = options_p->print; // Initialise screen-printing mode
} // init_generate_data
/*****************************************************************************/
bool configure_midi_output_port( // Configure which Midi output port to use
	GENER_TYP * gener_p,  // Pointer to structure of test-vector generation data
	const char port_names[NUM_VALID_PORTS][STR_NAME_LEN] // Array of valid port names
) // Returns true if configuration successful
{
	MIDI_TX_TYP * midi_tx_p = &(gener_p->tx_data_s); // Get pointer to structure of Midi data to transmit


  // RtMidiOut & RtMidiIn constructors
  try 
	{
    midi_tx_p->midi_out_p = new RtMidiOut();
  } // try
  catch ( RtMidiError &error ) 
	{
		printf("\nERROR: Failed to successfully construct RtMidi Output Object\n");
    error.printMessage();
    return false;
  } // catch

  try 
	{
    midi_tx_p->port_id = open_midi_port( midi_tx_p->midi_out_p ,port_names ,"Output" ,gener_p->print );
    if (NUM_VALID_PORTS == midi_tx_p->port_id) return false;
  } // try
  catch ( RtMidiError &error ) 
	{
		printf("\nERROR: Failed to successfully open Midi Output Port\n");
    error.printMessage();
    return false;
  } // catch

#if defined(__MACOSX_CORE__)
	// This is a cludge to fix slow response when Mac transmit over USB2Midi Cable
	if (USB2MIDI_CHINA == midi_tx_p->port_id)
	{
		midi_tx_p->pace_time = MAX_PACE_TIME;
	} // if ((0 == strcmp( port_names[USB2MIDI_CHINA] ,curr_portName))
#endif // defined(__MACOSX_CORE__)

	if (gener_p->print)
	{
		printf("Pace-Time = %dms \n" ,midi_tx_p->pace_time );
	} // if (gener_p->print)

	return true;
} // 	configure_midi_output_port
/*****************************************************************************/
static void build_fixed_midi_packet( // Build Midi-message packet of known length
	GENER_TYP * gener_p,  // Pointer to structure of test-vector generation data
	TST_VECT_TYP * tst_vect_p // Pointer to test-vector data 
) // Return status flag
{
	MSG_CONFIG_TYP * msg_config_p = tst_vect_p->msg_cfg_p; // Pointer to structure of Midi-message configuration data
	MIDI_TX_TYP * midi_tx_p = &(gener_p->tx_data_s); // Get pointer to structure of Midi data to transmit
	MIDI_PACKET_TYP * packet_p = &(midi_tx_p->packet_s); //Get pointer to packet of Midi-message data 
	U32_T chk_val = tst_vect_p->param; // Initialise check value with Midi-message data parameter
	S32_T byte_cnt; // Midi data byte counter
	U8_T ls_off = (msg_config_p->len - 1); // Midi-message offset for LS byte of check-value


	packet_p->midi_status = tst_vect_p->sts;
	packet_p->data_len = msg_config_p->len;

	// NB tests configured to send MS byte of check value first
	for (byte_cnt=0; byte_cnt<msg_config_p->len; byte_cnt++)
	{
		packet_p->midi_data[ls_off - byte_cnt] = (chk_val & MSK_7BIT); // Store LS 7-bits
		chk_val >>= MIDI_VAL1_BITS; // Get next 7-bits
	} // for byte_cnt

	return;
} // build_fixed_midi_packet
/*****************************************************************************/
static void build_varying_midi_packet( // Build Midi-message packet of variable length
	GENER_TYP * gener_p,  // Pointer to structure of test-vector generation data
	TST_VECT_TYP * tst_vect_p // Pointer to test-vector data 
) // Return status flag
// NB tests configured to send MS byte of check value first
{
	MSG_CONFIG_TYP * msg_config_p = tst_vect_p->msg_cfg_p; // Pointer to structure of Midi-message configuration data
	U8_T byte_data[USB_MAX_PKT_LEN]; // Array of Midi data bytes
	MIDI_TX_TYP * midi_tx_p = &(gener_p->tx_data_s); // Get pointer to structure of Midi data to transmit
	MIDI_PACKET_TYP * packet_p = &(midi_tx_p->packet_s); //Get pointer to packet of Midi-message data 
	U32_T chk_val = tst_vect_p->param; // Initialise check value with Midi-message data parameter
	S32_T num_bytes = 0; // No. of data bytes required to store check value
	S32_T data_cnt; // data byte counter
	S32_T shift_bits; // Number of data bits to place in each SysEx data byte
	S32_T bit_mask; // Used to mask required LS data bits from check value


	packet_p->midi_status = msg_config_p->status;

	// Determine System Exclusive message type
	switch(msg_config_p->id)
	{
		case SMALL_SYSEX_ID :	// Small Manufacturer's ID Length
			packet_p->midi_data[0] = MANUFACTURER_ID; // NB NON-zero signals Short Manufacturer's ID
			packet_p->data_len = 1; // Initialise 'Small' data-length
			bit_mask = MSK_7BIT; // NB Pack maximum No. of bits to each data byte to create minimum No of SysEx data bytes
			shift_bits = MIDI_VAL1_BITS; // Initialise shift bits for bit_mask
		break; // case SMALL_SYSEX_ID

		case LARGE_SYSEX_ID :	// Large Manufacturer's ID Length
			packet_p->midi_data[0] = 0; // NB Zero signals Long Manufacturer's ID
			packet_p->midi_data[1] = MANUFACTURER_ID;
			packet_p->midi_data[2] = MANUFACTURER_ID;
			packet_p->data_len = 3; // Initialise 'Large' data-length
			bit_mask = SYSEX_BIT_MASK; // NB Use smaller No. of data bits to create more SysEx data bytes
			shift_bits = SYSEX_SHIFT_BITS; // Initialise shift bits for bit_mask
		break; // case LARGE_SYSEX_ID

		default:
			printf("ERROR: Detected unexpected System Exclusive Identifier of %d \n" ,msg_config_p->id );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break;
	} // switch(msg_config_p->id)

	// Store variable number of Sys-Ex data bytes
	while ((chk_val > 0) && (num_bytes < USB_MAX_PKT_LEN))
	{
		byte_data[num_bytes] = (chk_val & bit_mask); // Mask out LS bits 		
		chk_val >>= shift_bits; // Remove stored bits
		num_bytes++; // Increment No. of data bytes
	} // while (chk_val > 0)

	// Check for overflow
	if (0 < chk_val)
	{
		printf("ERROR: Check value overflowed Midi-packet data length of %d \n" ,USB_MAX_PKT_LEN );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (0 < chk_val)

	// Reverse order of data bytes so MS-byte is first. NB This makes life easier at receiver!-)
	for (data_cnt=(num_bytes - 1); data_cnt>=0; data_cnt--)
	{
		packet_p->midi_data[packet_p->data_len] = byte_data[data_cnt];
		packet_p->data_len++; // Increment data-length
 	} // for data_cnt

	packet_p->midi_data[packet_p->data_len] = END_SYSEX_STS; // Add 'End-of-SysEx' byte
	packet_p->data_len++; // Increment data-length

	return;
} // build_varying_midi_packet
/*****************************************************************************/
void generate_midi_message( // Generates one Midi-message test
	GENER_TYP * gener_p  // Pointer to structure of test-vector generation data
) // Returns flag indicating if send message was successful
{
	TST_VECT_TYP * tst_vect_p = &(gener_p->tst_vect_s); // Get local pointer to test-vector data 
	TST_GEN_TYP * tst_gen_p = &(gener_p->tst_gen_s); // Get local pointer to test generation data 

	// Populate test-vector data for creating next Midi-message 
	gen_test_vector_data( tst_gen_p ,tst_vect_p );

	// Determine if System-Exclusive message
	if (STRT_SYSEX_STS == tst_vect_p->sts)
	{ // System Exclusive
		build_varying_midi_packet( gener_p ,tst_vect_p ); // Build variable length Midi-message
	} // if (STRT_SYSEX_STS == tst_vect_p->msg_cfg_p->status)
	else
	{ // NOT System Exclusive
		build_fixed_midi_packet( gener_p ,tst_vect_p ); // Build fixed length Midi-message
	} // else !(STRT_SYSEX_STS == tst_vect_p->sts)

	return;
} // generate_midi_message
/*****************************************************************************/

// xmidi_generate
