//*****************************************//
//  xmidi_common.h
//
//  Re-create xmidi_common using RtMidi
//
//*****************************************//

#ifndef XMIDI_COMMON_H
#define XMIDI_COMMON_H

#include <stdio.h>
#include <string.h>
#include <limits.h>

#include <bitset>
#include <iostream>
#include <cstdlib>
#include <typeinfo>
#include <assert.h>

#include "types64bit.h"

// Platform-dependent sleep routines.
#if defined(__WINDOWS_MM__)
  #include <windows.h>
  #define SLEEP_FN( milliseconds ) Sleep( (DWORD) milliseconds ) 
#else // Unix variants
  #include <unistd.h>
  #define SLEEP_FN( milliseconds ) usleep( (unsigned long) (milliseconds * 1000.0) )
#endif

#define USB_MAX_PKT_LEN 256

#define DEF_TESTS 256 // Default Number of tests
#define DEF_SEED 0 // By Default do NOT randomise tests
#define DEF_PRINT 0 // By default, screen printing is OFF
#define DEF_PACE_TIME 1 // Default Midi-message 'pacing-time' in millisecs

// NB Some configurations require a much longer pacing time. E.g. Mac-->USB2Midi_Cable
#define MAX_PACE_TIME 1 // Maximum Midi-message 'pacing-time' in millisecs. No required any more

#define CHAN_BITS 4  // No of bits used to represent Midi channels (instruments)
#define NUM_CHANS (1 << CHAN_BITS) // Number of Midi channels (instruments)
#define CHAN_MASK (NUM_CHANS - 1) // Mask to extract Midi channel number

#define MAX_BURST_LEN 16 // Maximum No. of Midi-messages in a burst (sent between synchronisation points)
#define NUM_RX_BUFS (MAX_BURST_LEN << 1) // Create enough Midi-message received packet buffers to hold 2 bursts

#define MIDI_STS_BITS 8 // No. of bits in Midi Status value (8) 
#define MIDI_VAL1_BITS 7 // No. of bits in single-parameter Midi data value (7) 
#define MIDI_VAL2_BITS (MIDI_VAL1_BITS << 1) // No. of bits in double-parameter Midi data value (14)

#define NUM_MIDI_1VALS (1 << MIDI_VAL1_BITS) // No. of possible values for a single Midi data parameter (128)
#define NUM_MIDI_2VALS (1 << MIDI_VAL2_BITS) // No. of possible values for a double Midi data parameter (16384)

#define MSK_14BIT (NUM_MIDI_2VALS - 1) // Mask for 14-bit Midi data value (16383)
#define MSK_7BIT (NUM_MIDI_1VALS - 1) // Mask for 7-bit Midi data value (127)
#define MSK_4BIT (MSK_7BIT >> 3) // Mask for 4-bit System-Exclusive check value (15)

#define LNX_IP_ADDR "10.0.102.125"
#define MAC_IP_ADDR "10.0.98.88"
#define WIN_IP_ADDR "10.0.102.157"

#define SYNC_ON 0 // Set synchronisation flag
 
#define NAP_TIME 500 // Sleep time in milli-seconds 

#define UNDEFINED (-1) // Undefined Midi value
#define UNTESTED (-2) // Un-tested Midi value

#define STR_NAME_LEN 256 // name string length
#define STR_MIDI_LEN 32 // Midi-message description string length

#define MANUFACTURER_ID 0x02 // "Big Briar" manufacturer's ID (NB Must be Non-zero).

// Channel-message Status bytes
#define NUM_1PARAM_MSGS 2 // No of 1-Parameter channel messages.
#define NUM_2PARAM_MSGS 5 // No of 2-Parameter channel messages.

#define MISSING_STS				  0x00 // Invalid Midi status value used to signal missing data

#define NOTE_OFF_STS				0x80
#define NOTE_ON_STS					0x90
#define KEY_PRESS_STS				0xA0
#define CONTROL_CHANGE_STS	0xB0
#define PROG_CHANGE_STS			0xC0
#define CHAN_PRESS_STS			0xD0
#define PITCH_WHEEL_STS			0xE0

// System-message Status bytes
#define NUM_SYS_MSGS 8 // Max. possible No of System messages. WARNING. Some are Undefined
#define TSTED_SYS_MSGS 5 // No. of tested System messages.

#define STRT_SYSEX_STS	0xF0
#define MIDI_TIME_STS		0xF1
#define SONG_PTR_STS		0xF2
#define SONG_SEL_STS		0xF3
#define UNDEF_SM_F4_STS	0xF4 // Undefined
#define UNDEF_SM_F5_STS	0xF5 // Undefined
#define TUNE_REQ_STS		0xF6
#define END_SYSEX_STS		0xF7

// Real-Time message Status bytes
#define NUM_RT_MSGS 8 // Max. possible No of Real-Time messages. WARNING. Some are Undefined
#define TSTED_RT_MSGS 4 // No of tested Real-Time messages.

#define TIME_CLK_STS		0xF8 //MB~ 0xF8
#define UNDEF_RT_F9_STS	0xF9 // Undefined
#define START_SONG_STS	0xFA //MB~ 0xFA
#define CONT_SONG_STS		0xFB //MB~ 0xFB 
#define STOP_SONG_STS		0xFC
#define UNDEF_RT_FD_STS	0xFD // Undefined
#define ACTIV_SENSE_STS	0xFE // Currently untested
#define RESET_STS				0xFF // Currently untested

// Sequence No. breakpoints for system messages
#define MIDI_TIME_SEQ 0							// Midi-Time_Code
#define SONG_SEL_SEQ (MIDI_TIME_SEQ + NUM_MIDI_1VALS) // Song Select
#define SONG_PTR_SEQ (SONG_SEL_SEQ + NUM_MIDI_1VALS)   // Song Position Pointer
#define REAL_TIME_SEQ (SONG_PTR_SEQ + NUM_MIDI_2VALS) // Real time messages
#define TUNE_REQ_SEQ (REAL_TIME_SEQ + TSTED_RT_MSGS) // Tuning test
#define SYSEX_SMALL_SEQ (TUNE_REQ_SEQ + 1) // 'Small' System Exclusive (NB Mop up to a NUM_MIDI_2VALS channel-message boundary 
#define SYSEX_LARGE_SEQ (REAL_TIME_SEQ + (NUM_MIDI_2VALS << 1)) // 'Large' System Exclusive (NB Starts at a NUM_MIDI_2VALS chan-mess boundary)

// Sequence No. breakpoints for 1-parameter channel messages (128 apart)
#define PROG_CHANGE_SEQ 0
#define CHAN_PRESS_SEQ (PROG_CHANGE_SEQ + NUM_MIDI_1VALS)
#define TWO_PARAM_SEQ (CHAN_PRESS_SEQ + NUM_MIDI_1VALS)

// Sequence No. breakpoints for 2-parameter channel messages (16384 apart)
#define NOTE_ON_SEQ (TWO_PARAM_SEQ)
#define NOTE_OFF_SEQ (NOTE_ON_SEQ + NUM_MIDI_2VALS)
#define KEY_PRESS_SEQ (NOTE_OFF_SEQ + NUM_MIDI_2VALS)
#define PITCH_WHEEL_SEQ (KEY_PRESS_SEQ + NUM_MIDI_2VALS)
#define CONTROL_CHANGE_SEQ (PITCH_WHEEL_SEQ + NUM_MIDI_2VALS)

#define SYSEX_SHIFT_BITS 4 // Bit-shift used to create check-value for 'Large' System-Exclusive Messages
#define NUM_SYSEX_VALS (1 << SYSEX_SHIFT_BITS) // No. of unique Sys-Ex data-byte values
#define SYSEX_BIT_MASK (NUM_SYSEX_VALS - 1) // Bit-mask use to extract Sys-Ex data-bits from Check value

#define VECT_OFF ((1 << 10) - 16) //MB~ Used to test Long System-Exclusive midi-messages
//MB~ #define VECT_OFF 0 //MB~

#define NUM_CHAN_TYPS 7 // Number of different Channel message types tested
#define NUM_SYS_TYPS  6 // Number of different System-common message types tested
#define NUM_REAL_TYPS 3 // Number of different System Real-time message types tested

#define FREQ_CHAN_TYPS 7 // 7 Frequency of occurance of each Channel message type
#define FREQ_SYS_TYPS 2 // 2 Frequency of occurance of each System-common message type
#define FREQ_REAL_TYPS 1 // 1 Frequency of occurance of each System Real-time message type

#define MSG_MAP_BITS 6 // 6 No. Of bits used to map test-vector identifier to Midi-message category
#define NUM_MSG_MAP (1 << MSG_MAP_BITS) // No. Of high-level Midi-message values
#define MSG_MAP_MASK (NUM_MSG_MAP - 1) // Bit-mask used to extract code for high-level Midi-message value

// Midi-message mapping ranges
#define CHAN_MAP_RNG	(FREQ_CHAN_TYPS * NUM_CHAN_TYPS) // (7 * 7)
#define SYS_MAP_RNG		(FREQ_SYS_TYPS  * NUM_SYS_TYPS) // (2 * 6)
#define REAL_MAP_RNG 	(FREQ_REAL_TYPS * NUM_REAL_TYPS) // (1 * 3)
#define ALL_MAP_RNG  (CHAN_MAP_RNG + SYS_MAP_RNG + REAL_MAP_RNG) // Check value. NB Must add to a power-of-2

// Check for correct mapping
#if (ALL_MAP_RNG != NUM_MSG_MAP)
#error ERROR: Incorrect mapping of Test-vector identifiers to Midi-message identifiers
#endif  // (ALL_MAP_RNG != NUM_MSG_MAP)

#define PARAM_BITS 18 // Total No. Midi-message parameter bits (4-bit channel_id + 2 x 7-bit data_id)

#define RND_BITS (MSG_MAP_BITS + PARAM_BITS) // Total number of random bits required
#define SHIFT_I_BITS (32 - RND_BITS) // Bit-shift used  to get Most-Significant Internal bits. NB Assumes 32-bit computation
#define NUM_RNDS (1 << RND_BITS) // No. of Unique random test-vect identifiers available (before repeating)
#define MAX_RND (NUM_RNDS - 1) // Maximum random test-vect identifier

/* The following definitions are used to evaluate the maximum expected Midi-message length.
 * This is required to handle corrupted messages, which could possibly be very long.
 * This max. message length occurs for a 'Large' SysEx message with the largest parameter value.
 * The max. No. of SysEx data-bytes depends on how this parameter value is allocated to the data bytes.
 * Currently 4 bits of the parameter-id are placed in each data-byte
 */
#define MAX_SYSEX_PARAM_LEN ((PARAM_BITS + SYSEX_SHIFT_BITS - 1) / SYSEX_SHIFT_BITS) // Max 'parameter' data-bytes (5)
#define MAX_SYSEX_LEN (1 + 3 + MAX_SYSEX_PARAM_LEN + 1) // Max. SysEx length: Start-SysEx +(3 x Manu-Id) +End-SysEx (10)

typedef enum MANU_ID_LEN_ETAG // Enumerate Manufacturers Identifier lengths (in bytes)
{
  SMALL_MANU_ID = 1, // One-byte Manufacturers Id
  LARGE_MANU_ID = 3, // Three-byte Manufacturers Id
  NUM_MANU_ID_LENS // Handy Value!-)
} MANU_ID_LEN_ENUM;

typedef enum COMMS_CONFIG_ETAG // Enumerate communication configurations
{
  TRANSMIT = 0, // Transmit Midi-messages 
  RECEIVE, // Receive Midi-massages
  NUM_COMMS_CONFIGS	// Handy Value!-)
} COMMS_CONFIG_ENUM;

typedef enum VALID_PORT_ETAG // Enumerate valid Midi ports
{
	// WARNING: Code relies on first value of zero
  XCOREUSB = 0, // USB Audio cable connected to xCORE
	USB2MIDI_ALESIS, // Alesis USB <--> Midi cable (converts between USB Audio-Midi and 3150 Byte/sec Midi (Din-plug)
  USB2MIDI_CHINA, // USB <--> Midi cable (converts between USB Audio-Midi and 3150 Byte/sec Midi (Din-plug)
	USB2MIDI_EMU, // E-MU USB <--> Midi cable (converts between USB Audio-Midi and 3150 Byte/sec Midi (Din-plug)
  NUM_VALID_PORTS	// Handy Value!-)
} VALID_PORT_ENUM;

/* All Midi Messages under test */
typedef enum MIDI_MSG_ETAG
{
	PROG_CHANGE_ID = 0, // Channel messages
	CHAN_PRESS_ID,
	NOTE_ON_ID,
	NOTE_OFF_ID,
	KEY_PRESS_ID,
	PITCH_WHEEL_ID,
	CONTROL_CHANGE_ID,

	TUNE_REQ_ID, // System-common messages
	MIDI_TIME_ID,
	SONG_SEL_ID,
	SONG_PTR_ID,
	SMALL_SYSEX_ID,
	LARGE_SYSEX_ID,

	START_SONG_ID, // System Real-time messages
	CONT_SONG_ID,
	STOP_SONG_ID,

  NUM_MIDI_MSGS	// Handy Value!-)
} MIDI_MSG_ENUM;

typedef enum FILE_MODE_ETAG // File I/O modes
{
  NO_FILE = 0, // No file I/O required
  INP_FILE, // Input file required
  OUT_FILE, // Output file required
  NUM_FILE_MODES	// Handy Value!-)
} FILE_MODE_ENUM;

// Structure containing file configuration data
typedef struct FILE_CONFIG_TAG
{
  char name[STR_NAME_LEN]; // file name
	FILE * ptr; // Pointer to file
	FILE_MODE_ENUM mode; // file I/O mode (e.g. INP_FILE/OUT_FILE)
} FILE_CONFIG_TYP;

// Structure containing MIDI-message configuration data
typedef struct MSG_CONFIG_TAG
{
  char name[STR_MIDI_LEN]; // Text description of Midi-message
	MIDI_MSG_ENUM id; // Midi-message identifier
 	U8_T status; // Midi Status byte
	S32_T len;	// No. of Midi data bytes
} MSG_CONFIG_TYP;

// Structure containing test generation data
typedef struct TST_GEN_TAG
{
	const MSG_CONFIG_TYP * msg_cfg_arr; // Pointer to of array of all Midi-message configuration data
	U32_T rand; // internal 32-bit random number (re-generated on each call)
	S32_T cnt; // counts number of random numbers generated
	bool on; // Flag set if test-vectors are randomised
} TST_GEN_TYP;

// Structure for packet of Midi-message data
typedef struct MIDI_PACKET_TAG
{
	U8_T midi_data[USB_MAX_PKT_LEN]; // Array of Midi data bytes
  R64_T diff_time; // Difference time (since previous time-stamp)
	S32_T data_len; // No. of active bytes of Midi-data
	U8_T midi_status; // Midi Status byte
} MIDI_PACKET_TYP;

// Structure of test-vector data
typedef struct TST_VECT_TAG
{
	U8_T sts; // Midi Status identifier
	U32_T param; // Parameter identifier (Encodes Midi data/channel info)
	MSG_CONFIG_TYP * msg_cfg_p; // Pointer to structure of Midi-message configuration data
	MIDI_PACKET_TYP * packet_p; // Pointer to packet of Midi-message data
} TST_VECT_TYP;

// Structure containing received midi data
typedef struct MIDI_RX_TAG
{
	MIDI_PACKET_TYP pkt_bufs[NUM_RX_BUFS]; // array of received packets

  RtMidiIn * midi_in_p; // Pointer to RtMidiIn Object
	S32_T byte_id; // Identifier for position of byte in Midi packet
} MIDI_RX_TYP;

// Structure containing midi data to transmit
typedef struct MIDI_TX_TAG
{
  std::vector<U8_T> msg_s; // Array of bytes to hold message
  std::bitset<8> bits; // Array of 8 bits (used for printing binary)
	MIDI_PACKET_TYP packet_s; // Packet of Midi-message data to transmit

  RtMidiOut * midi_out_p; // Pointer to RtMidiOut Object

	S32_T pace_time; // Midi-message pacing time in millisecs
	S32_T burst_tx_cnt; // Counts a burst of Midi messages written to Output port
	VALID_PORT_ENUM port_id; // Unigue identifier for Output Midi-port
} MIDI_TX_TYP;

// Structure containing command-line options data
typedef struct OPTIONS_TAG
{
  char file_nam[STR_NAME_LEN]; // Input/Output file name
	char ip_addr[STR_NAME_LEN]; // Ethernet IP address of transmitter (Server)
	S32_T seed; // Random Number seed, Zero for NOT random
	S32_T num_vects; // Number of test-vectors requested (per channel)
	FILE_MODE_ENUM file_mode; // File-mode i.e Input/Output
	bool loopback; // Do Loop-back test (H/W must be configured with Midi-Out connected to Midi-In)
	bool print; // Print to screen (if flag set)
	bool on_host; // Flag set if executable is being run on host-machine (other-wise target machine assumed) 
} OPTIONS_TYP;

/*****************************************************************************/
VALID_PORT_ENUM open_midi_port( // Opens a Midi port
	RtMidi * rtmidi_p, // Pointer to RtMidi object
	const char port_names[NUM_VALID_PORTS][STR_NAME_LEN], // Array of valid port names
	const char * flow_str, // String holding flow direction (Input or Output)  
	bool do_print // Flag set if screen-printing required
); // Returns port identifier if successful, or NUM_VALID_PORTS for failure
/*****************************************************************************/
void init_test_vector_data( // Initialise structure of test-vector data
	TST_VECT_TYP * tst_vect_p // Pointer to structure of test-vector data
);
/*****************************************************************************/
void init_test_generation_data( // Initialise structure for test generation data
	const MSG_CONFIG_TYP * inp_msg_cfg_arr, // Array all Midi-message configuration data
	TST_GEN_TYP * tst_gen_p, // Pointer to structure of test generation data
	S32_T inp_seed // Inmput seed value for random number generator
);
/*****************************************************************************/
S32_T gen_test_vector_id( // Generates a test vector identifier
	TST_GEN_TYP * tst_gen_p // Pointer to structure of test generation data
); // Returns a test-vector identifier
/*****************************************************************************/
void gen_test_vector_data( // Generate a test-vector of data for creating next Midi-message 
	TST_GEN_TYP * tst_gen_p, // Pointer to test generation data 
	TST_VECT_TYP * vect_p // Pointer to test-vector data 
);
/*****************************************************************************/
MSG_CONFIG_TYP * status_to_config( // Finds the configuration data for a status byte
	const MSG_CONFIG_TYP msg_cfg_arr[], // Array of Midi-message configuration data
	U8_T inp_status, // input Status byte
	U8_T byte_1st // first data-byte
);
/*****************************************************************************/
MIDI_MSG_ENUM map_vector_to_message( // Map a test-vector identifier to a Midi-message identifier
	S32_T vect_id // Input test-vector identifier
); // Returns a Midi-message identifier
/*****************************************************************************/
void print_midi_message( // Print a Midi-message as binary digits
	const MSG_CONFIG_TYP msg_cfg_arr[], // Array of Midi-message configuration data
	MIDI_PACKET_TYP * rx_pkt_p, // pointer to Midi-message packet
  std::bitset<8> * bits_p // Pointer to array of 8 bits (used for printing binary)
);
/*****************************************************************************/
bool packet_to_message( // Convert Midi packet to RtMidi message
	MIDI_TX_TYP * midi_tx_p // Pointer to structure of Midi data to transmit 
); // Return status flag
/*****************************************************************************/
bool read_midi_message( // Read a Midi-message from input file
	FILE_CONFIG_TYP * file_cfg_p, // Pointer to structure of file configuration data
	MIDI_PACKET_TYP * packet_p // Pointer to packet of Midi-message data 
); // Returns flag indicating if send message was successful
/*****************************************************************************/
bool write_midi_message( // Write a Midi-message to output file
	FILE_CONFIG_TYP * file_cfg_p, // Pointer to structure of file configuration data
	MIDI_PACKET_TYP * packet_p // Pointer to packet of Midi-message data 
); // Returns flag indicating if write message was successful
/*****************************************************************************/
#endif /* XMIDI_COMMON_H */
