//*****************************************//
//  xmidi_common.cpp
//
//  Simple program to test MIDI sysex sending and receiving.
//
//*****************************************//

#include "RtMidi.h"
#include "Socket_lite.h"
#include "xmidi_common.h"
#include "xmidi_check.h"
#include "xmidi_generate.h"
#include "xmidi_globals.h"

#define DBG_HAX 0

/*****************************************************************************/
static void usage( // Issue usage instructions
	void 
) 
{
  std::cout << "\nUsage: <executable-name> [-x] [-fFILENAME] [N] \n";
  std::cout << "\n   N : Number of tests (per channel).   N > 0 \n";
	std::cout << "\n  -aXX.XX.XX.XX : XX.XX.XX.XX is IP Address of Target PC. If absent Loop-Back test assumed \n";
	std::cout << "\n  -i : Use Input file \n";
	std::cout << "\n  -o : Use Output file \n";
	std::cout << "\n  -p : Print to screen \n";
	std::cout << "\n  -t : Add if running on target machine (other-wise host machine assumed) \n";
	std::cout << "\n  -x : Include System Message tests (one per test) \n";
//	std::cout << "\n  -fFILENAME : Supply Midi data file-name FILENAME \n";

  exit( 0 );
} // usage
/*****************************************************************************/
static void read_command_line( // Initialise global data with command-line options
	OPTIONS_TYP * options_p, // Pointer to command-options data structure
	int argc, // Number of command-line arguments
	char * argv[] // Array of command-line arguments pointers
)
{
	S32_T arg_cnt; // command-line argument counter
	char * arg_p; // pointer to command-line argument
	char arg_opt; // argument option
	bool num_found = false; // Clear Numerical argument
	bool bad_args = false; // Clear bad arguments flag


  // Minimal command-line check.
  if ( argc < 2 )
	{
		printf("ERROR: Too few arguments\n" );
		bad_args = true; // Set bad arguments flag
  } // if ( argc < 2 )
	else
	{
	  if ( argc > 7 )
		{
			printf("ERROR: Too many arguments\n" );
			bad_args = true; // Set bad arguments flag
	  } // if ( argc < 7 )
  } // else !( argc < 7 )

	// Initialise configuration data to default values

	// Loop through input options
	for (arg_cnt=1; arg_cnt<argc; arg_cnt++)
	{
		arg_p = argv[arg_cnt];

		// Check if command-option
		if ('-' == arg_p[0])
		{
			arg_opt = arg_p[1]; // Get command option

			switch(arg_opt)
			{
				case 'a':
					arg_p += 2; // Point to file-name string

					// Check string length
					if (STR_NAME_LEN <= strlen(arg_p))
					{
						printf("ERROR: IP Address '%s' too large, possibly malformed\n" ,arg_p );
						printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
						exit(-1);
			} // if (STR_NAME_LEN <= strlen(arg_p)) 
				
					strcpy( options_p->ip_addr ,arg_p );
					options_p->loopback = false; // Switch Off Loop-Back test
				break; // case 'x':

				case 's':
					arg_p += 2; // Point to seed value

					options_p->seed = (U32_T)atoi( arg_p ); // Assign seed value
	
					if (options_p->seed < 0)
					{ 
						printf("ERROR: Found negative Seed of %d \n" ,options_p->seed );
						bad_args = true; // Set bad arguments flag
					} // if (options_p->num_vects < 1)
				break; // case 'x':

				case 'i':
					if (OUT_FILE != options_p->file_mode)
					{
						options_p->file_mode = INP_FILE;
					} // if (OUT_FILE != options_p->file_mode)
					else
					{
						printf("WARNING: Prior assignment of Output file over-rides Input file\n");
					} // else !(OUT_FILE != options_p->file_mode)
				break; // case 'i':

				case 'o':
					if (INP_FILE != options_p->file_mode)
					{
						options_p->file_mode = OUT_FILE;
					} // if (INP_FILE != options_p->file_mode)
					else
					{
						printf("WARNING: Prior assignment of Input file over-rides Output file\n");
					} // else !(INP_FILE != options_p->file_mode)
				break; // case 'o':

				case 't':
					options_p->on_host = false; // executable is being run on target-machine (NOT host-machine) 
				break; // case 'x':

				case 'p':
					options_p->print = true; // Print to screen
				break; // case 'p':

#ifdef MB // Not currently supported
				case 'f':
					arg_p += 2; // Point to file-name string

					// Check string length
					if (STR_NAME_LEN <= strlen(arg_p))
					{
						printf("ERROR: File name '%s' too large, Reduce name, or increase STR_NAME_LEN\n" ,arg_p );
						printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
						exit(-1);
					} // else !(0 < options_p->sysex_num)
					
					strcpy( options_p->file_nam ,arg_p );
				break; // case 'f':
#endif //MB~

				default:
					printf("ERROR: Unknown command option: -%c\n" ,arg_opt );
					bad_args = true; // Set bad arguments flag
				break; // default
			} // switch(arg_opt)
		} // if (arg_p[0] == '-')
		else
		{ // Assume Test number argument
			// Check inumerical argument already found
			if (num_found)
			{
				printf("ERROR: Too many 'No-of-tests' fields: %s\n" ,arg_p );
				bad_args = true; // Set bad arguments flag
			} // if (num_found)
			else
			{
			  options_p->num_vects = (U32_T)atoi( arg_p ); // Assign message length
	
				if (options_p->num_vects < 1)
				{ 
					printf("ERROR: Too few tests: %d\n" ,options_p->num_vects );
					bad_args = true; // Set bad arguments flag
				} // if (options_p->num_vects < 1)

				num_found = true; // Set flag indication Numerical field found
			} // else !(num_found)
		} // else !(arg_p[0] == '-')
	} // for arg_cnt

// cludge = options_p->num_vects; //MB~
// options_p->num_vects = 1; //MB~

	if (NO_FILE == options_p->file_mode)
	{
		printf("ERROR: Missing File mode, choose -o for Output,  or -i for Input\n" );
		bad_args = true; // Set bad arguments flag
	} // if (NO_FILE == options_p->file_mode)

	// Check for bad command-line arguments
	if (bad_args)
	{
		usage();
	} // if (bad_args)
} // read_command_line_options
/*****************************************************************************/
static MSG_CONFIG_TYP configure_one_midi_message( // Initialisestructure of Midi message configuration data
	MIDI_MSG_ENUM msg_id, // Midi-message identifier
	const char * str_p, // Pointer to input string
	U8_T inp_status, // input status value
	S32_T num_bytes // Number of occurances of input status value
) // Returns structure of Midi-message configuration data
{
	MSG_CONFIG_TYP msg_config_s; // Structure of status data


	// Check string length
	if (STR_MIDI_LEN <= strlen( str_p ))
	{
		printf("ERROR: Message name '%s' too large, Reduce name, or increase STR_MIDI_LEN\n" ,str_p );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // else !(0 < gener_p->sysex_num)

	strcpy( msg_config_s.name ,str_p );

	msg_config_s.id = msg_id; // Assign  Midi-message identifier
	msg_config_s.status = inp_status; // Assign status value
	msg_config_s.len = num_bytes; // Assign number of data bytes

	return msg_config_s;
} // configure_one_midi_message
/*****************************************************************************/
static void configure_all_midi_message_data( // Configure array of all Midi message data
	GLOBAL_TYP * glob_p // Pointer to structure of global data
)
{
	// Configure Channel-message data
	
	glob_p->msg_cfg_arr[PROG_CHANGE_ID] = configure_one_midi_message( PROG_CHANGE_ID ,"Program-Change" ,PROG_CHANGE_STS ,1 );
	glob_p->msg_cfg_arr[CHAN_PRESS_ID] = configure_one_midi_message( CHAN_PRESS_ID ,"Channel-Pressure" ,CHAN_PRESS_STS ,1 );
	glob_p->msg_cfg_arr[NOTE_ON_ID] = configure_one_midi_message( NOTE_ON_ID ,"Note-On" ,NOTE_ON_STS ,2 );
	glob_p->msg_cfg_arr[NOTE_OFF_ID] = configure_one_midi_message( NOTE_OFF_ID ,"Note-Off" ,NOTE_OFF_STS ,2 ); // 
	glob_p->msg_cfg_arr[KEY_PRESS_ID] = configure_one_midi_message( KEY_PRESS_ID ,"Key-Pressure" ,KEY_PRESS_STS ,2 );
	glob_p->msg_cfg_arr[PITCH_WHEEL_ID] = configure_one_midi_message( PITCH_WHEEL_ID ,"Pitch-Wheel" ,PITCH_WHEEL_STS ,2 );
	glob_p->msg_cfg_arr[CONTROL_CHANGE_ID] = configure_one_midi_message( CONTROL_CHANGE_ID ,"Control-Change" ,CONTROL_CHANGE_STS ,2 );

	// Configure System-message data
	glob_p->msg_cfg_arr[TUNE_REQ_ID] = configure_one_midi_message( TUNE_REQ_ID ,"Tune_Request" ,TUNE_REQ_STS ,0 );
	glob_p->msg_cfg_arr[MIDI_TIME_ID] = configure_one_midi_message( MIDI_TIME_ID ,"Midi-Time-Code" ,MIDI_TIME_STS ,1 );
	glob_p->msg_cfg_arr[SONG_SEL_ID] = configure_one_midi_message( SONG_SEL_ID ,"Song-Select" ,SONG_SEL_STS ,1 );
	glob_p->msg_cfg_arr[SONG_PTR_ID] = configure_one_midi_message( SONG_PTR_ID ,"Song-Position" ,SONG_PTR_STS ,2 );
	glob_p->msg_cfg_arr[SMALL_SYSEX_ID] = configure_one_midi_message( SMALL_SYSEX_ID ,"Small-SysEx" ,STRT_SYSEX_STS ,UNDEFINED ); // Variable length
	glob_p->msg_cfg_arr[LARGE_SYSEX_ID] = configure_one_midi_message( LARGE_SYSEX_ID ,"Large-SysEx" ,STRT_SYSEX_STS ,UNDEFINED ); // Variable length

	// Configure Real-Time message data
	glob_p->msg_cfg_arr[START_SONG_ID] = configure_one_midi_message( START_SONG_ID ,"Start-Song" ,START_SONG_STS ,0 );
	glob_p->msg_cfg_arr[CONT_SONG_ID] = configure_one_midi_message( CONT_SONG_ID ,"Continue-Song" ,CONT_SONG_STS ,0 );
	glob_p->msg_cfg_arr[STOP_SONG_ID] = configure_one_midi_message( STOP_SONG_ID ,"Stop-Song" ,STOP_SONG_STS ,0 );

} // configure_all_midi_message_data
/*****************************************************************************/
void init_test_generation_data( // Initialise structure for test generation data
	const MSG_CONFIG_TYP * inp_msg_cfg_arr, // Array all Midi-message configuration data
	TST_GEN_TYP * tst_gen_p, // Pointer to structure of test generation data
	S32_T inp_seed // Inmput seed value for random number generator
)
{
	tst_gen_p->cnt = 0; // Clear number of test-vector identifiers generated
	tst_gen_p->msg_cfg_arr = inp_msg_cfg_arr; // Set pointer to array all Midi-message configuration data
	tst_gen_p->rand = (U32_T)inp_seed; // Initialise internal 32-bit random number from input seed value

	// Check if random test-vectors are required
	if (inp_seed)
	{
		tst_gen_p->on = true; // Switch ON random number generator
	} // if (inp_seed)
	else
	{
		tst_gen_p->on = false; // Switch OFF random number generator
	} // if (inp_seed)
} // init_test_generation_data
/*****************************************************************************/
void init_common_data( // Initialise common data
	GLOBAL_TYP * glob_p // Pointer to structure of global data
) // Return pointer to structure of file configuration data
{
	FILE * file_port_p; // Pointer to input file with Ethernet Port Number
	char file_port_name[STR_NAME_LEN]; // Name of input file containing Ethernet Port Number


	configure_all_midi_message_data( glob_p );	// Configure array of all Midi message data

	// Assign valid Midi-port names
	strcpy( glob_p->port_names[USB2MIDI_ALESIS] ,"USB2MIDI" ); //  USB-Audio <--Alesis--> Midi (Din-Plug) Cables (In & Out)
	strcpy( glob_p->port_names[USB2MIDI_CHINA] ,"USB Midi Cable" ); //  USB-Audio <--Chinese--> Midi (Din-Plug) Cables (In & Out)
	strcpy( glob_p->port_names[USB2MIDI_EMU] ,"E-MU XMidi1X1" ); //  USB-Audio <--EMU--> Midi (Din-Plug) Cables (In & Out)

#if defined(__WINDOWS_MM__)
	// The Windows Thesycon driver registers its own name rather than the xCORE name
	strcpy( glob_p->port_names[XCOREUSB] ,"XMOS MIDI" ); // USB-Audio <--> xCORE
	strcpy( file_port_name ,"midi_inputs.dir\\port_number.txt" ); // Assign file-name containing Ethernet port No.
#else //defined(__WINDOWS_MM__)
	// Linux & Mac drivers register the xCORE name
	strcpy( glob_p->port_names[XCOREUSB] ,"xCORE USB Audio" ); // USB-Audio <--> xCORE
	strcpy( file_port_name ,"midi_inputs.dir/port_number.txt" ); // Assign file-name containing Ethernet port No.
#endif //!defined(__WINDOWS_MM__)

	// Initialise socket data 
	glob_p->sckt_p = NULL; // Clear pointer to Socket object for Ethernet synchronisation between platforms
	glob_p->sckt_id = 0; // Clear Unique socket identifier

	file_port_p = fopen( file_port_name ,"r" );	// Open input file
	// Check file-open succeeded
	if (NULL == file_port_p)
	{
		printf("ERROR: Failed to open file '%s' \n" ,file_port_name );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (NULL == file_cfg_p->ptr)

	fscanf( file_port_p ,"%s" ,glob_p->port_num_str ); // Read port-number string
	fclose( file_port_p );	// Close port_number file

} // init_common_data
/*****************************************************************************/
void init_midi_rx_data( // Initialise structure for received midi data
	MIDI_RX_TYP * midi_rx_p // Pointer to structure of received Midi data 
)
{
  midi_rx_p->midi_in_p = 0; // Clear pointer to RtMidi-In Object
} // init_midi_rx_data
/*****************************************************************************/
void init_test_vector_data( // Initialise structure of test-vector data
	TST_VECT_TYP * tst_vect_p // Pointer to structure of test-vector data
)
{
	tst_vect_p->sts = 0; // Clear Status Byte
	tst_vect_p->param = 0; // Clear channel/data parameter
	tst_vect_p->packet_p = NULL; // Pointer to packet of Midi-message data
	tst_vect_p->msg_cfg_p = NULL; // Pointer to Midi-message configuration data
} // init_test_vector_data
/*****************************************************************************/
void init_midi_tx_data( // Initialise structure for transmitted midi data
	MIDI_TX_TYP * midi_tx_p // Pointer to structure of Midi data to transmit 
)
{
  midi_tx_p->midi_out_p = 0; // Clear pointer to RtMidi-Out Object
  midi_tx_p->burst_tx_cnt = 0;	// Clear Midi-message write counter
  midi_tx_p->pace_time = DEF_PACE_TIME;	// Clear Midi-message write counter
  midi_tx_p->port_id = NUM_VALID_PORTS; // Set to impossible value
} // init_midi_tx_data
/*****************************************************************************/
FILE_CONFIG_TYP * init_file_config_data( // Initialise file configuration data
	GLOBAL_TYP * glob_p // Pointer to structure of global data
) // Return pointer to structure of file configuration data
{
	FILE_CONFIG_TYP * file_cfg_p = &(glob_p->file_cfg_s); // Get pointer to structure of file configuration data
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Get pointer to command-options data structure
	char err_str[STR_NAME_LEN]; // String for (part of) error message


	file_cfg_p->mode = options_p->file_mode;  // Initialise file mode

	// Assign file name prefix
	if ((RECEIVE == glob_p->config_id) && (OUT_FILE == file_cfg_p->mode))
	{ // Create 'results' file-name
		// Check if default file-name required
		if (0 == strlen(options_p->file_nam))
		{ // Empty file-name
#if defined(__WINDOWS_MM__)
			sprintf( file_cfg_p->name ,"midi_results.dir/midi_win_n%03d_s%d.dat" ,options_p->num_vects ,options_p->seed );
#else //defined(__WINDOWS_MM__)
		#if defined(__MACOSX_CORE__)
			sprintf( file_cfg_p->name ,"midi_results.dir/midi_mac_n%03d_s%d.dat" ,options_p->num_vects ,options_p->seed );
#else // defined(__MACOSX_CORE__)
			// Linux Variants
			sprintf( file_cfg_p->name ,"midi_results.dir/midi_lnx_n%03d_s%d.dat" ,options_p->num_vects ,options_p->seed );
		#endif // !defined(__MACOSX_CORE__)
#endif //!defined(__WINDOWS_MM__)
		} // if (0 == strlen(options_p->file_nam))
		else
		{ // Use file_name supplied on command line
			strcpy( file_cfg_p->name ,options_p->file_nam ); // Initialise filename
		} // else !(0 == strlen(options_p->file_nam))
	} // if ((RECEIVE == glob_p->config_id) && (OUT_FILE == file_cfg_p->mode))
	else
	{ // Create 'Generated Tests' file-name
#if defined(__WINDOWS_MM__)
		sprintf( file_cfg_p->name ,"midi_inputs.dir/midi_gen_n%03d_s%d.dat" ,options_p->num_vects ,options_p->seed );
#else //defined(__WINDOWS_MM__)
		sprintf( file_cfg_p->name ,"midi_inputs.dir/midi_gen_n%03d_s%d.dat" ,options_p->num_vects ,options_p->seed );
#endif //!defined(__WINDOWS_MM__)
	} // else !((RECEIVE == glob_p->config_id) && (OUT_FILE == file_cfg_p->mode))

	// Determine mode of opening file
	switch(file_cfg_p->mode)
	{ 
		case NO_FILE : 
			return NULL; // Nothing to do
		break; // case NO_FILE

		case INP_FILE :
			file_cfg_p->ptr = fopen( file_cfg_p->name ,"rb" );	// Open input file
			strcpy( err_str ,"reading");
		break; // case INP_FILE

		case OUT_FILE :
			file_cfg_p->ptr = fopen( file_cfg_p->name ,"wb" );	// Open output file
			strcpy( err_str ,"writing");
		break; // case INP_FILE

		default: 
			printf("ERROR: Unsupported File-Mode of %d requested\n" ,file_cfg_p->mode );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(file_cfg_p->mode)

	// Check file-open succeeded
	if (NULL == file_cfg_p->ptr)
	{
		printf("ERROR: Failed to open file '%s' for %s\n" ,file_cfg_p->name ,err_str );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (NULL == file_cfg_p->ptr)

	return file_cfg_p; // Return pointer to configured data structure
} // init_file_config_data
/*****************************************************************************/
void init_command_options( // Initialise command-options data
	OPTIONS_TYP * options_p, // Pointer to command-options data structure
	int argc, // Number of command-line arguments
	char * argv[] // Array of command-line arguments pointers
)
{
	// Initialise configuration data to default values

	strcpy( options_p->file_nam ,"" ); // Initialise file-name to empty string
	strcpy( options_p->ip_addr ,"" ); // Initialise IP Address to empty string
	options_p->seed = DEF_SEED;
	options_p->print = DEF_PRINT;
	options_p->num_vects = DEF_TESTS;
	options_p->file_mode = NO_FILE;
	options_p->loopback = true; // Default is loop-back test
	options_p->on_host = true; // Default assumes executable is being run on host-machine (NOT target machine) 

	// Update configuration data with command-line options
	read_command_line( options_p ,argc ,argv );
} // init_command_options
/*****************************************************************************/
static void init_bind_socket( // Initialise socket by binding target IP address to port number
	GLOBAL_TYP * glob_p, // Pointer to structure of global data
	OPTIONS_TYP * options_p // Pointer to command-options data structure
)
{
	// Bind the IP Address of this machine (Server) to the supplied Ethernet port number 
	if (0 != glob_p->sckt_p->socBind( options_p->ip_addr ,glob_p->port_num_str ))
	{ // Bind failed, Try closing port and re-opening

		delete glob_p->sckt_p; // Close old Socket

		glob_p->sckt_p = new Socket(); // Get pointer to new Socket Object

		if (0 != glob_p->sckt_p->socBind( options_p->ip_addr ,glob_p->port_num_str ))
		{ // Bind failed twice, Abort
			printf("ERROR: Bind failed twice Aborting\n" );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		} // if (0 != glob_p->sckt_p->socBind( options_p->ip_addr ,glob_p->port_num_str ))
	} // if (0 != glob_p->sckt_p->socBind( options_p->ip_addr ,glob_p->port_num_str ))

	if (options_p->print) printf("Waiting for Socket to be opened\n");
	glob_p->sckt_p->socListen(); // Wait for Client socket set-up

	glob_p->sckt_p->socAccept(); // Get unique socket identifier
} // init_bind_socket
/*****************************************************************************/
static void init_connect_socket( // Initialise socket for connecting to remote target machine
	GLOBAL_TYP * glob_p, // Pointer to structure of global data
	OPTIONS_TYP * options_p // Pointer to command-options data structure
)
{
	S32_T fail_cnt = 0; // Counts number of connection failures
	bool failed = true; // Flag set if failed to connect


	SLEEP_FN( 10 ); // Wait for spurious Midi-messages in pipe-line data to empty?

	while (failed && (fail_cnt < 10))
	{
		failed = glob_p->sckt_p->socConnect( options_p->ip_addr ,glob_p->port_num_str ); // IP address of Linux Server

		if (failed)
		{
	    fprintf( stderr ,"ReTrying\n");
			fail_cnt++; // Increment No of connection failures

			SLEEP_FN( 1000 ); // Wait for transmitter to set-up port
		} // if (failed)
	} // while (failed && (fail_cnt < 10))

	if (failed)
	{
		fprintf( stderr ,"ERROR: Persistently failed to Connect to Port. Aborting\n");
		exit(-1);
	} // if (failed)

} // init_connect_socket
/*****************************************************************************/
void init_socket( // Initialise socket
	GLOBAL_TYP * glob_p // Pointer to structure of global data
)
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Pointer to command-options data structure
	glob_p->sckt_p = new Socket(); // Get pointer to Socket Object


	if (0 == strlen(options_p->ip_addr))
	{
		printf("ERROR: Zero-length IP Address found \n" );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (0 == strlen(options_p->ip_addr))

  // Determine if on host machine
	if (options_p->on_host)
	{ // We are running on host-machine
		init_connect_socket( glob_p ,options_p ); // Connect to remote target machine via port number
	} // if (options_p->on_host)
	else
	{ // We are running on target-machine
		init_bind_socket( glob_p ,options_p ); // Bind target IP address to port number
	} // else !(options_p->on_host)

} // init_socket
/*****************************************************************************/
VALID_PORT_ENUM open_midi_port( // Opens a Midi port
	RtMidi * rtmidi_p, // Pointer to RtMidi object
	const char port_names[NUM_VALID_PORTS][STR_NAME_LEN], // Array of valid port names
	const char * flow_str, // String holding flow direction (Input or Output)  
	bool do_print // Flag set if screen-printing required
) // Returns port identifier if successful, or NUM_VALID_PORTS for failure
{
	U32_T nPorts = rtmidi_p->getPortCount(); // Get number of ports
	std::string curr_portName; // Name of current port
	char * sub_str; // Pointer to sub-string (used for Port-name matching)
 	U32_T port_cnt = 0; // port counter
 	VALID_PORT_ENUM name_cnt; // counter for valid port names


	// Check for avaibale ports
	if (0 == nPorts)
	{
		std::cout << "ERROR: No " << flow_str << " ports available!" << std::endl;
		return NUM_VALID_PORTS;
	} // if (0 == nPorts)

	// Search for target port
	for (port_cnt=0; port_cnt<nPorts; port_cnt++)
	{
		curr_portName = rtmidi_p->getPortName(port_cnt); // Get name of current port
// printf("%d:Detected Port= %s\n" ,port_cnt ,curr_portName.c_str() ); //MB~

		name_cnt = (VALID_PORT_ENUM)0; // counter for valid port names

		while (name_cnt < NUM_VALID_PORTS)
		{ // Search for valid S/W portname in H/W port-name
			sub_str = (char *)strstr( (const char *)curr_portName.c_str() ,(const char *)port_names[name_cnt] );
// printf("%d:Valid Port = %s\n" ,name_cnt ,port_names[name_cnt] ); //MB~

			// Check if target port found
			if (NULL != sub_str)
			{ // target port found
		    if (do_print) std::cout << "Opening " << flow_str << " port "<< curr_portName << std::endl;

			  rtmidi_p->openPort(port_cnt);

				return name_cnt;
			} // if (NULL != sub_str)

			name_cnt = (VALID_PORT_ENUM)((int)name_cnt + 1); // Move to next valid port-name
		} // while (name_cnt < NUM_VALID_PORTS)
	} // for port_cnt

	// If we reach here, then NO valid port found"
	std::cout << "ERROR: Target " << flow_str << " port " << " NOT found" << std::endl;
	std::cout << "       Candidates are:-" << std::endl;

	// Search for target port
	for (port_cnt=0; port_cnt<nPorts; port_cnt++)
	{
		curr_portName = rtmidi_p->getPortName(port_cnt); // Get name of current port
  	std::cout << flow_str << " port "<< curr_portName << std::endl;
	} // for port_cnt

	return NUM_VALID_PORTS;
} // open_midi_port 
/*****************************************************************************/
static U32_T gen_random_test_vector( // Generates a pseudo-random test vector from the supplied seed
	TST_GEN_TYP * tst_gen_p // Pointer to structure of test generation data
) // Returns a test-vector identifier
{
	S32_T vect_id; // test-vector identifier


	// generate new 32-bit random number
	tst_gen_p->rand = (tst_gen_p->rand * (U32_T)1103515245) + (U32_T)12345;

	// Extract Most-significant bits to use as random test-vector id
	vect_id = (S32_T)((tst_gen_p->rand >> SHIFT_I_BITS) & MAX_RND); 

	return vect_id;
} // gen_random_test_vector
/*****************************************************************************/
S32_T gen_test_vector_id( // Generates a test vector identifier
	TST_GEN_TYP * tst_gen_p // Pointer to structure of test generation data
) // Returns a test-vector identifier
{
	S32_T vect_id; // test-vector identifier


	// Determine if random test-vectors required
	if (tst_gen_p->on)
	{ // Random test-vectors required
		vect_id = gen_random_test_vector( tst_gen_p ); // generate a psuedo-random vector id
	} // if (tst_gen_p->on)
	else
	{ // Random test-vectors NOT required
		vect_id = tst_gen_p->cnt + VECT_OFF; // vector id is derived linearly from vector cnt
	} // if (tst_gen_p->on)

	tst_gen_p->cnt++; // Inrement No. of test-vector identifiers generated

	return vect_id;
} // gen_test_vector_id
/*****************************************************************************/
MIDI_MSG_ENUM map_vector_to_message( // Map a test-vector identifier to a Midi-message identifier
	S32_T vect_id // Input test-vector identifier
) // Returns a Midi-message identifier
/*
 * This function maps a Midi-message value, currently in the range [0..63]
 * Into a High-level Midi-message category, one of [Channel, System-Common, System Real-time]
 * This mapping is used to weight the number of tests allocated to eech Midi-message category
 *
 * Currently, the weighting for each category are [7, 2, 1]
 * E.g. a Note-On message (Channel) is tested 3.5 times more frequently than a Song-Select message (System-common)
 *
 * The Midi-message value is encoded in the LS-bits of the Test-vector identifier (currently 6 bits)
 *
 * Currently, the Midi-message category assignments are as follows:-
 *
 * 	0..48	Channel Messages (7 * 7).
 * 49..58	System-common Messages (2 * 5).
 * 59..63 Song Select. (1 * 5)
 */
{
	U32_T msg_map_val = vect_id & MSG_MAP_MASK; // Extract 6 LS-bits containing Midi-message mapping value
	U32_T msg_id = 0; // Initialise Midi-message Identifier to zero


	// Determine high-level Midi-test category ...


	// Determine if channel message
	if (CHAN_MAP_RNG > msg_map_val)
	{	// Channel Midi-message
		msg_id += (msg_map_val / FREQ_CHAN_TYPS); // Generate Midi-message identifier

		return (MIDI_MSG_ENUM)msg_id; // Return decoded Midi-message identifier
	} // if (CHAN_MAP_RNG <= msg_map_val)

	msg_map_val -= CHAN_MAP_RNG; // Remove channel-message mapping range
	msg_id += NUM_CHAN_TYPS; // Add channel-message identifier range

	// Determine if system-common message
	if (SYS_MAP_RNG > msg_map_val)
	{	// System-common Midi-message
		msg_id += (msg_map_val / FREQ_SYS_TYPS); // Generate Midi-message identifier

		return (MIDI_MSG_ENUM)msg_id; // Return decoded Midi-message identifier
	} // if (SYS_MAP_RNG <= msg_map_val)

	msg_map_val -= SYS_MAP_RNG; // Remove System-common message mapping range
	msg_id += NUM_SYS_TYPS; // Add System-common identifier range

	// Determine if system real-time message
	if (REAL_MAP_RNG > msg_map_val)
	{	// System real-time Midi-message
		msg_id += (msg_map_val / FREQ_REAL_TYPS); // Generate Midi-message identifier

		return (MIDI_MSG_ENUM)msg_id; // Return decoded Midi-message identifier
	} // if (SYS_MAP_RNG <= msg_map_val)

	msg_map_val -= REAL_MAP_RNG; // Remove system real-time messagemapping range
	msg_id += NUM_REAL_TYPS; // Add system real-time identifier range

	printf("ERROR: test-vector mapping NOT decoded correctly");
	printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
	exit(-1);

	return (MIDI_MSG_ENUM)0;
} // map_vector_to_message
/*****************************************************************************/
void gen_test_vector_data( // Generate a test-vector of data for creating next Midi-message 
	TST_GEN_TYP * tst_gen_p, // Pointer to test generation data 
	TST_VECT_TYP * tst_vect_p // Pointer to test-vector data 
)
{
	MSG_CONFIG_TYP * msg_config_p; // Pointer to structure of Midi-message configuration data
	MIDI_MSG_ENUM msg_id; // Midi-message type/status identifier
	S32_T vect_id; // test-vector identifier
	U8_T status_id; // Status identifier
	U32_T param_id; // data parameter identifier


	vect_id = gen_test_vector_id( tst_gen_p ); // Generate random test vector identifier 

	// Get Midi-message identifier from 6 LS-bits of test-vector identifier
	msg_id = map_vector_to_message( vect_id );

	param_id = (vect_id >> MSG_MAP_BITS); // Preset parameter identifier with remaining 18 MS-bits of test-vector id
	msg_config_p = (MSG_CONFIG_TYP *)&(tst_gen_p->msg_cfg_arr[msg_id]);	// Point to Configuration data for current Midi-message
	status_id = msg_config_p->status; // Preset status identifier with configuration status

	// Determine if Channel message
	if (STRT_SYSEX_STS > status_id)
	{ // Channel message
		status_id += (param_id & CHAN_MASK); // Add channel identifier into status byte

		param_id >>= CHAN_BITS; // Remove channel identifier from data parameter
	} // if (STRT_SYSEX_STS > status_id)

	// Determine expected Midi-message length
	switch(	msg_config_p->len )
	{
		case 0 : // NO data bytes
			param_id = 0; // Clear data check value 
		break; // case 0

		case 1 : // 1 data byte
			param_id &= MSK_7BIT; // Mask out 7 LS-bits as data check value 
		break; // case 1

		case 2 : // 2 data bytes
			param_id &= MSK_14BIT; // Mask out 14 LS-bits as data check value 
		break; // case 2

		case UNDEFINED : // Variable No. of data bytes
			// Nothing to do
		break; // case UNDEFINED

		default :
			printf("ERROR: Found unexpected Midi-message data length of %d \n" ,msg_config_p->len );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(	msg_config_p->len )

	tst_vect_p->msg_cfg_p = msg_config_p; // Store pointer to configuration data
	tst_vect_p->sts = status_id; // Store finalised status byte
	tst_vect_p->param = param_id; // Store finalised parameter data

// printf("GEN: Vect_id=%02d Msg_Id=%02d Sts=0x%08X Param=0x%08X " ,vect_id ,msg_id ,tst_vect_p->sts ,tst_vect_p->param ); //MB~

	return; 
} // gen_test_vector_data
/*****************************************************************************/
MSG_CONFIG_TYP * status_to_config( // Finds the configuration data for a status byte
	const MSG_CONFIG_TYP msg_cfg_arr[], // Array of Midi-message configuration data
	U8_T inp_status, // input Status byte
	U8_T byte_1st // first data-byte
)
{
	MSG_CONFIG_TYP * msg_cfg_p = (MSG_CONFIG_TYP *)&(msg_cfg_arr[0]); // Point to first element of configuration data
	U8_T sts_byt = inp_status; // Initialise status-byte to input Status
	int msg_cnt; // Midi-message type counter


	// Determine if a channel status byte
	if (STRT_SYSEX_STS > sts_byt)
	{ // Channel status byte
		sts_byt >>= CHAN_BITS;  // Remove Channel bits 
		sts_byt <<= CHAN_BITS;  // Pad LS-bits with zero
	} // if (STRT_SYSEX_STS > sts_byt)

	// loop through array of configuration data
	for (msg_cnt=0; msg_cnt<NUM_MIDI_MSGS; msg_cnt++)
	{
		if (sts_byt == msg_cfg_p->status)
		{
			// Check for Special-case of System-exclusive status
			if (STRT_SYSEX_STS == sts_byt)
			{ // System-exclusive
				
				// Determine if 'Small' or 'Large' SysEx Message
				if (0 == byte_1st)
				{ // 'Large' SysEx message
					return (MSG_CONFIG_TYP *)&(msg_cfg_arr[LARGE_SYSEX_ID]); // Return pointer to 'Large' SysEx configutation data
				} // if (0 == byte_1st)
				else
				{ // 'Small' SysEx message
					return (MSG_CONFIG_TYP *)&(msg_cfg_arr[SMALL_SYSEX_ID]); // Return pointer to 'Small' SysEx configutation data
				} // if (0 == byte_1st)
			} // if (STRT_SYSEX_STS == sts_byt)
			else
			{ // NOT System-exclusive
				 return msg_cfg_p;  // Return pointer to current configutation data
			} // if (STRT_SYSEX_STS == sts_byt)
		} // if (sts_byt == msg_cfg_p->status)

		msg_cfg_p++; // Increment pointer to next element in array of configuration data
	} // for msg_cnt

	return NULL; // Return NULL if Invalid Status byte
} // status_to_config
/*****************************************************************************/
void print_midi_message( // Print a Midi-message as binary digits
	const MSG_CONFIG_TYP msg_cfg_arr[], // Array of Midi-message configuration data
	MIDI_PACKET_TYP * rx_pkt_p, // pointer to Midi-message packet
  std::bitset<8> * bits_p // Pointer to array of 8 bits (used for printing binary)
)
{
	MSG_CONFIG_TYP * msg_cfg_p; // Pointer to Midi-message configuration data
	S32_T byte_cnt; // Byte counter
	S32_T num_bytes = rx_pkt_p->data_len; // Number of data bytes


	// Check fo Missing Message
	if (MISSING_STS == rx_pkt_p->midi_status)
	{
	  std::cout << " ERROR: MISSING MESSAGE" << std::endl;

		return; // Early return
	} // if (MISSING_STS == rx_pkt_p->midi_status)

	// print status byte
	*bits_p = (int)rx_pkt_p->midi_status; // Convert to binary digits
	std::cout << " " << *bits_p;

	// Check for corrupted message-length
	if (num_bytes > MAX_SYSEX_LEN)
	{	// Message-length unexpectedly long
		num_bytes = MAX_SYSEX_LEN; // Terminate message-printing early
	} // if (num_bytes > MAX_SYSEX_LEN)

	// Loop through any data bytes
	for ( byte_cnt=0; byte_cnt<num_bytes; byte_cnt++ )
	{
		*bits_p = (int)rx_pkt_p->midi_data[byte_cnt]; // Convert to binary digits
	  std::cout << " " << *bits_p;
	} // for byte_cnt

	// Check for early termination of message-printing
	if (num_bytes == MAX_SYSEX_LEN)
	{
	  std::cout << " ABORT";
	} // if (num_bytes == MAX_SYSEX_LEN)

	// Get pointer to Midi-message configuration data
	msg_cfg_p = status_to_config( msg_cfg_arr ,rx_pkt_p->midi_status ,rx_pkt_p->midi_data[0] );

	// Check for valid configuration data
	if (NULL == msg_cfg_p)
	{ // Invalid configuration data
		printf(" UNSUPPORTED Configuration" );
	} // if (NULL == msg_cfg_p)
	else
	{ // Valid configuration data
		printf(" %s" ,msg_cfg_p->name );
	} // if (NULL == msg_cfg_p)

  std::cout << std::endl;

} // print_midi_message
/*****************************************************************************/
static bool send_message( // Wrapper for RtMidi sendmessge()
	MIDI_TX_TYP * midi_tx_p // Pointer to structure of Midi data to transmit 
)
{
  std::vector<U8_T> * msg_p = &(midi_tx_p->msg_s); // Local pointer to Array of bytes to hold message


  try 
	{
		midi_tx_p->midi_out_p->sendMessage( msg_p ); // Call RtMidi function
  } // try
  catch ( RtMidiError &error ) 
	{
		printf("\nERROR: Failed to successfully send RtMidi message\n");
    error.printMessage();
    return false;
  } // catch

  return true;
} // send_message
/*****************************************************************************/
bool packet_to_message( // Convert Midi packet to RtMidi message
	GENER_TYP * gener_p // Pointer to structure of test-vector generation data
) // Return status flag
{
	MIDI_TX_TYP * midi_tx_p = &(gener_p->tx_data_s); // Get pointer to structure of Midi data to transmit
	MIDI_PACKET_TYP * packet_p = &(midi_tx_p->packet_s); // Packet of Midi-message data 
  std::vector<U8_T> * msg_p = &(midi_tx_p->msg_s); // Local pointer to Array of bytes to hold message
	S32_T byte_cnt; // Midi data byte counter



	msg_p->clear();
  msg_p->push_back( packet_p->midi_status );

	for (byte_cnt=0; byte_cnt<packet_p->data_len; byte_cnt++)
	{ 	
	  msg_p->push_back( packet_p->midi_data[byte_cnt] );
	} // for byte_cnt

	if (gener_p->print)
	{
		printf("%1d:" ,packet_p->data_len ); //MB~
		print_midi_message( gener_p->msg_cfg_arr_p ,packet_p ,&(midi_tx_p->bits) ); // Print Midi-message as binary digits 
	} // if (gener_p->print)

  midi_tx_p->burst_tx_cnt++; // Increment message write counter

	if (false == send_message( midi_tx_p ) )
	{
		printf("\nERROR: Failed to send all test bytes\n");
		return false; // Send message id 
	} // if (false == send_message( midi_tx_p ) )

	// As Midi data rate is 3125 Bytes/sec. Each take approximately 1 ms.
	SLEEP_FN( midi_tx_p->pace_time ); // Pace transmission of Midi-data

	return true;
} // packet_to_message
/*****************************************************************************/
static U8_T read_file_byte( // Read one byte from file
	FILE_CONFIG_TYP * file_cfg_p // Pointer to structure of file configuration data
) // Returns byte read from file
/*
 * fread does NOT return end-of-file until a read is made beyond the end-of-file
 * Therefore we have to do a read that will fail, to detect the end-of-file
 */
{
  U8_T out_byt = 0; // output byte read from file
	int items_req = 1; // Request one item at a time from binary file
	int items_read; // No. of items actually retrieved from file 


	items_read = fread( &out_byt ,sizeof(U8_T) ,items_req ,file_cfg_p->ptr ); // Get one byte

	// Test for correct No. of items retrieved
	if (items_read < items_req)
	{
		if (0 == feof(file_cfg_p->ptr))
		{
			printf("ERROR: Failed to read byte from file '%s'\n" ,file_cfg_p->name );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		} // if (0 == feof(file_cfg_p->ptr))
	} // if (items_read < items_req)

	return out_byt;
} // read_file_byte 
/*****************************************************************************/
bool read_midi_message( // Read a Midi-message from input file
	FILE_CONFIG_TYP * file_cfg_p, // Pointer to structure of file configuration data
	MIDI_PACKET_TYP * packet_p // Pointer to packet of Midi-message data 
) // Returns flag indicating if send message was successful
{
  U8_T msg_len; // Midi-message length
	int byte_cnt = 0; // Byte counter


	msg_len = read_file_byte( file_cfg_p ); // Read Midi-message length byte

	// Test for sensible length
	if (1 > msg_len)
	{
		printf("ERROR: Zero Midi-message length found in file '%s'\n" ,file_cfg_p->name );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (1 > msg_len)

	packet_p->midi_status = read_file_byte( file_cfg_p ); // Read Midi-status byte

	packet_p->data_len = (msg_len - 1); // Assign No. of Midi data bytes

	// Read any data bytes
	for (byte_cnt=0; byte_cnt<packet_p->data_len; byte_cnt++)
	{
	  packet_p->midi_data[byte_cnt] = read_file_byte( file_cfg_p ); // Read Midi-data byte
	} // for byte_cnt

	return true;
} // read_midi_message
/*****************************************************************************/
static bool write_file_byte( // Write one byte to file
	FILE_CONFIG_TYP * file_cfg_p, // Pointer to structure of file configuration data
  const U8_T inp_byt // Input byte to be written to file
) // Returns byte read from file
{
	int items_req = 1; // Request one item at a time be written to the binary output file
	int items_written; // No. of items actually written


	items_written = fwrite( &inp_byt ,sizeof(U8_T) ,items_req ,file_cfg_p->ptr ); // Get one byte

	// Test for correct No. of items written
	if (items_written < items_req)
	{
		printf("ERROR: Failed to write byte 0x%02X to file '%s'\n" ,inp_byt ,file_cfg_p->name );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );

		return false;
	} // if (items_written < items_req)

	return true;
} // write_file_byte 
/*****************************************************************************/
bool write_midi_message( // Write a Midi-message to output file
	FILE_CONFIG_TYP * file_cfg_p, // Pointer to structure of file configuration data
	MIDI_PACKET_TYP * packet_p // Pointer to packet of Midi-message data 
) // Returns flag indicating if write message was successful
{
  U8_T msg_len = (packet_p->data_len + 1); // Midi-message length (including status byte)
	int byte_cnt = 0; // Byte counter


	// Test for sensible length
	if (1 > msg_len)
	{
		printf("ERROR: Zero Midi-message length found while writing to file '%s'\n" ,file_cfg_p->name );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (1 > msg_len)

	if (false == write_file_byte( file_cfg_p ,msg_len )) return false; // Write Midi-message length byte

	if (false == write_file_byte( file_cfg_p ,packet_p->midi_status)) return false; // Write Midi-status byte

	// Write any data bytes
	for (byte_cnt=0; byte_cnt<packet_p->data_len; byte_cnt++)
	{
		if (false == write_file_byte( file_cfg_p ,packet_p->midi_data[byte_cnt] )) return false; // Write Midi-data byte
	} // for byte_cnt

	return true;
} // write_midi_message
/*****************************************************************************/
#if (1 == DBG_HAX)
/*****************************************************************************/
static void dbg_pkt_hacks( // Hack data in packet to test code. e.g. swap bytes
	GENER_TYP * gener_p // Pointer to structure of test-vector generation data
)
{
	MIDI_TX_TYP * midi_tx_p = &(gener_p->tx_data_s); // Get pointer to structure of Midi data to transmit
	MIDI_PACKET_TYP * packet_p = &(midi_tx_p->packet_s); // Packet of Midi-message data 

	
#ifdef MB // Test Missed messages
	if (0xC1 == packet_p->midi_status)
	{
		printf("REMOVED Message 0x%02X \n" ,packet_p->midi_status );
		return; // Return without sending message
	} // if (0xC1 == packet_p->midi_status)
#endif //MB~

#ifdef MB // Test extra messages
if (0xFB == packet_p->midi_status)
{
	packet_to_message( gener_p ); // Send message to Midi-Out port

	// Build extra message
	packet_p->midi_status = 0xFB;
	packet_p->data_len = 0; 
//	packet_p->midi_data[0] = 0x01;

	printf("INSERTED Message=%2d: 0x%02X%02X \n" ,packet_p->data_len ,packet_p->midi_status ,packet_p->midi_data[0] ); //MB~

  midi_tx_p->burst_tx_cnt--; // Decrement message write counter
} // if (0xD6 != packet_p->midi_status)
#endif //MB~

#ifdef MB // Test Swap messages
if (packet_p->midi_status == 0xFB)
{
	packet_p->midi_status = 0xFA;
	packet_p->data_len = 0; 
}
else
{
	if (packet_p->midi_status == 0xFA)
	{ 
		packet_p->midi_status = 0xFB;
		packet_p->data_len = 0; 
	} // if (packet_p->midi_status

}
#endif //MB~

#ifdef MB // Test Pre-Repeated message
if (packet_p->midi_status == 0xFB)
{
	packet_p->midi_status = 0xFC;
	packet_p->data_len = 0; 

	packet_p->midi_data[0] = 0x04;
} // if (packet_p->midi_status
#endif //MB~

	packet_to_message( gener_p ); // Send 'Normal' message to Midi-Out port

	return;
} // dbg_pkt_hacks 
/*****************************************************************************/
#endif // (1 == DBG_HAX)
/*****************************************************************************/
bool do_midi_message_test( // Do one Midi-message test
	GLOBAL_TYP * glob_p // Pointer to structure of global data
) // Returns flag indicating if send message was successful
/*
 * This function forms the 'message hub' where input/output is assigned,
 * based on if transmitting or receiving Midi-messages
 */
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Get pointer to command-options data structure
	GENER_TYP * gener_p = &(glob_p->gener_s); // Pointer to structure of test-vector generation data
	FILE_CONFIG_TYP * file_cfg_p = gener_p->file_cfg_p; // Get pointer to structure of file configuration data
	MIDI_TX_TYP * midi_tx_p = &(gener_p->tx_data_s); // Get pointer to structure of Midi data to transmit
	MIDI_PACKET_TYP * packet_p = &(midi_tx_p->packet_s); // Packet of Midi-message data 

	// Determine communications configuration
	switch(glob_p->config_id)
	{
		case TRANSMIT : // Transmit mode 
			// Decide if Midi-messages are read from file
			if (INP_FILE == gener_p->file_cfg_p->mode)
			{ // Read Midi-messages from input file 
				if (false == read_midi_message( file_cfg_p ,packet_p )) return false;
			} // if (INP_FILE == gener_p->file_cfg_p->mode)
			else
			{
				generate_midi_message( gener_p ); // Generate Midi-message
	
				if (false == write_midi_message( file_cfg_p ,packet_p )) return false; // Write Midi-message to file
			} // else !(INP_FILE == gener_p->file_cfg_p->mode)
	
#if (1 == DBG_HAX)
	dbg_pkt_hacks( gener_p ); // MB~ 'packet hacks' used to test code
#else // (1 == DBG_HAX)
			if (false == packet_to_message( gener_p )) return false; // Send message to Midi-Out port
#endif // (1 == DBG_HAX)
		break; // case TRANSMIT

		case RECEIVE : // Receive mode 
			// Check if configured for loop-back
			if (options_p->loopback)
			{ // Loop-back configured
				generate_midi_message( gener_p ); // Generate Midi-message

#if (1 == DBG_HAX)
	dbg_pkt_hacks( gener_p ); // MB~ 'packet hacks' used to test code
#else // (1 == DBG_HAX)
				if (false == packet_to_message( gener_p )) return false; // Send message to Midi-Out port
#endif // (1 == DBG_HAX)
			} // if (options_p->loopback)
		break; // case RECEIVE

		default:
			printf("\nERROR: NOT in Transmitting or Receiving Mode!\n");
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch( glob_p->config_id)

	return true;
} // do_midi_message_test
/*****************************************************************************/
static void wait_for_receiver( // Wait for previous test results to be received
	GLOBAL_TYP * glob_p, // Pointer to structure of global data
	S32_T burst_rx_len // Received message burst length
)
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Get pointer to command-options data structure
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 
	GENER_TYP * gener_p = &(glob_p->gener_s); // Pointer to structure of test-vector generation data
	MIDI_TX_TYP * midi_tx_p = &(gener_p->tx_data_s); // Get pointer to structure of Midi data to transmit
	S32_T est_chk_len; // Currently expected length checked-message burst
	S32_T sleep_cnt = 0; // Counts number of naps
	S32_T next_off; // Next offset after current burst of checked resullts


	// Calculate expected burst length, from configured length, and previous burst error
	est_chk_len = burst_rx_len + check_p->burst_err; // Preset expected length of checked-message burst
	check_p->burst_err = 0; // Clear new burst error

	// Wait a while for received messages to be checked
	while ((check_p->burst_chk_cnt < est_chk_len) && (0 < est_chk_len))
	{
		SLEEP_FN( 5 ); // Sleep for 5 milliseconds

		// Waiting for check_p->burst_chk_cnt to increment

		sleep_cnt++; // Increment number of naps

		// Interrupt every 5x64 = 320 ms
		if (0 == (sleep_cnt & 0x3F)) // 0x3F
		{ // Try smaller estimate of checked-messages 
			est_chk_len--; // Decrement estimated burst-length 
			check_p->burst_err++; // Increment new burst length error
			printf("Estimated Check Length=%d \n" ,est_chk_len );

		} // if (0 == (timeout_ms & 0xFF))
	} //	while

	if (0 >= est_chk_len)
	{
		printf("\nERROR: Timed-out waiting for Receiver burst Tx=%d  Rx=%d \n" ,midi_tx_p->burst_tx_cnt ,check_p->burst_chk_cnt );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (0 >= est_chk_len)

	// Calculate Next offset after current burst of checked-resullts
	next_off = update_circular_buffer( check_p->prev_off ,est_chk_len ,NUM_RX_BUFS );

	if (options_p->print)
	{
		printf("                                    --- Start Of Burst --- \n" );
		display_results( check_p ,check_p->prev_off ,next_off ); // display results for previous burst
	} // if (options_p->print)

	check_p->prev_off = next_off; // Store checked-results offset for next display

	check_p->burst_chk_cnt = 0; // NB Reset count before more mycallback() interrupts
} // wait_for_receiver
/*****************************************************************************/
static void synchronise( // Synchronise transmitter and receiver at end of each burst
	GLOBAL_TYP * glob_p, // Pointer to structure of global data
	S32_T burst_rx_len // Received message burst length
)
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Get pointer to command-options data structure
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data 
	GENER_TYP * gener_p = &(glob_p->gener_s); // Pointer to structure of test-vector generation data
	MIDI_TX_TYP * midi_tx_p = &(gener_p->tx_data_s); // Get pointer to structure of Midi data to transmit
	char byt_val = 1; // Dummy byte value


	if (options_p->loopback)
	{ // Loop-back configured
		wait_for_receiver( glob_p ,burst_rx_len ); // Wait for previous test results to be received
	} // if (options_p->loopback)
	else
	{
		// Determine communications configuration
		switch(glob_p->config_id)
		{
			case TRANSMIT : // Transmit mode 
				// Wait for all Midi-messages for one test vector to be received
				glob_p->sckt_p->socReceive( &byt_val ,1 );
			break; // case TRANSMIT
	
			case RECEIVE : // Receive mode 
				wait_for_receiver( glob_p ,burst_rx_len ); // Wait for previous test results to be received
	
				// Signal all Midi-messages for one test vector have been received
				glob_p->sckt_p->socSend( &byt_val ,1 );
			break; // case RECEIVE
	
			default:
				printf("\nERROR: NOT in Transmitting or Receiving Mode!\n");
				printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
				exit(-1);
			break; // default
		} // switch( glob_p->config_id)
	} // else !(options_p->loopback)

	// Clear burst counters
  check_p->burst_chk_cnt = 0;
  midi_tx_p->burst_tx_cnt = 0;
} // synchronise
/*****************************************************************************/
bool do_test_vector_burst( // Generates a burst of xmidi test vectors
	GLOBAL_TYP * glob_p, // Pointer to structure of global data
	S32_T burst_cnt, // burst counter
	S32_T burst_len // Current burst length
) // Returns flag indicating if send message was successful
{
	S32_T msg_cnt; // Midi-message counter


	// loop through Midi-messages in burst
	for (msg_cnt=0; msg_cnt<burst_len; msg_cnt++)
	{
		if (false == do_midi_message_test( glob_p ))
		{
			printf("ERROR detected. Returning early");
			printf("      From file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			return false;
		} // if (false == do_midi_message_test( glob_p ))
	} // for msg_cnt

	// Determine is 1st burst
	if (0 < burst_cnt)
	{ // NOT 1st burst
		synchronise( glob_p ,burst_len ); // Wait for receiver to check results
	} // if (0 < burst_cnt)
	else
	{ // Special case of 1st burst: NB Checker needs to inspect next Midi-message
		synchronise( glob_p ,(burst_len - 1) ); // Wait for receiver to check results
	} // if (0 < burst_cnt)

	return true;
} // do_test_vector_burst
/*****************************************************************************/
static void flush_results_buffer( // Flushes final (uncheckable) results from buffer
	GLOBAL_TYP * glob_p // Pointer to structure of global data
)
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Get pointer to command-options data structure
	CHECK_TYP * check_p = &(glob_p->check_s); // Pointer to structure of Midi check data
	bool do_flush = false; // Preset to do NOT flush


	if (options_p->loopback)  
	{ // Loop-back configured
		do_flush = true; // Set flush flag
	} // if (options_p->loopback)
	else
	{
		// Determine communications mode
		if (RECEIVE == glob_p->config_id)
		{
			do_flush = true; // Set flush flag
		} // if (RECEIVE == glob_p->config_id)
	} // else ! (options_p->loopback)

	if (do_flush)
	{
		printf("                                    --- End of Stream Integrity Checks ---\n");

		// Wait a while for (NON test-vector) Midi-messages
		SLEEP_FN( 1000 ); // Sleep for 1000 milliseconds

		// Display remaining (uncheckable) results in buffer
		display_results( check_p ,check_p->prev_off ,check_p->buf_off );

		printf("                                    TEST FINISHED\n");
	} // if (do_flush)

} // flush_results_buffer
/*****************************************************************************/
bool do_all_test_vectors( // Generates all xmidi test vectors
	GLOBAL_TYP * glob_p // Pointer to structure of global data
) // Returns flag indicating if send message was successful
{
	OPTIONS_TYP * options_p = &(glob_p->options_s); // Get pointer to command-options data structure
	S32_T num_bursts = (options_p->num_vects / MAX_BURST_LEN); // No. of full bursts of Midi-messages 
	S32_T rem_vects = (options_p->num_vects - (num_bursts * MAX_BURST_LEN)); // Remaining test-vectors in last burst
	S32_T burst_cnt; // burst counter


	// loop through bursts of test vectors
	for (burst_cnt=0; burst_cnt < num_bursts; burst_cnt++) 
	{
		if (false == do_test_vector_burst( glob_p ,burst_cnt ,MAX_BURST_LEN )) return false;
	} // for (burst_cnt=0; burst_cnt < num_bursts; burst_cnt++) 

	// Determine if partial busrt required
	if (0 < rem_vects)
	{ // Do partial burst
		if (false == do_test_vector_burst( glob_p ,num_bursts ,rem_vects )) return false;
	} // if (0 < rem_vects)

	if (options_p->print) flush_results_buffer( glob_p ); // Flushes final (uncheckable) results from buffer

	return true;
} // do_all_test_vectors
/*****************************************************************************/
U8_T gen_final_test_result( // Generates final 8-bit test result to return to shell
	CHECK_TYP * check_p  // Pointer to structure of Midi check data 
) // Return 8-bit test result
{
	U8_T tst_res; // test result returned to shell


	// Only return values in range 0..UCHAR_MAX allowed
	if (UCHAR_MAX > check_p->total_fails)
	{
		tst_res = (U8_T)check_p->total_fails;
	} // if (UCHAR_MAX > check_p->total_fails)
	else
	{
		tst_res = (U8_T)UCHAR_MAX;
	} // if (UCHAR_MAX > check_p->total_fails)

	return tst_res; // Return 8-bit test result
} // gen_final_test_result
/*****************************************************************************/
// xmidi_common
