//*****************************************//
//  xmidi_check.h
//
//  Re-create xmidi_check using RtMidi
//
//*****************************************//

#ifndef XMIDI_CHECK_H
#define XMIDI_CHECK_H

#include <iostream>
#include <cstdlib>
#include <typeinfo>
#include <assert.h>
#include "RtMidi.h"
#include "types64bit.h"
#include "xmidi_callback.h"

/* All Midi States */
typedef enum MIDI_STATE_ETAG
{
  START_MESS = 0, // Waiting for a Midi status byte
  CHAN_MESS,	// Processing channel message
  TIME_MESS,	// Processing Real-Time message
  SYST_MESS,	// Processing system message (NOT system exculsive)
  EXCL_MESS,	// Processing system exculsive message
  END_MESS,		// End of message
  END_TEST,		// End of test-vectors
  NUM_MIDI_STATES	// Handy Value!-)
} MIDI_STATE_ENUM;

// Structure containing System-Exclusive check data
typedef struct SYSEX_TAG
{
	MANU_ID_LEN_ENUM manu_id_len; // Length of Manufacturer's Identifier in bytes
	S32_T shift_bits; // Number of data bits in each SysEx data byte
	S32_T bit_mask; // Used to mask required LS data bits from check value
} SYSEX_TYP;

// Structure of MIDI results statistics for one type of Midi-message
typedef struct RES_STATS_TAG
{
	S32_T fail_sts; // Counts failed status bytes
	S32_T good_sts; // Counts successful status bytes 
	S32_T fail_param; // Counts failed test parameters
	S32_T good_param; // Counts successful test parameters
	S32_T fail_len; // Counts failed message-length tests
	S32_T good_len; // Counts successful message-length tests
	S32_T fail_byts; // Counts failed byte checks (Non Real-time) 
	S32_T good_byts; // Counts successful byte checks  (Non Real-time) 
	S32_T missed; // Counts 'missing' messages
	S32_T extra; // Counts extra messages
	S32_T swapped; // Counts swapped messages
	S32_T pre_repeats; // Counts post-repeated messages
	S32_T pst_repeats; // Counts messages overwritten by next (pre-repeated) message
} RES_STATS_TYP;

// Structure of verification data for one test-vector
typedef struct VERIFY_TAG
{
	TST_VECT_TYP chk_vect_s; // Structure of test-vector check data
	TST_VECT_TYP res_vect_s; // Structure of test-vector result data
	RES_STATS_TYP * res_stats_p; // Pointer to structure of results statistics for this test-vector
} VERIFY_TYP;

// Structure containing MIDI check data
typedef struct CHECK_TAG
{
	RES_STATS_TYP res_stats_s[NUM_MIDI_MSGS]; // Array of results statistics structures for each type of Midi-messsage
	MIDI_RX_TYP rx_data_s; // Structure of received Midi data
	FILE_CONFIG_TYP * file_cfg_p; // Pointer to structure of file configuration data
	const MSG_CONFIG_TYP * msg_cfg_arr_p; // Pointer to array of Midi-message configuration data
	TST_GEN_TYP tst_gen_s; // Structure of test generation data (re-written for each Midi-message)
	SYSEX_TYP sysex_s; // Structure of System-Exclusive decoding data
	MIDI_PACKET_TYP chk_pkt_arr[2]; // Array of Midi-message data packets for check data (Current & New)
	VERIFY_TYP new_verify_s; // Structure of new verification data (re-written for each Midi-message)
	VERIFY_TYP cur_verify_s; // Structure of current verification data under test (re-written for each Midi-message)

  std::bitset<8> bits; // Array of 8 bits (used for printing binary)

	S32_T real_time_check[NUM_RT_MSGS]; // Array of real-time check values
  R64_T diff_time; // Difference time (since previous time-stamp)
	volatile S32_T burst_chk_cnt; // Count of No. of checked messages in current burst
	volatile S32_T burst_err; // received burst length error (Dynamically changes)
	volatile S32_T buf_off; // Offset into buffer of received Midi-messages
	volatile S32_T prev_off; // Previous offset into buffer

	MIDI_STATE_ENUM state; // Current Midi State
	S32_T num_vects; // Number of test-vectors expected
	S32_T total_fails; // Total number of failed tests of all types
	S32_T toggle; // Toggles between 0 & 1, used for indexing chk_pkt_arr[2]

	bool first_msg; // Flag set until first good Midi-message received
	bool extra_msg; // Flag set if extra Midi-message detected
	bool missed_msg; // Flag set if missed Midi-message detected
	bool valid_res; // Flag set if valid result (ready to check)
	bool chk_tst_vect; // Set flag if checking against test vector
	COMMS_CONFIG_ENUM config_id; // Identifies communication configuration

	bool print; // Print to screen (if flag set)
} CHECK_TYP;

/*****************************************************************************/
void init_verification_data( // Initialise structure of test-vector verification data
	VERIFY_TYP * verify_p, // Pointer to structure of test-vector verification data
	const MIDI_PACKET_TYP * inp_pkt_p // input pointer to structure of Midi packet data
);
/*****************************************************************************/
void init_check_data( // Initialise structure for midi check data
	MSG_CONFIG_TYP msg_cfg_arr[], // Array of structures of Midi-message configuration data
	OPTIONS_TYP * options_p, // Pointer to command-option data structure
	CHECK_TYP * check_p // Pointer to structure of Midi check data
);
/*****************************************************************************/
void init_sysex_data( // Initialise structure of System-Exclusive decoding data
	SYSEX_TYP * sysex_p, // Pointer to structure of System-Exclusive decoding data
	MANU_ID_LEN_ENUM inp_len, // Length of Manufacturer's Identifier in bytes
	S32_T inp_bits, // Number of data bits in each SysEx data byte
	S32_T inp_mask // Used to mask required LS data bits from check value
);
/*****************************************************************************/
bool configure_midi_input_port( // Configure which Midi input port to use
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data
	const char port_names[NUM_VALID_PORTS][STR_NAME_LEN] // Array of valid port names
); // Returns true if configuration successful
/*****************************************************************************/
S32_T update_circular_buffer( // Updates offset into circular buffer, & handles wrap-around
	S32_T inp_off, // input buffer offset
	S32_T inp_inc, // input increment
	S32_T buf_len // buffer length
);
/*****************************************************************************/
void display_results( // Print all result in buffer
	CHECK_TYP * check_p, // Pointer to structure of Midi check data
	S32_T disp_off, // Starting offset into display buffer
	S32_T disp_len // No. of buffer entries to be displayed
);
/*****************************************************************************/
void print_all_message_results( // Print all results statistics for each Midi-message type
	CHECK_TYP * check_p // Pointer to structure of Midi check data
);
/*****************************************************************************/

#endif /* XMIDI_CHECK_H */
