//*****************************************//
//  xmidi_check.cpp
//
//  Simple program to test MIDI sysex sending and receiving.
//
//*****************************************//

#include "RtMidi.h"
#include "xmidi_common.h"
#include "xmidi_check.h"

/*****************************************************************************/
void init_message_results_data( // Initialise structure of results statistics for one Midi-message type
	RES_STATS_TYP * res_stats_p // Pointer to structure of results statistics for one Midi-message type
)
{
	// Initialise all results counters for current Midi-message type 
	res_stats_p->fail_sts = 0;		// failed status bytes
	res_stats_p->good_sts = 0;		// successful status bytes 
	res_stats_p->fail_param = 0;	// failed test parameters
	res_stats_p->good_param = 0;	// successful test parameters
	res_stats_p->fail_len = 0;		// failed message-length tests
	res_stats_p->good_len = 0;		// successful message-length tests
	res_stats_p->fail_byts = 0;		// failed byte checks (Non Real-time) 
	res_stats_p->good_byts = 0;		// successful byte checks  (Non Real-time) 
	res_stats_p->missed = 0;			// 'missing' messages
	res_stats_p->extra = 0;				// extra messages
	res_stats_p->swapped = 0;			// swapped messages
	res_stats_p->pre_repeats = 0;	// post-repeated messages
	res_stats_p->pst_repeats = 0;	// messages overwritten by next (pre-repeated) message
} // // init_message_results_data 
/*****************************************************************************/
void init_sysex_data( // Initialise structure of System-Exclusive decoding data
	SYSEX_TYP * sysex_p, // Pointer to structure of System-Exclusive decoding data
	MANU_ID_LEN_ENUM inp_len, // Length of Manufacturer's Identifier in bytes
	S32_T inp_bits, // Number of data bits in each SysEx data byte
	S32_T inp_mask // Used to mask required LS data bits from check value
)
{
	// Initialise for 'Small' System exclusive data
	sysex_p->manu_id_len = inp_len; // Length of Manufacturer's Identifier
	sysex_p->shift_bits = inp_bits; // shift bits for bit_mask
	sysex_p->bit_mask = inp_mask; // Bit-Mask 
} // init_sysex_data
/*****************************************************************************/
void init_verification_data( // Initialise structure of test-vector verification data
	VERIFY_TYP * verify_p, // Pointer to structure of test-vector verification data
	const MIDI_PACKET_TYP * inp_pkt_p // input pointer to structure of Midi packet data
)
{
	init_test_vector_data( &(verify_p->chk_vect_s) ); // Initialise test-vector check data
	init_test_vector_data( &(verify_p->res_vect_s) ); // Initialise test-vector result data

	verify_p->chk_vect_s.packet_p = (MIDI_PACKET_TYP *)inp_pkt_p;	// Point to input location for check packet
	verify_p->res_stats_p = NULL; // Pointer to structure of results statistics for this test-vector
} // init_verification_data
/*****************************************************************************/
void init_check_data( // Initialise structure for midi check data
	MSG_CONFIG_TYP msg_cfg_arr[], // Array of structures of Midi-message configuration data
	OPTIONS_TYP * options_p, // Pointer to command-option data structure
	CHECK_TYP * check_p // Pointer to structure of Midi check data
)
{
	int msg_cnt; // Midi-message type counter


	// Loop through all Midi-message types
	for (msg_cnt=0; msg_cnt<NUM_MIDI_MSGS; msg_cnt++)
	{ 
		init_message_results_data( &(check_p->res_stats_s[msg_cnt]) );
	} // for msg_cnt

  check_p->real_time_check[0] = 0; // Timing-Clock
  check_p->real_time_check[1] = UNDEFINED;
  check_p->real_time_check[2] = 1; // Start-Song
  check_p->real_time_check[3] = 2; // Continue-Song
  check_p->real_time_check[4] = 3; // Stop-Song
  check_p->real_time_check[5] = UNDEFINED;
  check_p->real_time_check[6] = UNTESTED; // Active-Sensing: Currently un-tested
  check_p->real_time_check[7] = UNTESTED; // Reset: Currently un-tested

  check_p->first_msg = true; // Set flag until first good Midi-message received
	check_p->extra_msg = false; // Preset flag to extra Midi-message NOT detected
	check_p->missed_msg = false; // Preset flag to missed Midi-message NOT detected
	check_p->valid_res = true; // Preset flag to valid result (ready to check)
  check_p->state = START_MESS; // Initialise Midi state to 'Wait for message start'
	check_p->burst_chk_cnt = 0;	// Clear count of Checked Midi-messages in current burst
	check_p->burst_err = 0; // Error between configured and received burst length
	check_p->buf_off = 0;	// Clear offset into buffer of received Midi-messages
	check_p->prev_off = 0;	// Clear previous offset into buffer
  check_p->diff_time = 0; // Clear Difference time (since previous time-stamp)
	check_p->total_fails = 0;	// Clear total number of failures (of all types)
	check_p->msg_cfg_arr_p = msg_cfg_arr; // Set pointer to first element of Midi-message configuration data
	check_p->toggle = 1;	// Toggles between 0 & 1, used for indexing chk_pkt_arr[2] 

	// Initialise test generation data (NB re-written for each Midi-message)
	init_test_generation_data( msg_cfg_arr ,&(check_p->tst_gen_s) ,options_p->seed );

	check_p->num_vects = options_p->num_vects; // Initialise number of test-vectors expected
	check_p->print = options_p->print; // Initialise screen-printing mode

} // init_check_data
/*****************************************************************************/
bool configure_midi_input_port( // Configure which Midi input port to use
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data
	const char port_names[NUM_VALID_PORTS][STR_NAME_LEN] // Array of valid port names
) // Returns true if configuration successful
{
  // RtMidiIn constructors
  try 
	{
    check_p->rx_data_s.midi_in_p = new RtMidiIn();
  } // try
  catch ( RtMidiError &error ) 
	{
		printf("\nERROR: Failed to successfully construct RtMidi Input Object\n");
    error.printMessage();
    return false;
  } // catch

  // Don't ignore sysex, timing, or active sensing messages.
  check_p->rx_data_s.midi_in_p->ignoreTypes( false ,false ,false );

  try 
	{
    if ( NUM_VALID_PORTS == open_midi_port( check_p->rx_data_s.midi_in_p  ,port_names ,"Input" ,check_p->print ))  return false;
  } // try
  catch ( RtMidiError &error ) 
	{
		printf("\nERROR: Failed to successfully open Midi Input Port\n");
    error.printMessage();
    return false;
  } // catch

  check_p->rx_data_s.midi_in_p->setCallback( &mycallback ,(void *)check_p ); // NB Pass in pointer to global data structure

	return true;
} // 	configure_midi_input_port
/*****************************************************************************/
S32_T update_circular_buffer( // Updates offset into circular buffer & handles wrap-around
	S32_T inp_off, // input buffer offset
	S32_T inp_inc, // input increment
	S32_T buf_len // buffer length
)
{
	S32_T out_off = inp_off + inp_inc; // Add Increment to input buffer offset


	// Handle Wrap-around
	if (buf_len <= out_off)
	{ // Offset is Too-Big
		out_off -= buf_len;
	} // if (buf_len <= out_off)
	else
	{	// Handle -ve Wrap-around
		if (0 > out_off) out_off += buf_len;
	} // if (buf_len <= out_off)

	return out_off; // Return new offset
} // update_circular_buffer 
/*****************************************************************************/
void display_partial_results( // Print a set of contiguous buffer results
	CHECK_TYP * check_p, // Pointer to structure of Midi check data
	S32_T first_off, // Starting offset into display buffer
	S32_T last_off // Last buffer offset to be displayed 
)
{
	MIDI_RX_TYP * midi_rx_p = &(check_p->rx_data_s); // Get pointer to structure of received Midi data 
	MIDI_PACKET_TYP * rx_pkt_p; // Pointer to next results buffer
	S32_T buf_cnt; // Buffer counter


	// Loop through contiguous buffer entries
	for(buf_cnt=first_off; buf_cnt <= last_off; buf_cnt++)
	{
		rx_pkt_p = &(midi_rx_p->pkt_bufs[buf_cnt]); // Get pointer to next results buffer
		printf("                                    Diff_Time=%7.5f:" ,rx_pkt_p->diff_time );

		print_midi_message( check_p->msg_cfg_arr_p ,rx_pkt_p ,&(check_p->bits) ); // print a Midi-message as binary digits
	} // for buf_cnt
} // display_partial_results
/*****************************************************************************/
void display_results( // Print all result in buffer
	CHECK_TYP * check_p, // Pointer to structure of Midi check data
	S32_T curr_off, // Starting display-buffer offset for current results burst
	S32_T next_off // Starting display-buffer offset for next results burst
)
{
	S32_T last_off; // Last buffer offset to be displayed 


	last_off = update_circular_buffer( next_off ,(-1) ,NUM_RX_BUFS ); // Evaluate last display offset for current burst

	// Determine if wrap-around
	if (curr_off > last_off)
	{ // Wrap-around
		display_partial_results( check_p ,curr_off ,(NUM_RX_BUFS - 1) );
		display_partial_results( check_p ,0 ,last_off );
	} // if (last_off < NUM_RX_BUFS)
	else
	{ // NO wrap-around
		display_partial_results( check_p ,curr_off ,last_off );
	} // if (last_off < NUM_RX_BUFS)

} // display_results
/*****************************************************************************/
static void print_result_pair( // Print Good/Fail pair of results
	int fail_cnt, // accumulated failures
	int good_cnt // accumulated successes
)
{
	int num_tests = (good_cnt + fail_cnt); // Total number of tests


	if (fail_cnt)
	{
		printf("%3d " ,fail_cnt );
	} // if (fail_cnt)
	else
	{
		printf(" .. ");
	} // if (fail_cnt)

	if (num_tests)
	{
		printf("(%04d) " ,num_tests );
	} // if (num_tests)
	else
	{
		printf("(....) " );
	} // if (fail_cnt)
} // print_result_pair
/*****************************************************************************/
static void print_one_result( // Print single result
	int fail_cnt // accumulated failures
)
{
	if (fail_cnt)
	{
		printf("%6d " ,fail_cnt );
	} // if (fail_cnt)
	else
	{
		printf("  .... ");
	} // if (fail_cnt)

} // print_one_result
/*****************************************************************************/
static void print_message_results_data( // print structure of results statistics for one Midi-message type
	RES_STATS_TYP * res_sum_p, // Pointer to results sum structure
	RES_STATS_TYP * res_stats_p // Pointer to structure of results statistics for one Midi-message type
)
{
	// print result pairs

	print_result_pair( res_stats_p->fail_sts ,res_stats_p->good_sts );			// Status-byte results 
	print_result_pair( res_stats_p->fail_param ,res_stats_p->good_param );	// Chan/Data Parameter results 
	print_result_pair( res_stats_p->fail_len ,res_stats_p->good_len );			// Message length results 
	print_result_pair( res_stats_p->fail_byts ,res_stats_p->good_byts );		// Message byte results 

	print_one_result( res_stats_p->missed );				// 'missing' messages
	print_one_result( res_stats_p->extra );				// extra messages
	print_one_result( res_stats_p->swapped );			// swapped messages
	print_one_result( res_stats_p->pre_repeats );	// post-repeated messages
	print_one_result( res_stats_p->pst_repeats );	// messages overwritten by next (pre-repeated) message

	// update results sums, AFTER printing (as this function also used to print sum results!-)
	res_sum_p->fail_sts += res_stats_p->fail_sts;
	res_sum_p->good_sts += res_stats_p->good_sts;
	res_sum_p->fail_param += res_stats_p->fail_param;
	res_sum_p->good_param += res_stats_p->good_param;
	res_sum_p->fail_byts += res_stats_p->fail_byts;
	res_sum_p->good_byts += res_stats_p->good_byts;
	res_sum_p->fail_len += res_stats_p->fail_len;
	res_sum_p->good_len += res_stats_p->good_len;

	res_sum_p->missed += res_stats_p->missed;
	res_sum_p->extra += res_stats_p->extra;
	res_sum_p->swapped += res_stats_p->swapped;
	res_sum_p->pre_repeats += res_stats_p->pre_repeats;
	res_sum_p->pst_repeats += res_stats_p->pst_repeats;
} // print_message_results_data
/*****************************************************************************/
void print_all_message_results( // Print all results statistics for each Midi-message type
	CHECK_TYP * check_p // Pointer to structure of Midi check data
)
{
	RES_STATS_TYP res_sum; // Used to sum results for results-type
	RES_STATS_TYP * res_sum_p = &res_sum; // Get local pointer to results sum structure
	int msg_cnt; // Midi-message type counter


	init_message_results_data( res_sum_p ); // Clear results sum

	printf("\n   Status     Param      Length     Bytes    Missed  Extra  Swap  PreRpt PstRpt Message-Type\n");
	printf(" ---------  ---------  ---------  ---------  ------ ------ ------ ------ ------ ------------\n");

	// Loop through all Midi-message types
	for (msg_cnt=0; msg_cnt<NUM_MIDI_MSGS; msg_cnt++)
	{ 
		print_message_results_data( res_sum_p ,&(check_p->res_stats_s[msg_cnt]) );

		printf(" %s\n" ,check_p->msg_cfg_arr_p[msg_cnt].name );
	} // for msg_cnt

	// Sum total failures, BEFORE printing sum results, (as this corrupts sum_results)
	check_p->total_fails = res_sum_p->fail_sts + res_sum_p->fail_param + res_sum_p->fail_len + res_sum_p->fail_byts
		+ res_sum_p->missed + res_sum_p->extra + res_sum_p->swapped + res_sum_p->pre_repeats + res_sum_p->pst_repeats;

	printf(" ---------  ---------  ---------  ---------  ------ ------ ------ ------ ------ ------------\n");
	print_message_results_data( res_sum_p ,res_sum_p );
	printf(" Total \n");
	printf(" ---------  ---------  ---------  ---------  ------ ------ ------ ------ ------ ------------\n");

	printf("\nNB Shows:   Number_Of_Failures (Total_Number_Tested) \n");

} // print_all_message_results
/*****************************************************************************/
// xmidi_check
