//*****************************************//
//  xmidi_callback.cpp
//
//  Simple program to test MIDI sysex sending and receiving.
//
//*****************************************//

#include "RtMidi.h"
#include "xmidi_common.h"
#include "xmidi_check.h"
#include "xmidi_callback.h"

/*****************************************************************************/
static void clear_vector_data( // Clear verification identification data at start of new message 
	TST_VECT_TYP * tst_vect_p // Pointer to test-vector data 
)
{
	tst_vect_p->sts = 0; // Clear status value result
	tst_vect_p->param = 0; // Clear parameter value result
	tst_vect_p->msg_cfg_p = NULL; // Clear Pointer to structure of Midi-message configuration data
} // clear_vector_data
/*****************************************************************************/
static void copy_vector_data( // Deep copy of vector data
	TST_VECT_TYP * dest_vect_p, // Pointer to destination structure of test-vector verification data
	TST_VECT_TYP * saws_vect_p // Pointer to source structure of test-vector verification data
)
{
	dest_vect_p->sts = saws_vect_p->sts;
	dest_vect_p->param = saws_vect_p->param;
	dest_vect_p->msg_cfg_p = saws_vect_p->msg_cfg_p;

	*(dest_vect_p->packet_p) = *(saws_vect_p->packet_p);
} // copy_vector_data
/*****************************************************************************/
static void clear_verify_data( // Clear verification data at start of new message 
	CHECK_TYP * check_p,   // Pointer to structure of Midi check data 
	VERIFY_TYP * verify_p // Pointer to structure of test-vector verification data
)
{
	TST_VECT_TYP * new_chk_p = &(verify_p->chk_vect_s); // Get local pointer to new test-vector check data 
	TST_VECT_TYP * new_res_p = &(verify_p->res_vect_s); // Get local pointer to new test-vector result data 


	verify_p->res_stats_p = NULL; // Clear pointer to results statistics

	clear_vector_data( new_res_p ); // Clear test test-vector result data 
	clear_vector_data( new_chk_p ); // Clear test test-vector check data

	new_chk_p->packet_p = &(check_p->chk_pkt_arr[check_p->toggle]); // Point to new space in array
	check_p->toggle = (1 - check_p->toggle); // Update array index (NB Only 2 elements Current & New)
} // clear_verify_data
/*****************************************************************************/
static void update_verify_data( // Update verification data ready for next new message 
	CHECK_TYP * check_p,   // Pointer to structure of Midi check data 
	VERIFY_TYP * cur_verify_p, // Pointer to current structure of test-vector verification data
	VERIFY_TYP * new_verify_p // Pointer to new structure of test-vector verification data
)
{
	TST_VECT_TYP * cur_vect_p; // Pointer to current test-vector data 
	TST_VECT_TYP * new_vect_p; // Pointer to new test-vector data 


	// Determine if the previous current result was an Extra message
	if (check_p->extra_msg)
	{ // Extra Message: Only update result data
		cur_vect_p = &(cur_verify_p->res_vect_s); // get local pointer to current test-vector result data
		new_vect_p = &(new_verify_p->res_vect_s); // get local pointer to new test-vector result data

		copy_vector_data( cur_vect_p ,new_vect_p ); // Copy result data (Current = New) 
		clear_vector_data( new_vect_p ); // Clear new test-vector result data 

		return; // Early return without incrementing result buffer
	} // if (check_p->extra_msg)

	check_p->buf_off = update_circular_buffer( check_p->buf_off ,1 ,NUM_RX_BUFS ); // Increment result buffer offset

	// Determine if there was Missing result message
	if (check_p->missed_msg)
	{ // Just update check data
		cur_vect_p = &(cur_verify_p->chk_vect_s); // get local pointer to current test-vector check data
		new_vect_p = &(new_verify_p->chk_vect_s); // get local pointer to new test-vector check data

		copy_vector_data( cur_vect_p ,new_vect_p ); // Copy check data (Current = New) 
		cur_verify_p->res_stats_p = new_verify_p->res_stats_p; // Update pointer to test-vector results type
		clear_vector_data( new_vect_p ); // Clear new test-vector check data 

		return; // Early return
	} // if (check_p->missed_msg)

	// Normal Message: Update all verification data ...

	check_p->cur_verify_s = check_p->new_verify_s;  // Copy chaeck & result data (Current = New) 
	clear_verify_data( check_p ,&(check_p->new_verify_s) ); // Clear check & result verification data

	return;
} // update_verify_data
/*****************************************************************************/
static bool check_byte( // Compares one input byte against check byte
	RES_STATS_TYP * res_stats_p, // Pointer to results statistics for current Midi-message-type
  U8_T inp_byt, // Input byte to check (Received from Midi-In port) 
  U8_T chk_byt // Check byte (read from input file)
) // Returns flag indicating outcome of test
{
	bool fail_flg = false; // Preset flag to successful


	// Do simple byte check
	if (inp_byt != chk_byt)
	{
		res_stats_p->fail_byts++; // Increment count of failed bytes
		fail_flg = true; // Set flag to check failed
	} // if (inp_byt != chk_byt)
	else
	{
		res_stats_p->good_byts++; // Increment count of successful bytes
	} // if (inp_byt != chk_byt)

	return fail_flg; // Return flag indicating outcome of test
} // check_byte
/*****************************************************************************/
static bool check_message_length( // Check length of received Midi-message
	VERIFY_TYP * verify_p // Pointer to current verification data 
) // Returns flag indicating outcome of test
{
	RES_STATS_TYP * res_stats_p = verify_p->res_stats_p; // Get local pointer to results statistics for current test-vector Midi-message-type
	MIDI_PACKET_TYP * chk_pkt_p = verify_p->chk_vect_s.packet_p; // Get local pointer to check packet (read from input file)
	MIDI_PACKET_TYP * res_pkt_p = verify_p->res_vect_s.packet_p; // Get local pointer to current results packet (from Midi-In port)
	bool fail_flg = false; // Preset flag to successful


	if (res_pkt_p->data_len != chk_pkt_p->data_len)
	{
		res_stats_p->fail_len++; // Increment count of failed message-length tests
		fail_flg = true; // Set flag to check failed
	} // if (res_pkt_p->data_len != chk_pkt_p->data_len)
	else
	{
		res_stats_p->good_len++; // Increment count of successful message-length tests
	} // if (res_pkt_p->data_len != chk_pkt_p->data_len)

	return fail_flg; // Return flag indicating outcome of test
} // check_message_length
/*****************************************************************************/
static bool check_midi_message( // Check received Midi-message byte-for-byte
	VERIFY_TYP * verify_p // Pointer to current verification data 
) // Returns flag indicating outcome of test
{
	RES_STATS_TYP * res_stats_p = verify_p->res_stats_p; // Pointer to results statistics for current test-vector Midi-message-type
	MIDI_PACKET_TYP * chk_zz_pkt_p = verify_p->chk_vect_s.packet_p; // Check packet (read from input file)
	MIDI_PACKET_TYP * res_zz_pkt_p = verify_p->res_vect_s.packet_p; // result packet (from Midi-In port)
	S32_T byte_cnt; // Byte counter
	bool fail_flg = false; // Preset flag to successful


	// Check length of received Midi-message
	fail_flg = check_message_length( verify_p );

	// Check status byte
	if (true == check_byte( res_stats_p ,res_zz_pkt_p->midi_status ,chk_zz_pkt_p->midi_status ))
	{
		fail_flg = true; // Set flag to check failed
	} // if (true == check_byte( res_stats_p ,res_zz_pkt_p->midi_status ,chk_zz_pkt_p->midi_status ))

	// Loop through any received data-bytes
  for (byte_cnt=0; byte_cnt<chk_zz_pkt_p->data_len; byte_cnt++)
	{
		if (true == check_byte( res_stats_p ,res_zz_pkt_p->midi_data[byte_cnt] ,chk_zz_pkt_p->midi_data[byte_cnt] ))
		{
			fail_flg = true; // Set flag to check failed
 		} // if (true == check_byte( res_stats_p ,res_zz_pkt_p->midi_data[byte_cnt] ,chk_zz_pkt_p->midi_data[byte_cnt] ))
  } // for byte_cnt

	return fail_flg; // Return flag indicating outcome of test
} // check_midi_message
/*****************************************************************************/
static void get_new_check_bytes( // Build a new packet of check data by reading check-bytes from input-file 
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
	TST_VECT_TYP * tst_vect_p // Pointer to test-vector data 
)
{
	FILE_CONFIG_TYP * file_cfg_p = check_p->file_cfg_p; // local pointer to structure of file configuration data
	bool success = true; //MB~ 


	// Determine mode of check file)
	switch(file_cfg_p->mode)
	{
		case INP_FILE : // Check received bytes against file of check data
			// Get Midi-message from Input file

			success = read_midi_message( file_cfg_p ,tst_vect_p->packet_p );

			if (false == success)
			{
				printf("ERROR: read_midi_message failed\n" );
				printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
				exit(-1);
			} // if (false == read_midi_message( file_cfg_p ,tst_vect_p->packet_p ))
// printf("GNCB:B Chk_Sts C=0x%02X N=0x%02X \n" ,check_p->cur_verify_s.chk_vect_s.packet_p->midi_status ,check_p->new_verify_s.chk_vect_s.packet_p->midi_status );
		break; // case INP_FILE

		case OUT_FILE : // Results Dumped to file (for off-line checking)
			// Nothing to do
		break; // case OUT_FILE

		default:
			printf("ERROR: Unexpected File-mode %d detected\n" ,file_cfg_p->mode );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(file_cfg_p->mode)

	return;
} // get_new_check_bytes
/*****************************************************************************/
static void get_new_check_data( // Get check data for New Midi-message from configuration data
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
	VERIFY_TYP * verify_p // Pointer to current verification data 
)
{
	TST_VECT_TYP * tst_vect_p = &(verify_p->chk_vect_s); // Get local pointer to test-vector data 
	TST_GEN_TYP * tst_gen_p = &(check_p->tst_gen_s); // Get local pointer to test generation data 
	MIDI_MSG_ENUM msg_id; // Midi-message idewntifier


	// Populate the test-vector with the same data that was used to create the next Midi-message 
	gen_test_vector_data( tst_gen_p ,tst_vect_p );
	msg_id = tst_vect_p->msg_cfg_p->id; // Get Midi-message type identifier 

	verify_p->res_stats_p = &(check_p->res_stats_s[msg_id]); // Pointer to results statistics for this Midi-message-type

	// Decide if Midi-message should be checked byte-for-byte
	if (RECEIVE == check_p->config_id)
	{ // checking Midi-message
		get_new_check_bytes( check_p ,&(verify_p->chk_vect_s) );
	} // if (RECEIVE == check_p->config_id)
} // get_new_check_data
/*****************************************************************************/
static bool check_vector_result( // Calculate xmidi check result for Midi-message
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
	VERIFY_TYP * verify_p // Pointer to current verification data 
) // Returns flag indicating outcome of test
{
	RES_STATS_TYP * res_stats_p = verify_p->res_stats_p; // get local pointer to results statistics for current test-vector Midi-message-type
	TST_VECT_TYP * vect_mb_chk_p = &(verify_p->chk_vect_s); // Get local pointer to test-vector check data 
	TST_VECT_TYP * vect_mb_res_p = &(verify_p->res_vect_s); // Get local pointer to test-vector result data 
	bool fail_flg = false; // Preset flag to successful


	// Decide if xmidi test result required
	if (check_p->chk_tst_vect)
	{
		// Check if Status Byte test passed
		if (vect_mb_res_p->sts == vect_mb_chk_p->sts)
		{ // Success
			res_stats_p->good_sts++; 	
		} // if (vect_mb_res_p->sts == vect_mb_chk_p->sts)
		else
		{ // Failed
			res_stats_p->fail_sts++;
			fail_flg = false; // Set flag NOT successful
		} // if (vect_mb_res_p->sts == vect_mb_chk_p->sts)

		// Check if Data parameter test passed
		if (vect_mb_res_p->param == vect_mb_chk_p->param)
		{ // Success
			res_stats_p->good_param++; 	
		} // if (vect_mb_res_p->param == vect_mb_chk_p->param)
		else
		{ // Failed
			res_stats_p->fail_param++; 	
			fail_flg = false; // Set flag NOT successful
	
		} // if (vect_mb_res_p->param == vect_mb_chk_p->param)
	} // if (check_p->chk_tst_vect)

	return fail_flg; // Return flag indicating outcome of test
} // check_vector_result
/*****************************************************************************/
static bool check_message_bytes( // Calculate check result for Real-Time message
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
	VERIFY_TYP * verify_p // Pointer to current verification data 
) // Returns flag indicating outcome of test
{
	FILE_CONFIG_TYP * file_cfg_p = check_p->file_cfg_p; // local pointer to structure of file configuration data
	MIDI_PACKET_TYP * res_pkt_p = check_p->cur_verify_s.res_vect_s.packet_p; // Get local pointer to current packet of Midi-message result data 
	bool fail_flg = false; // Preset flag to successful


	// Determine mode of check file)
	switch(file_cfg_p->mode)
	{
		case INP_FILE : // Check received bytes against file of check data
			// Check Midi-packet byte-for byte
			fail_flg = check_midi_message( verify_p ); // Returns outcome of message bytes check
		break; // case INP_FILE

		case OUT_FILE : // Dump to results file (for off-line checking)
			if (false == write_midi_message( file_cfg_p ,res_pkt_p )) // Write Midi-message to file
			{
				printf("ERROR: write_midi_message failed\n" );
				printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
				exit(-1);
			} // if (false == write_midi_message( file_cfg_p ,packet_p )) // Write Midi-message to file
		break; // case OUT_FILE

		default:
			printf("ERROR: Unexpected File-mode %d detected\n" ,file_cfg_p->mode );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(file_cfg_p->mode)

	return fail_flg; // Return flag indicating outcome of test
} // check_message_bytes
/*****************************************************************************/
static void update_results( // Update results
	CHECK_TYP * check_p, // Get pointer to structure of Midi check data 
	VERIFY_TYP * verify_p // Pointer to current verification data 
)
{
	TST_VECT_TYP * chk_vect_p = &(verify_p->chk_vect_s); // Get local pointer to test-vector check data 
	TST_VECT_TYP * res_vect_p = &(verify_p->res_vect_s); // Get local pointer to test-vector result data 
	MIDI_PACKET_TYP * chk_pkt_p = verify_p->chk_vect_s.packet_p; // Get local pointer to packet check data 
	MIDI_PACKET_TYP * res_pkt_p = verify_p->res_vect_s.packet_p; // Get local pointer to packet result data 
	bool fail_flg = false; // Preset flag to successful

/*
printf("UR:V Res_Sts C=0x%02X N=0x%02X \n" ,res_vect_p->sts ,check_p->new_verify_s.res_vect_s.sts );
printf("UR:V Chk_Sts C=0x%02X N=0x%02X \n" ,chk_vect_p->sts ,check_p->new_verify_s.chk_vect_s.sts );
printf("UR:B Res_Sts C=0x%02X N=0x%02X \n" ,res_pkt_p->midi_status ,check_p->new_verify_s.res_vect_s.packet_p->midi_status );
printf("UR:B Chk_Sts C=0x%02X N=0x%02X \n" ,chk_pkt_p->midi_status ,check_p->new_verify_s.chk_vect_s.packet_p->midi_status );
*/
	fail_flg = check_vector_result( check_p ,verify_p );

	// Decide if Midi-message should be checked byte-for-byte
	if (RECEIVE == check_p->config_id)
	{ // checking Midi-message

		if (true == check_message_bytes( check_p ,verify_p ))
		{
			fail_flg = true; // Set flag to check failed
		} // if (true == check_message_bytes( check_p ,verify_p ))
	} // if (RECEIVE == check_p->config_id)

	if (fail_flg)
	{
		printf("  State Pkt_Sts Pkt_Len Status Param \n" );
		printf("Res: %2d    0x%02x     %2d   0x%02X  0x%08X \n" 
			,check_p->state ,res_pkt_p->midi_status ,res_pkt_p->data_len ,res_vect_p->sts ,res_vect_p->param );
		printf("Chk:       0x%02x     %2d   0x%02X  0x%08X \n" 
  		,chk_pkt_p->midi_status ,chk_pkt_p->data_len ,chk_vect_p->sts ,chk_vect_p->param ); // MB~
	} // if (fail_flg)

	check_p->burst_chk_cnt++; // Increment No. of checked messages in this burst
} // update_results
/*****************************************************************************/
static bool vectors_equal( // Compares 2 structures of test-vector data  
	TST_VECT_TYP * vect_res_p, // Pointer to structure of test-vector result data 
	TST_VECT_TYP * vect_chk_p // Pointer to structure of test-vector check data 
) // Returns true if vectors are equal
{
	// Compare each component of vector
	if (vect_res_p->param != vect_chk_p->param) return false;
	if (vect_res_p->sts != vect_chk_p->sts) return false;

	return true; // If we get this far vectors must be equal
} // vectors_equal 
/*****************************************************************************/
static void insert_missed_message( // Insert a Missed-message into display buffers
	CHECK_TYP * check_p, // Pointer to structure of Midi check data 
	S32_T curr_buf_off // Current offset into buffer of received Midi-messages 
)
{
	MIDI_RX_TYP * midi_rx_p = &(check_p->rx_data_s); // Get pointer to structure of received Midi data 
	S32_T prev_buf_off; // previous offset into buffer
	S32_T next_buf_off; // Next offset into buffer


	prev_buf_off = update_circular_buffer( curr_buf_off ,(-1) ,NUM_RX_BUFS ); // Get previous result buffer offset
	next_buf_off = update_circular_buffer( curr_buf_off ,1 ,NUM_RX_BUFS ); // Get next result buffer offset

	// Ripple buffer packets along by one, and then insert 'Missed message' packet
	midi_rx_p->pkt_bufs[next_buf_off] = midi_rx_p->pkt_bufs[curr_buf_off];
	midi_rx_p->pkt_bufs[curr_buf_off] = midi_rx_p->pkt_bufs[prev_buf_off];
	midi_rx_p->pkt_bufs[prev_buf_off].midi_status = MISSING_STS; // Signal Missing Midi-message
	
	// Re-assign test-vector result Midi-packet pointers
	check_p->new_verify_s.res_vect_s.packet_p = &(midi_rx_p->pkt_bufs[next_buf_off]);
	check_p->cur_verify_s.res_vect_s.packet_p = &(midi_rx_p->pkt_bufs[curr_buf_off]); 
	
	// Increment No. of 'checked' messages in this burst, after insertion of Missed-message
	check_p->burst_chk_cnt++;
} // insert_missed_message
/*****************************************************************************/
static void check_stream_integrity( // Tests for Missing/Extra messages
	CHECK_TYP * check_p, // Pointer to structure of Midi check data 
	bool new_res_eq_cur_chk // Flag indicating if new_result vector equals current_check vector
) // Return flag for do all checks
{
	TST_GEN_TYP * tst_gen_p = &(check_p->tst_gen_s); // Get local pointer to test generation data 
	VERIFY_TYP * new_verify_p = &(check_p->new_verify_s); // Get local pointer to new verification data 
	VERIFY_TYP * cur_verify_p = &(check_p->cur_verify_s); // Get local pointer to current verification data 
	RES_STATS_TYP * cur_stats_p = cur_verify_p->res_stats_p; // get local pointer to results statistics for current test-vector Midi-message-type
	TST_VECT_TYP * new_chk_p = &(new_verify_p->chk_vect_s); // Get local pointer to new test-vector check data 
	TST_VECT_TYP * new_res_p = &(new_verify_p->res_vect_s); // Get local pointer to new test-vector result data 
	TST_VECT_TYP * cur_res_p = &(cur_verify_p->res_vect_s); // Get local pointer to current test-vector result data 

	bool cur_res_eq_new_chk; // Flag set if current_result vector equals new_check vector
	bool new_res_eq_new_chk; // Flag set if new_result vector equals new_check vector


	// compare current result vector against new check vector
	cur_res_eq_new_chk = vectors_equal( cur_res_p ,new_chk_p ); 

	// Determine if Missing or Extra message received
	if (new_res_eq_cur_chk)
	{ // Possible Extra Message

		// Determine if extra spurious message
		if (cur_res_eq_new_chk)
		{
			printf("WARNING: Swapped Message\n");
			cur_stats_p->swapped++; // Increment swapped-message counter
		} // if (cur_res_eq_new_chk)
		else
		{
			printf("WARNING: Detected Extra Message\n");

			// Find Midi-message configuration data for Extra-Message
			cur_res_p->msg_cfg_p = status_to_config( check_p->msg_cfg_arr_p ,cur_res_p->packet_p->midi_status ,cur_res_p->packet_p->midi_data[0] );

			// Get pointer to results statistics for Extra-Message type
			cur_stats_p = &(check_p->res_stats_s[cur_res_p->msg_cfg_p->id]);

			cur_stats_p->extra++; // Increment extra-message counter
			check_p->extra_msg = true; // Set flag indicating Extra message found. NB Prevents update of check data
			check_p->valid_res= false; // Do NOT check extra message
		} // else !(cur_res_eq_new_chk)
	} // else (new_res_eq_cur_chk)
	else
	{ // Possible Missing Message

		// Determine if missing message
		if (cur_res_eq_new_chk)
		{ // Some type of Missing Message

			// compare new result and check vectors
			new_res_eq_new_chk = vectors_equal( new_res_p ,new_chk_p ); 
	
			// Determine if repeated result message
			if (new_res_eq_new_chk)
			{
				printf("WARNING: This Message Overwritten by Pre-Repeated Message\n");
				cur_stats_p->pre_repeats++; // Increment pre-repeated message counter
			} // if (new_res_eq_new_chk)
			else
	 		{
				printf("WARNING: Missed Message\n");

				insert_missed_message( check_p ,check_p->buf_off );

				cur_stats_p->missed++; // Increment missed-message counter
				check_p->missed_msg = true; // Set flag indicating Missed message detected. NB Prevents update of result data

				// NB Missing Message lost from stream, so skip check data as well!-)

				// Decide if end of test
				if (tst_gen_p->cnt < check_p->num_vects)
				{ // NOT end of test
					// Update just check data part of current verification data, ready for next message
					update_verify_data( check_p ,cur_verify_p ,new_verify_p );

					get_new_check_data( check_p ,new_verify_p ); // Get new check data
/*
printf("CSI:MV Res_Sts C=0x%02X N=0x%02X \n" ,cur_res_p->sts ,new_res_p->sts );
printf("CSI:MV Chk_Sts C=0x%02X N=0x%02X \n" ,cur_chk_p->sts ,new_chk_p->sts );
printf("CSI:MB Res_Sts C=0x%02X N=0x%02X \n" ,check_p->cur_verify_s.res_vect_s.packet_p->midi_status ,check_p->new_verify_s.res_vect_s.packet_p->midi_status );
printf("CSI:MB Chk_Sts C=0x%02X N=0x%02X \n" ,check_p->cur_verify_s.chk_vect_s.packet_p->midi_status ,check_p->new_verify_s.chk_vect_s.packet_p->midi_status );
*/
					// Check data now re-aligned with result data, so switch off Missed-message flag
					check_p->missed_msg = false;
				} // if (tst_gen_p->cnt < check_p->num_vects)
			} // else !(new_res_eq_new_chk)
		} // if (cur_res_eq_new_chk)
	} // else !(new_res_eq_cur_chk)

	return; // Return flag for do all checks
} // check_stream_integrity
/*****************************************************************************/
static void check_message_stream( // Top-level Midi-message checker, tests for Missing/Extra messages
	CHECK_TYP * check_p // Get pointer to structure of Midi check data 
)
{
	VERIFY_TYP * new_verify_p = &(check_p->new_verify_s); // Get local pointer to new verification data 
	VERIFY_TYP * cur_verify_p = &(check_p->cur_verify_s); // Get local pointer to current verification data 
	TST_VECT_TYP * new_chk_p = &(new_verify_p->chk_vect_s); // Get local pointer to new test-vector check data 
	TST_VECT_TYP * new_res_p = &(new_verify_p->res_vect_s); // Get local pointer to new test-vector result data 
	TST_VECT_TYP * cur_chk_p = &(cur_verify_p->chk_vect_s); // Get local pointer to current test-vector check data 
	TST_VECT_TYP * cur_res_p = &(cur_verify_p->res_vect_s); // Get local pointer to current test-vector result data 

	bool cur_res_eq_cur_chk; // Flag set if current_result vector equals current_check vector
	bool new_res_eq_cur_chk; // Flag set if new_result vector equals current_check vector
	bool cur_res_eq_new_chk; // Flag set if current_result vector equals new_check vector

/*
printf("CMS:S Res_Sts C=0x%02X N=0x%02X \n" ,cur_res_p->sts ,new_res_p->sts );
printf("CMS:S Chk_Sts C=0x%02X N=0x%02X \n" ,cur_chk_p->sts ,new_chk_p->sts );
printf("CMS:P Res_Param C=0x%02X N=0x%02X \n" ,cur_res_p->param ,new_res_p->param );
printf("CMS:P Chk_Param C=0x%02X N=0x%02X \n" ,cur_chk_p->param ,new_chk_p->param );
printf("CMS:B Res_Sts C=0x%02X N=0x%02X \n" ,check_p->cur_verify_s.res_vect_s.packet_p->midi_status ,check_p->new_verify_s.res_vect_s.packet_p->midi_status );
printf("CMS:B Chk_Sts C=0x%02X N=0x%02X \n" ,check_p->cur_verify_s.chk_vect_s.packet_p->midi_status ,check_p->new_verify_s.chk_vect_s.packet_p->midi_status );
*/
	check_p->extra_msg = false; // Preset to NO extra message
	check_p->missed_msg = false; // Preset to NO missed message

	// compare result vectors against current check vectors
	cur_res_eq_cur_chk = vectors_equal( cur_res_p ,cur_chk_p );
	new_res_eq_cur_chk = vectors_equal( new_res_p ,cur_chk_p );

	if (cur_res_eq_cur_chk)
	{ // Good current result

		// Determine if repeated message
		if (new_res_eq_cur_chk)
		{ // Possible repeated message

			// compare current result vector against new check vector
			cur_res_eq_new_chk = vectors_equal( cur_res_p ,new_chk_p ); 

			// Determine if repeated message
			if (0 == cur_res_eq_new_chk)
			{ // Repeated Message detected
				cur_verify_p->res_stats_p->pst_repeats++; // Increment post-repeated message counter
				printf("WARNING: Detected Post-Repeated Message\n");

				// NB Current result is Good, New result is Fail and will be handled on the next iteration
/*
printf("CMS:RV Res_Sts C=0x%02X N=0x%02X \n" ,cur_res_p->sts ,new_res_p->sts );
printf("CMS:RV Chk_Sts C=0x%02X N=0x%02X \n" ,cur_chk_p->sts ,new_chk_p->sts );
printf("CMS:RB Res_Sts C=0x%02X N=0x%02X \n" ,check_p->cur_verify_s.res_vect_s.packet_p->midi_status ,check_p->new_verify_s.res_vect_s.packet_p->midi_status );
printf("CMS:RB Chk_Sts C=0x%02X N=0x%02X \n" ,check_p->cur_verify_s.chk_vect_s.packet_p->midi_status ,check_p->new_verify_s.chk_vect_s.packet_p->midi_status );
*/
			} // if (0 == cur_res_eq_new_chk)
		} // if (new_res_eq_cur_chk)
	} // if (cur_res_eq_cur_chk)
	else
	{ // Bad current result, do stream checks
		check_stream_integrity( check_p ,new_res_eq_cur_chk ); // Do Midi-message stream integrity checks
	} // else !(cur_res_eq_cur_chk)

	// Determine if further checks are possible
	if (check_p->valid_res)
	{ // Check valid result data
		update_results( check_p ,cur_verify_p ); // Check current Midi-message 
	} // if (check_p->valid_res)

} // check_message_stream
/*****************************************************************************/
static MIDI_STATE_ENUM start_channel_message( // From status, evaluate No. of channel-message bytes (excluding status)
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T status // Received status byte
) // Returns new Midi-state. NB Currently always CHAN_MESS
{
	MIDI_PACKET_TYP * res_pkt_p = check_p->new_verify_s.res_vect_s.packet_p; // Get local pointer to new packet of Midi-message result data 
  U8_T msg_typ = (status & 0xF0); // NB 4 MS bits are channel-message type


	// Determine channel-message type
	switch(msg_typ)
	{
		case NOTE_OFF_STS: // Note-Off
			res_pkt_p->data_len = 2;
		break; // NOTE_OFF_STS

		case NOTE_ON_STS: // Note-On
			res_pkt_p->data_len = 2;
		break; // NOTE_ON_STS

		case KEY_PRESS_STS: // Poly Key pressure
			res_pkt_p->data_len = 2;
		break; // KEY_PRESS_STS

		case CONTROL_CHANGE_STS:	// Control Change
			res_pkt_p->data_len = 2;
		break; // CONTROL_CHANGE_STS

		case PROG_CHANGE_STS: // Program Change
			res_pkt_p->data_len = 1;
		break; // PROG_CHANGE_STS

		case CHAN_PRESS_STS:	// Channel pressure
			res_pkt_p->data_len = 1;
		break; // CHAN_PRESS_STS

		case PITCH_WHEEL_STS: // Key Pitch Bend
			res_pkt_p->data_len = 2;
		break; // PITCH_WHEEL_STS

		default:
			printf("ERROR: Unsupported  Midi channel-message %d\n" ,msg_typ );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(msg_typ)

	return CHAN_MESS;
} // start_channel_message
/*****************************************************************************/
static MIDI_STATE_ENUM start_system_message( // From status, determines No. of system-message bytes (excluding status)
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T status // Received status byte
) // Returns new Midi-state
{
	MIDI_PACKET_TYP * res_pkt_p = check_p->new_verify_s.res_vect_s.packet_p; // Get local pointer to new packet of Midi-message result data 


	// Determine system-message type
	switch(status)
	{
		case SONG_PTR_STS : // Song Pointer
			res_pkt_p->data_len = 2;
			return SYST_MESS;
		break; // SONG_PTR_STS

		case MIDI_TIME_STS: // Midi-Time Code
			res_pkt_p->data_len = 1;
			return SYST_MESS;
		break; // MIDI_TIME_STS

		case SONG_SEL_STS:	// Song Select
			res_pkt_p->data_len = 1;
			return SYST_MESS;
		break; // SONG_SEL_STS

		case TUNE_REQ_STS: // Tune Request
			res_pkt_p->data_len = 0; // No data bytes expected

			return END_MESS;
		break; // TUNE_REQ_STS

		default:
			printf("ERROR: Unsupported  Midi system-message %d\n" ,status );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(status)
} // start_system_message
/*****************************************************************************/
static void parse_real_time_message( // Calculate check result for Real-Time message
	CHECK_TYP * check_p  // Pointer to structure of Midi check data 
)
{
	MIDI_PACKET_TYP * res_pkt_p = check_p->new_verify_s.res_vect_s.packet_p; // Get local pointer to new packet of Midi-message result data 

	res_pkt_p->data_len = 0; // No data bytes expected

} // parse_real_time_message
/*****************************************************************************/
static MIDI_STATE_ENUM parse_message_data_bytes( // Expecting Midi-message data-byte
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T data_byt // Received input data byte
) // Returns Midi-state
{
	MIDI_RX_TYP * midi_rx_p = &(check_p->rx_data_s); // Get pointer to structure of received Midi data 
	TST_VECT_TYP * vect_res_p = &(check_p->new_verify_s.res_vect_s); // Get local pointer to test-vector result data 
	MIDI_PACKET_TYP * res_pkt_p = check_p->new_verify_s.res_vect_s.packet_p; // Get local pointer to new packet of Midi-message result data 


	// NB MS byte of check value sent first
	vect_res_p->param <<= MIDI_VAL1_BITS;
	vect_res_p->param += data_byt;

	res_pkt_p->midi_data[midi_rx_p->byte_id] = data_byt; // Store data byte
	midi_rx_p->byte_id++; // Increment byte identifier

	// Check for end of message
	if (midi_rx_p->byte_id == res_pkt_p->data_len)
	{ // End-of Midi message
		check_p->state = END_MESS; // Switch to Midi start state
	} // if (midi_rx_p->byte_id == res_pkt_p->data_len)

	return check_p->state; // Returns (possible new) Midi-state
} // parse_message_data_bytes
/*****************************************************************************/
static void parse_sysex_data( // Expecting system-exclusive data-byte
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T data_byt // Received input data byte
)
{
	MIDI_RX_TYP * midi_rx_p = &(check_p->rx_data_s); // Get pointer to structure of received Midi data 
	SYSEX_TYP * sysex_p = &(check_p->sysex_s); // Get local pointer to System-Exclusive decoding data 
	TST_VECT_TYP * vect_res_p = &(check_p->new_verify_s.res_vect_s); // Get local pointer to test-vector result data 
	MIDI_PACKET_TYP * res_pkt_p = check_p->new_verify_s.res_vect_s.packet_p; // Get local pointer to new packet of Midi-message result data 


	// Check for Manufacturers Id
	if (sysex_p->manu_id_len <= midi_rx_p->byte_id)
	{ // NOT Manufacturers ID
		// NB MS bytes of SysEx data sent first
		vect_res_p->param <<= sysex_p->shift_bits; // Shift left to make room for LS data bits
		vect_res_p->param += data_byt; // Add-in data bits at LS end of param
	} // if (sysex_p->manu_id_len <= midi_rx_p->byte_id)
	else
	{ // Manufacturers ID

		// Check for Short or Long Manufacturers Id
		if ((0 == data_byt) && (SMALL_MANU_ID == sysex_p->manu_id_len))
		{ // Long Manufacturers Id - Re-initialise decoding data for 'Large' SysEx 
			init_sysex_data( sysex_p ,LARGE_MANU_ID ,SYSEX_SHIFT_BITS ,SYSEX_BIT_MASK );
 		} // if ((0 == data_byt) && (1 == sysex_p->manu_id_len))
	} // else !(sysex_p->manu_id_len <= midi_rx_p->byte_id)

	res_pkt_p->midi_data[midi_rx_p->byte_id] = data_byt; // Store data byte
	midi_rx_p->byte_id++; // Increment byte identifier
// printf("PSD:P~2 Res_Param P=0x%08X B=0x%02X \n" ,check_p->new_verify_s.res_vect_s.param ,data_byt );
} // parse_sysex_data
/*****************************************************************************/
static MIDI_STATE_ENUM parse_system_exclusive_byte( // Expecting system-exclusive byte 
	CHECK_TYP * check_p,  // Pointer to structure of Midi check data 
  U8_T inp_byt // Received input byte
) // Returns Midi-state
	/*
	 * To Determine the type of System-Exclusive byte, the largest byte ID is tested 1st, 
   * and then in descending numerical order. This enables early returns to be used.
	 */
{
	MIDI_RX_TYP * midi_rx_p = &(check_p->rx_data_s); // Get pointer to structure of received Midi data 
	MIDI_PACKET_TYP * res_pkt_p = check_p->new_verify_s.res_vect_s.packet_p; // Get local pointer to new packet of Midi-message result data 


	// Check for Real-Time message
	if (END_SYSEX_STS < inp_byt)
	{	// This is allowed in the Midi-standard, but this should NOT occur during these tests
		printf("WARNING: Detected Real-time message, embedded in System-Exclusive message \n" );

		return check_p->state; // Returns current Midi-state
	} // if (END_SYSEX_STS < inp_byt)

	// Check for End of System-Exclusive message
	if (END_SYSEX_STS == inp_byt) 
	{
		res_pkt_p->midi_data[midi_rx_p->byte_id] = inp_byt; // Store 'data byte'
		midi_rx_p->byte_id++; // Increment byte identifier

		res_pkt_p->data_len = midi_rx_p->byte_id; // Assign input message length

		// Check for Invalid Sys-Ex message
		if (1 == res_pkt_p->data_len)
		{ // Empty Sys-Ex message
			check_p->valid_res = false; // Set flag to invalid Midi-message

			printf("WARNING: Empty System Exclusive message detected \n" );
		} // if (1 == res_pkt_p->data_len)

		return END_MESS; // Return next Midi-state
	} // if (END_SYSEX_STS == inp_byt) 

	// Check for System-Exclusive data
	if (inp_byt < NOTE_OFF_STS)
	{ // system exclusive data
		parse_sysex_data( check_p ,inp_byt );

		return check_p->state; // Returns current Midi-state
	} // if (inp_byt < NOTE_OFF_STS)

  // If we get this far, System-Exclusive byte has an Illegal value
	printf("ERROR: Expected System Exclusive Byte Found %d\n" ,inp_byt );
	printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
	exit(-1);

} // parse_system_exclusive_byte
/*****************************************************************************/
static void init_midi_receive( // Initialise structure of received Midi data ready for new packet
	CHECK_TYP * check_p, // Get pointer to structure of Midi check data 
  U8_T sts_byte // Received Status byte
)
/* 
 * By successive calls of update_midi_state() the Callback data will be copied to 
 * an empty slot in the packet buffer, so we can display later at end of test vector. 
 * A pointer to the current buffer slot is maintained.
 */
{
	MIDI_RX_TYP * midi_rx_p = &(check_p->rx_data_s); // Get pointer to structure of received Midi data 
	TST_VECT_TYP * vect_res_p = &(check_p->new_verify_s.res_vect_s); // Get local pointer to test-vector result data 
	MIDI_PACKET_TYP * new_pkt_p; // Pointer to new packet of Midi-message result data 


	// Check for buffer overflow
	if (NUM_RX_BUFS <= check_p->buf_off)
	{ // Buffer Overflowed
		printf("ERROR: Receive Buffer Overflowed\n" );
		printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
		exit(-1);
	} // if (NUM_RX_BUFS <= check_p->buf_off)

	// Pointer to next space in Midi packet buffer
	new_pkt_p = &(midi_rx_p->pkt_bufs[check_p->buf_off]);
	check_p->new_verify_s.res_vect_s.packet_p = new_pkt_p; // Store new pointer

	new_pkt_p->diff_time = check_p->diff_time; // Store elapsed time since last message in packet buffer
	check_p->diff_time = 0; // Clear accumulated time differences

	midi_rx_p->byte_id = 0; // Reset position of 1st byte in midi-message
	check_p->valid_res = true; // Preset to valid Midi-message

	// Store status byte
	new_pkt_p->midi_status = sts_byte; // Update received packet Midi-status
	vect_res_p->sts = sts_byte; // Store status byte result for test-vector processing (NB maybe altered)

	return;
} // init_midi_receive
/*****************************************************************************/
static MIDI_STATE_ENUM check_for_message_start( // Checks for start of new Midi-message, and updates Midi-state
	CHECK_TYP * check_p, // Get pointer to structure of Midi check data 
  U8_T inp_byte // Received Input byte
) // Returns next Midi-state
// NB A Midi-message starts with a 'status byte'
{
	MIDI_STATE_ENUM new_state = check_p->state; // Preset new Midi-state to old value


	// Found start of new Midi-message ...

	init_midi_receive( check_p ,inp_byte );	// Initialise structure of received Midi data ready for new packet

	/*
	 * To Determine the type of Midi-message, the largest status ID is tested 1st, 
   * and then in descending numerical order. This enables early returns to be used.
	 */

	// Check for System Real-Time message
	if (END_SYSEX_STS < inp_byte)
	{ // System Real-Time message
		parse_real_time_message( check_p );

		return END_MESS;
	} // if (END_SYSEX_STS < inp_byte)

	// Check for (Non-SysEx) System-common message
	if (STRT_SYSEX_STS < inp_byte) 
	{	// (Non-SysEx) System-common message
		new_state = start_system_message( check_p ,inp_byte );

		return new_state;
	} // if (STRT_SYSEX_STS < inp_byte) 

	// Check for System-Exclusive message
	if (STRT_SYSEX_STS == inp_byte)  
	{	// System-Exclusive message
		// Initialise for 'Small' System-Exclusive decoding data (NB This may be changed during SysEx message parsing)
		init_sysex_data( &(check_p->sysex_s) ,SMALL_MANU_ID ,MIDI_VAL1_BITS ,MSK_7BIT );

// printf("CFMS:P Res_Param P=0x%08X S=0x%02X \n" ,check_p->new_verify_s.res_vect_s.param ,inp_byte );
		return EXCL_MESS;
	} // if (STRT_SYSEX_STS == inp_byte)  

	// Check for channel-message
	if (MSK_7BIT < inp_byte)
	{ // Channel-message
		new_state = start_channel_message( check_p ,inp_byte );

		return new_state;
	} // if (MSK_7BIT < inp_byte)

	assert( 0 == 1 ); // Unreachable statement

	return START_MESS; // Wait for start of new message
} // check_for_message_start
/*****************************************************************************/
static void abort_message( // Abort incompletely received Midi-message
	CHECK_TYP * check_p, // Pointer to structure of Midi check data 
	VERIFY_TYP * verify_p // Pointer to verification data for aborted message
)
{
	TST_VECT_TYP * vect_res_p = &(verify_p->res_vect_s); // Get local pointer to vector result data 
	MIDI_PACKET_TYP * pkt_res_p = vect_res_p->packet_p; // Get local pointer to packet of Midi-message result data 


	printf("WARNING: Aborting Following Incompletely Received Message \n");
	print_midi_message( check_p->msg_cfg_arr_p ,pkt_res_p ,&(check_p->bits) ); // print a Midi-message as binary digits

	clear_vector_data( vect_res_p ); // Clear redundant test vector result data 
} // abort_message 
/*****************************************************************************/
static void process_message( // Processing after new Midi-message received
	CHECK_TYP * check_p // Get pointer to structure of Midi check data 
)
{
	VERIFY_TYP * new_verify_p = &(check_p->new_verify_s); // Get local pointer to new verification data 
	VERIFY_TYP * cur_verify_p = &(check_p->cur_verify_s); // Get local pointer to current verification data 


	// Determine if the previous current result was an Extra message
	if (0 == check_p->extra_msg)
	{ // NOT an Extra message
		get_new_check_data( check_p ,new_verify_p ); // Get test-vector check data for new Midi-message
	} // if (0 == check_p->extra_msg)

	// Determine if first message
	if (check_p->first_msg)
	{ // 1st message
		check_p->first_msg = false; // Switch OFF first message flag
	} // if (check_p->first_msg)
	else
	{ // Now we have at least 2 messages. stream integrity can be checked
		check_message_stream( check_p ); // Top-level Midi-message checker
	} // else !(check_p->first_msg)

	// Update current verification data ready for next message
// printf("       PM NORM res_sts=0x%02X  chk_sts=0x%02X \n" ,cur_verify_p->res_vect_s.sts ,cur_verify_p->chk_vect_s.sts ); 
	update_verify_data( check_p ,cur_verify_p ,new_verify_p );

	return;
} // process_message
/*****************************************************************************/
static void process_status_byte( // Process a Midi status byte (0x80..0xFF)
	CHECK_TYP * check_p, // Get pointer to structure of Midi check data 
  U8_T sts_byte // Received status byte
)
{
	VERIFY_TYP * new_verify_p = &(check_p->new_verify_s); // Get local pointer to new verification data 


	// Determine current Midi state
	switch(check_p->state)
	{
		case START_MESS: // Start a new message
			check_p->state = check_for_message_start( check_p ,sts_byte ); // Get new Midi-state
		break; // START_MESS

		case CHAN_MESS: // Incomplete Channel Message
			abort_message( check_p ,new_verify_p ); // Abort current message 
			check_p->state = check_for_message_start( check_p ,sts_byte ); // Get new Midi-state
		break; // CHAN_MESS 

		case SYST_MESS: // Incomplete System-Common Message
			abort_message( check_p ,new_verify_p ); // Abort current message 
			check_p->state = check_for_message_start( check_p ,sts_byte ); // Get new Midi-state
		break; // SYST_MESS 

		case EXCL_MESS: // Possibly incomplete System-exclusive message
			// Determine if Valid status byte
			if (END_SYSEX_STS == sts_byte)
			{ // Valid status byte
				check_p->state = parse_system_exclusive_byte( check_p ,sts_byte );
			} // if (END_SYSEX_STS == sts_byte)
			else
			{ // Invalid status byte
				abort_message( check_p ,new_verify_p ); // Abort current message 
				check_p->state = check_for_message_start( check_p ,sts_byte ); // Get new Midi-state
			} // if (END_SYSEX_STS == sts_byte)
		break; // EXCL_MESS 

		case END_TEST: // End of test, ignore any spurious extra messages
			printf("WARNING: After Test Finished, received Status byte 0x%0X \n" ,sts_byte );
			check_p->state = END_TEST; // Absorbing state
		break; // END_TEST

		case END_MESS : case TIME_MESS :
			printf("ERROR: Unreachable state=%d \n" ,check_p->state );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // END_MESS

		default:
			printf("ERROR: Unsupported  Midi State %d\n" ,check_p->state );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(check_p->state)

} // process_status_byte
/*****************************************************************************/
static void process_data_byte( // Process a Midi data byte (0x00..0x7F)
	CHECK_TYP * check_p, // Get pointer to structure of Midi check data 
  U8_T data_byte // Received data byte
)
{
	// Determine current Midi state
	switch(check_p->state)
	{
		case START_MESS: // Start a new message
			printf("WARNING: Skipping NON Start-of-message byte=0x%0X \n" ,data_byte );
		break; // START_MESS

		case CHAN_MESS: // Continue parsing channel-message
			check_p->state = parse_message_data_bytes( check_p ,data_byte );
		break; // CHAN_MESS 

		case SYST_MESS: // Continue parsing System-message
			check_p->state = parse_message_data_bytes( check_p ,data_byte );
		break; // SYST_MESS 

		case EXCL_MESS: // Continue parsing System-exclusive message
			check_p->state = parse_system_exclusive_byte( check_p ,data_byte );
		break; // EXCL_MESS 

		case END_TEST: // End of test, ignore any spurious extra messages
			printf("WARNING: After Test Finished, received Data byte 0x%0X \n" ,data_byte );
			check_p->state = END_TEST; // Absorbing state
		break; // END_TEST

		case END_MESS : case TIME_MESS :
			printf("ERROR: Unreachable state=%d \n" ,check_p->state );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // END_MESS

		default:
			printf("ERROR: Unsupported  Midi State %d\n" ,check_p->state );
			printf("       In file %s,  at line %d\n" ,__FILE__ ,__LINE__ );
			exit(-1);
		break; // default
	} // switch(check_p->state)

} // process_data_byte
/*****************************************************************************/
static void update_midi_state( // Update Midi state based on value of input byte
	CHECK_TYP * check_p, // Get pointer to structure of Midi check data 
  U8_T inp_byte // Received input byte
)
{
	TST_GEN_TYP * tst_gen_p = &(check_p->tst_gen_s); // Get local pointer to test generation data 
	VERIFY_TYP * cur_verify_p = &(check_p->cur_verify_s); // Get local pointer to current verification data 


	// Determine if Midi status byte
	if (MSK_7BIT < inp_byte)
	{ // Midi status byte
		process_status_byte( check_p ,inp_byte );
	} // if (MSK_7BIT < inp_byte)
	else
	{ // Midi data byte
		process_data_byte( check_p ,inp_byte );
	} // if (MSK_7BIT < inp_byte)

	// Determine if end-of-message
	if (END_MESS == check_p->state)
	{ // End-of-message

		// Determine if newly received Midi-message requires processing
		if ((check_p->valid_res) && (check_p->state != END_TEST))
		{ // Valid Midi-message
			process_message( check_p ); // Process newly received  Midi-message (e.g. checking)
		} // if (check_p->valid_res)

		// Decide if end of test
		if (tst_gen_p->cnt < check_p->num_vects)
		{ // NOT End-of_tests
			check_p->state = START_MESS; // Wait for start of new message
		} // if (tst_gen_p->cnt < check_p->num_vects)
		else
		{ // End-of_tests
			update_results( check_p ,cur_verify_p ); // Check last Midi-message 

			check_p->state = END_TEST; // Absorbing state
		} // else !(tst_gen_p->cnt < check_p->num_vects)
	} // if (END_MESS == check_p->state)
} // update_midi_state
/*****************************************************************************/
void mycallback( // Callback function, executed when Midi-In port receives data
	R64_T deltatime, // difference between time stamps
	std::vector< U8_T > * msg_p, // Pointer to vector containing received data
	void * void_ptr // userData
)
/* This callback code should be resilient to the following events:-
 * (1) The 1st byte received is NOT a status byte, the parser dumps bytes until a status byte (message start) is found
 * (2) Midi-messages can be spread across a number of RtMidi packets.
 * (3) After a stream error, the parser re-synchronises on the next status byte (message start)
 * (4) A Missing Midi-message
 * (5) An Extra Midi-message
 * (6) A Repeated Midi-message (both pre-repeated and post-repeated types)
 * (7) A pair of Swapped Midi-messages
 * (8) Spurious 0xF0FE (empty SysEx) messages
 * (9) Extra bytes after the end of the expected tests
 */
{
	CHECK_TYP * check_p = (CHECK_TYP *)void_ptr; // Retrieve pointer to structure of Midi check data 
  U32_T nBytes = msg_p->size(); // Get number of message bytes received
  U8_T inp_byte; // Received data byte


	check_p->diff_time += deltatime; // Update difference between Midi-message time stamps

	// Check for Empty message
	if (0 == nBytes)
	{
		printf("ERROR: Empty RtMidi Packet found\n");
		return;
  } // if ( nBytes > 0 )

	if (UCHAR_MAX < nBytes)
	{
		printf("ERROR: RtMidi Packet greater than 255 found\n");
    return;
	} // if (UCHAR_MAX < nBytes)

	// Sensible message length so continue

	// Loop through received bytes
  for ( U32_T byte_cnt=0; byte_cnt<nBytes; byte_cnt++ )
	{
		inp_byte = msg_p->at(byte_cnt); // Get next input byte

// if (inp_byte == 0x9F) inp_byte = 0x7F; // MB~ Test of Skip to start-of-message

#ifdef MB // MB~ Test of Abort Incomplete message
if (inp_byte == 0xF0)
{ // Insert a new message
	update_midi_state( check_p ,0xF0 ); // Update Midi state based on new received byte
	update_midi_state( check_p ,0x00 ); // Update Midi state based on new received byte
	update_midi_state( check_p ,0x02 ); // Update Midi state based on new received byte
	update_midi_state( check_p ,0x02 ); // Update Midi state based on new received byte
	update_midi_state( check_p ,0x3F ); // Update Midi state based on new received byte
printf("0xF0 0x02 0x3F ");
} // if (inp_byte == 0x1A)
#endif //MB~

// printf("0x%02X " ,inp_byte ); if ((byte_cnt + 1) == nBytes)  printf(":CB\n" ); //MB~

		update_midi_state( check_p ,inp_byte ); // Update Midi state based on new received byte
  } // for byte_cnt
} // mycallback
/*****************************************************************************/
// xmidi_callback
