/*
 * Copyright XMOS Limited - 2014
 */

#define _CRT_SECURE_NO_DEPRECATE 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "Socket_lite.h"

#define BACKLOG 10

static unsigned int s_numSockets = 0;

#if defined(WIN32)
#define socklen_t int
#endif

/*****************************************************************************/
Socket::Socket() : m_blocking(true)
{
#if defined(WIN32)
  WORD wVersionRequested;
  WSADATA wsaData;
  int err;

  wVersionRequested = MAKEWORD(2, 2);

  err = WSAStartup( wVersionRequested ,&wsaData );
  if (err != 0)
	{
    fprintf( stderr ,"ERROR in Socket(): WSAStartup() unable to create socket. Err_No=%d \n" ,err );
    perror("ERROR in Socket(): WSAStartup() unable to create socket");
  } // if (err != 0)

#endif

  m_socketId = socket(AF_INET, (int)SOCK_STREAM, 0);
  if (m_socketId == INVALID_SOCKET_ID)
	{
    fprintf( stderr , "ERROR in Socket(): socket() unable to create socket. INVALID_SOCKET_ID=%d \n" ,m_socketId );
		perror("ERROR in Socket(): socket() returned invalid INVALID_SOCKET_ID");
  } // if (m_socketId == INVALID_SOCKET_ID)

#if !defined(WIN32)
  int yes = 1;
  if (setsockopt( m_socketId, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(int)) == -1)
	{
    fprintf( stderr ,"ERROR in Socket(): setsockopt() failed to set SO_REUSEADDR socket option\n");
		perror("ERROR in Socket(): setsockopt() failed");
  }
#endif

  m_id = s_numSockets;
  ++s_numSockets;
} // Socket::Socket() 
/*****************************************************************************/
Socket::~Socket()
{

 fprintf( stderr ,"Closing socket_%d \n" ,m_id ); //MB~
#if defined(WIN32)
  closesocket(m_socketId);
#else
  if ( -1 == shutdown( m_socketId ,SHUT_RDWR ))
	{
    fprintf( stderr ,"ERROR in ~Socket(): Failed to SHUTDOWN Socket_%d " ,m_id );
		perror("ERROR in ~Socket(): shutdown failed");
  } // if ( -1 == close(m_socketId)

  if ( -1 == close(m_socketId))
	{
    fprintf( stderr ,"ERROR in ~Socket(): Failed to close Socket_%d " ,m_id );
		perror("ERROR in ~Socket(): close failed");
  } // if ( -1 == close(m_socketId)

#endif
} // Socket::~Socket()
/*****************************************************************************/
char *Socket::getIpAddressFromHostName(const char *ipAddress)
{
  char *ipAddressString = 0;

  struct hostent *remoteHost = gethostbyname(ipAddress);
  if (remoteHost == NULL)
    return 0;

  ipAddressString = inet_ntoa(*(struct in_addr *)*remoteHost->h_addr_list);

  return ipAddressString;
} // *Socket::getIpAddressFromHostName
/*****************************************************************************/
bool Socket::socBind( 
	const char * ipAddress, 
	const char * port
) // Returns status flag (0:success)
{
  unsigned int portNumber = 0;


  if (sscanf( port ,"%d" ,&portNumber ) != 1)
	{
    fprintf( stderr ,"ERROR in socBind(): sscanf() failed, ipaddress: %s, port: %s\n" ,ipAddress ,port );
    perror("ERROR in socBind(): sscanf() failed");
		return true; // Return status flag = failed
	} // if (sscanf(port, "%d", &portNumber) != 1)

  char *ipAddressString = getIpAddressFromHostName(ipAddress);
  if (ipAddressString == 0)
	{
    fprintf( stderr ,"ERROR in socBind(): getIpAddressFromHostName() failed. ipaddress: %s, port: %s\n" ,ipAddress ,port );
    perror("ERROR in socBind(): getIpAddressFromHostName() failed");
		return true; // Return status flag = failed
  } // if (ipAddressString == 0)

  struct sockaddr_in socketAddress;
  socketAddress.sin_family = AF_INET;
  socketAddress.sin_port = htons((unsigned short)portNumber);
  socketAddress.sin_addr.s_addr = inet_addr(ipAddressString);
  memset(socketAddress.sin_zero, '\0', sizeof(socketAddress.sin_zero));

  if (-1 == bind(m_socketId, (struct sockaddr *)&socketAddress, sizeof(socketAddress)))
	{
    fprintf( stderr ,"ERROR in socBind(): bind() failed for socket_%d " ,m_id );
    perror("ERROR in socBind(): bind() failed");

		return true; // Return status flag = failed
  } // if (-1 == bind(m_socketId, (struct sockaddr *)&socketAddress, sizeof(socketAddress)))

	return false; // Return status flag = success
} // Socket::socBind
/*****************************************************************************/
void Socket::socListen()
{
  if (listen(m_socketId, BACKLOG) == -1)
	{
    fprintf( stderr ,"ERROR in socListen(): listen() failed for socket_%d " ,m_id );
    
    perror("ERROR in socListen(): listen() failed");
  } // if (listen(m_socketId, BACKLOG) == -1)
} // Socket::socListen()
/*****************************************************************************/
void Socket::socAccept()
{
  struct sockaddr_in acceptAddress;
  int sinSize = sizeof(acceptAddress);

  SOCKET_ID acceptSocketId = accept(m_socketId, (struct sockaddr *)&acceptAddress, (SOCKLEN *)&sinSize);

  if ((acceptSocketId == -1) && m_blocking)
	{
    fprintf( stderr ,"ERROR in socAccept(): accept() failed. for socket_%d " ,m_id );
    perror("ERROR in socAccept(): accept() failed");
  } // if ((acceptSocketId == -1) && m_blocking)

  m_socketId = acceptSocketId; // Update active socket Id
} // Socket::socAccept()
/*****************************************************************************/
int Socket::socReceive(char *buffer, unsigned int bufferLength)
{
  int numBytes;
  if (m_blocking) 
	{
    numBytes = recv(m_socketId, buffer, bufferLength, 0);
  } else 
	{
#if defined(__LINUX__) || defined(__linux__) || defined (__CYGWIN__) || defined (__APPLE__)
    numBytes = recv(m_socketId, buffer, bufferLength, MSG_DONTWAIT);
#elif defined(WIN32)
    // Not actually tested non-blocking on Windows machines, but it won't build with the new recv() format
    numBytes = recv(m_socketId, buffer, bufferLength, 0);
#else
    /* Error - get compiler to die */
    #include "UnsupportedPlatform"
#endif
  }

  if ((numBytes == -1) && m_blocking)
  {
    fflush(stdout);
  #if defined(__LINUX__) || defined(__linux__) || defined (__CYGWIN__) || defined (__APPLE__)
    usleep (10000);
  #endif
    fprintf( stderr ,"ERROR in socReceive(): recv() failed for socket_%d " ,m_id );
    perror("ERROR in socReceive(): recv() failed" );
  }

  return numBytes;
} // Socket::socReceive
/*****************************************************************************/
bool Socket::socConnect( 
	const char *ipAddress, 
	const char *port
) // Returns status flag (0:success)
{
  unsigned int portNumber = 0;


  if (1 != sscanf(port, "%d", &portNumber))
	{
    fprintf( stderr ,"ERROR in socConnect(): sscanf() failed. ipaddress: %s, port: %s\n" ,ipAddress ,port );
    perror("ERROR in socConnect(): sscanf() failed");
		return true; // Return status flag = failed
  } // if (1 != sscanf(port, "%d", &portNumber))

  char *ipAddressString = getIpAddressFromHostName(ipAddress);
  if (0 == ipAddressString)
	{
    fprintf( stderr ,"ERROR in socConnect(): getIpAddressFromHostName() failed. ipaddress: %s, port: %s\n" ,ipAddress ,port );
    perror("ERROR in socConnect(): getIpAddressFromHostName() failed");
		return true; // Return status flag = failed
  } // if (0 == ipAddressString)

  struct sockaddr_in destinationAddress;
  destinationAddress.sin_family = AF_INET;
  destinationAddress.sin_port = htons((unsigned short)portNumber);
  destinationAddress.sin_addr.s_addr = inet_addr(ipAddressString);
  memset(destinationAddress.sin_zero, '\0', sizeof(destinationAddress.sin_zero));

  if (-1 == connect(m_socketId, (struct sockaddr *)&destinationAddress, sizeof(destinationAddress)))
	{
    fprintf( stderr ,"WARNING in socConnect(): connect() failed. ipaddress: %s, port: %d\n" ,ipAddress ,portNumber );
    perror("WARNING in socConnect(): connect() failed");
		return true; // Return status flag = failed
  } // if (-1 == connect(m_socketId, (struct sockaddr *)&destinationAddress, sizeof(destinationAddress)))

	return false; // Return status flag = success
} // Socket::socConnect
/*****************************************************************************/
int Socket::socSend(const char *buffer ,unsigned int bufferLength)
{
  int bytesSent = send( m_socketId ,buffer ,bufferLength ,0 );

  if (bytesSent == -1)
	{
    fprintf( stderr ,"ERROR in socSend(): send() failed for socket_%d " ,m_id );
    perror("ERROR in socSend(): send() failed");
  } // if (bytesSent == -1)

  return bytesSent;
} // Socket::socSend 
/*****************************************************************************/
// Socket_lite.cpp
