#! /usr/bin/env python
#
# Description: This is the stand-alone script for running all Midi-tests on all platform configurations
#    An instance of this script will be run on each H/W configuration
#    The (remote) target platform under test is supplied as an input argument one of [Windows7 Mac Linux]
#    Currently, the (local) host platform is hardwired as 'Linux'
#
#    This script-file can be called directly, or do_all_tsts() can be called seperately  
#    as part of a suite of tests (Currently the MFA test suite).
#    The MFA test suite conforms to a general test method used for all usb-audio_testing (UAT).
#
# Author: J.M.Beaumont
# Date:  20-AUG-2014

import datetime
import json
import os
import socket
import subprocess
import sys
import time

from global_data import ConfigClass
from global_data import GlobalClass
from global_data import PlatformClass
from run_executable import run_executable 

scripts_path = os.path.dirname(os.path.realpath(__file__))  # Get path to scripts (current) directory
(midi_path ,scripts_dir) = os.path.split(scripts_path) # split off scripts directory
(repo_path ,midi_dir) = os.path.split(midi_path) # split off midi directory
(root_path ,repo_dir) = os.path.split(repo_path) # split off repository directory

sys.path.append(os.path.join(root_path,'usb_audio_testing/scripts'))

###############################################################################
def log_results( log_file_obj ,test_name ,num_fails ):
    """Records the results of one test in the log-file

    DESCRIPTION: This function is called as part of the general usb-audio_testing (UAT) method
    This function writes a time-stamp, the test-name and the test-result to the global log file
    """

    # Get time-stamp as Hours:Minutes:Seconds.Microseconds
    time_str = str(datetime.datetime.now().time())

    # Determine outcome of test
    if (0 < num_fails):
        result_str = ' FAILED'
    else:
        result_str = ' PASSED'

    file_str = time_str + ': ' + test_name + result_str  # Build result-line to write to file
    log_file_obj.write( file_str + '\n' ) # Write result-line to file

# def log_results
###############################################################################
def unidirectional_tst( host_config_dict ,targ_config_dict ,host_tx ,port_str ,log_file_obj ):
    """Test Midi communications in one direction between 2 platforms

    DESCRIPTION: A (local) Host platform (currently Linux) and a remote platform are tested
    in one of the following 2 configurations, depending on the value of flag host_tx:-
    (1) The (local) Host platform Transmits and the Remote platform Receives
    (2) The (local) Host platform Receives and the Remote platform Transmits
    """

    glob_data = GlobalClass()  # Initialise global data (See global_data.py)

    host_os_type_str = host_config_dict['os_type'] 
    targ_os_type_str = targ_config_dict['os_type'] 

    cfg_data = ConfigClass( host_os_type_str ,targ_os_type_str ,host_tx )

    if (True == host_tx):
        # Midi-transmit is on (local) host machine 
        host_data = cfg_data.tx_plat
        targ_data = cfg_data.rx_plat  # Get remote-platform data from Receiving Platform
    elif (False == host_tx):
        # Midi-transmit is on (remote) target machine 
        host_data = cfg_data.rx_plat # Do Midi-receive test on (local) Host platform
        targ_data = cfg_data.tx_plat  # Get remote-platform data from Transmiting Platform
    else:
        print >>sys.stderr, "ERROR. Unrecognised logical value: " ,cfg_data.host_tx
        sys.exit()  # Force exit

    # Build Script-file directory path as seen from remote platform
    targ_scripts_dir = targ_config_dict['root_path'] + glob_data.midi_dir + glob_data.scripts_dir
    
    # Define command to run Midi executable on remote platform
    targ_cmd = targ_data.lsh_cmd + targ_scripts_dir + targ_data.script_name + ' '

    # Build (local) Host command to enable execution of remote command on remote platform
    targ_ip_addr = targ_config_dict['ip_addr'] + ' '
    targ_user_name = targ_config_dict['usr_name'] + '@'
    sys_cmd = targ_data.ssh_cmd + targ_user_name + targ_ip_addr + targ_cmd + port_str + ' &'

    # E.g. sys_cmd = 'ssh 10.0.102.157 cmd /C H:/git.dir/midi_test.dir/sw_rtmidi/app_xmidi_rtmidi/scripts.dir/lnx2plat_rx_tst.py &'
    return_sts = subprocess.call( sys_cmd , shell=True )  # Spawn Midi-receive executable on remote platform
    if (0 != return_sts):
        print >>sys.stderr, "FAILED to run Midi-receive executable on remote Windows7 platform", return_sts
        sys.exit()  # Force exit
    
    # Run Midi executable on (local) Host platform ...

    num_fails = -1; # Preset to non-zero
    num_fails = run_executable( host_data ,targ_ip_addr ,port_str ) # Do Midi-test on (local) Host platform
    
    # If this script is called directly, there are only 2 arguments,
    # otherwise is is assumed this script is being called from a UAT test suite

    # Check if log-file being used
    if (log_file_obj):
        log_results( log_file_obj ,cfg_data.res_log_name ,num_fails ) # Write results to log-file

    return
# def unidirectional_tst
###############################################################################
def bidirectional_tsts( targ_config_dict ,host_config_dict ,port_str ,log_file_obj ):
    """Test Midi communications in both directions between 2 platforms

    DESCRIPTION: A (local) Host platform (currently Linux) and a remote platform are tested
    in the following 2 configurations:-
    (1) The (local) Host platform Transmits and the Remote platform Receives (tx_host=True)
    (2) The (local) Host platform Receives and the Remote platform Transmits (tx_host=False)
    """

    # Do Host_Tx-->Remote_Rx unidirectional test
    unidirectional_tst( host_config_dict ,targ_config_dict ,True ,port_str ,log_file_obj )

    # Do Remote_Tx-->Host_Rx unidirectional test
    unidirectional_tst( host_config_dict ,targ_config_dict ,False ,port_str ,log_file_obj )

# def bidirectional_tsts
###############################################################################
def do_midi_tsts( targ_config_dict ,host_config_dict ,log_file_obj ):
    """Do initialisation and then call a function to do a set of bi-directional Midi tests between 2 platforms

    DESCRIPTION: A target (remote) platform and a host (local) platform (currently Linux) are tested as follows:-
    Check for valid platforms,
    Configure platform data,
    Open socket between 2 platforms,
    Run set of bi-directional tests,
    If log_file_obj exists, write test results to the log-file
    """


    # Get OS-type strings for host & target platforms
    host_os_type_str = host_config_dict['os_type']
    targ_os_type_str = targ_config_dict['os_type']
    
    platform_set = ['Linux' ,'Mac' ,'Windows7'] # set of supported platforms
    
    # Check for valid target platform
    if (targ_os_type_str in platform_set):
        print 'Running MIDI tests for target platform:' ,targ_os_type_str
    else:
        print >>sys.stderr, 'ERROR. Unknown platform' ,targ_os_type_str ,'Expected one of' ,platform_set
        sys.exit()  # Force exit
    
    # Check for valid host platform
    if (host_os_type_str in platform_set):
        print '                      On host platform:' ,host_os_type_str
    else:
        print >>sys.stderr, 'ERROR. Unknown platform' ,host_os_type_str ,'Expected one of' ,platform_set
        sys.exit()  # Force exit

    # Configure platform data
    host_data = PlatformClass( host_os_type_str ) # Initialise host data (See global_data.py)
    glob_data = GlobalClass()  # Initialise global data (See global_data.py)
    
    # Build input-file directory path as seen from Host platform
    host_inputs_dir = host_config_dict['root_path'] + glob_data.midi_dir + glob_data.inputs_dir
    
    # This section finds an unused ethernet port ...
    
    my_sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    my_sock.bind(('localhost', 0))  # NB 0 signals get unused random port
    my_addr, my_port = my_sock.getsockname()  # Get address and number for port
    my_sock.close()

    port_str = str(my_port) # Convert port_number to string

    # Write Port-number to file. NB This will be read by host-side Midi executables
    file_name = host_inputs_dir + glob_data.port_file_name
    out_file = open(file_name ,'w')
    out_file.write( port_str + '\n' )
    out_file.close()
    
    # Do set of bidirectional Midi tests
    bidirectional_tsts( targ_config_dict ,host_config_dict ,port_str ,log_file_obj )
    
    time.sleep(1) # Sleep for 1 second to allow last test to complete
# def do_midi_tsts
###############################################################################
# start of main

# NB This 'if statement' prevents main() being run when functions are called
if __name__ == "__main__":
    # Import some functions from midi_tests.py ...
    from midi_test import get_uat_args
    from midi_test import get_configuration_data


    arg_len = len(sys.argv) # Get number of command line arguments (including script-name)
    
    if (3 != arg_len):
        print >>sys.stderr, "ERROR. Expected only two argument: the json-file-name & the target-platform-id"
        sys.exit()  # Force exit

    # Get input data from command-line arguments
    uat_args_obj = get_uat_args()    # Get UAT arguments object
    
    # Get configuration dictionaries from UAT arguments object
    (targ_config_dict ,host_config_dict) = get_configuration_data( uat_args_obj )

    log_file_obj = {} # Create empty (un-used) log-file object

    # Do all midi tests for current configuration
    do_midi_tsts( targ_config_dict ,host_config_dict ,log_file_obj )
    
    sys.exit()  # Force exit
# if __name__ == "__main__":

# end of main
###############################################################################
