import re


def check_analyzer_output(analyzer_output, xsig_config):
    """Verify that the output from xsig is correct"""

    def get_line_matches(lines, expected):
        matches = []
        for line in lines:
            match = re.search(expected, line)
            if match:
                matches.append(match.group(1))
        return matches

    analyzer_output = analyzer_output.splitlines()

    failures = []
    # Check for any errors
    for line in analyzer_output:
        if re.match(".*(?<!no )(error|problem)", line.lower()):
            failures.append(line)

    num_chans = len(xsig_config)
    analyzer_channels = [[] for _ in range(num_chans)]
    for line in analyzer_output:
        match = re.search(r"^Channel (\d+):", line)
        if not match:
            continue

        channel = int(match.group(1))
        if channel not in range(num_chans):
            failures.append(f"Invalid channel number {channel}")
            continue

        analyzer_channels[channel].append(line)

        if re.match(r"Channel \d+: Lost signal", line):
            failures.append(line)

    for idx, channel_config in enumerate(xsig_config):
        if channel_config[0] == "volcheck":
            vol_changes = get_line_matches(analyzer_channels[idx], r".*Volume change by (-?\d+)")

            if len(vol_changes) < 2:
                failures.append(f"Initial volume and initial change not found on channel {idx}")
                continue

            _ = int(vol_changes.pop(0))
            initial_change = int(vol_changes.pop(0))
            if initial_change >= 0:
                failures.append(
                    f"Initial change is not negative on channel {idx}: {initial_change}"
                )
            initial_change = abs(initial_change)
            exp_vol_changes = [1.0, -0.5, 0.5]
            if len(vol_changes) != len(exp_vol_changes):
                failures.append(
                    f"Unexpected number of volume changes on channel {idx}: {vol_changes}"
                )
                continue

            for vol_change, exp_ratio in zip(vol_changes, exp_vol_changes):
                expected = initial_change * exp_ratio
                if abs(int(vol_change) - expected) > 2:
                    failures.append(
                        f"Volume change not as expected on channel {idx}: actual {vol_change}, expected {expected}"
                    )

        elif channel_config[0] == "sine":
            exp_freq = channel_config[1]
            chan_freqs = get_line_matches(analyzer_channels[idx], r"^Channel \d+: Frequency (\d+)")
            if not len(chan_freqs):
                failures.append(f"No signal seen on channel {idx}")
            for freq in chan_freqs:
                if int(freq) != exp_freq:
                    failures.append(
                        f"Incorrect frequency on channel {idx}; got {freq}, expected {exp_freq}"
                    )

        elif channel_config[0] == "ramp":
            exp_ramp = channel_config[1]
            ramps = get_line_matches(analyzer_channels[idx], r".*step = (-?\d+)")

            if len(ramps) == 0:
                failures.append(f"No ramp seen on channel {idx}")

            for ramp in ramps:
                if int(ramp) != exp_ramp:
                    failures.append(
                        f"Incorrect ramp on channel {idx}: got {ramp}, expected {exp_ramp}"
                    )

            for line in analyzer_channels[idx]:
                if re.match(".*discontinuity", line):
                    failures.append(line)

        elif channel_config[0] == "zero":
            if len(analyzer_channels[idx]):
                for line in analyzer_channels[idx]:
                    failures.append(line)

        else:
            failures.append(f"Invalid channel config {channel_config}")

    return failures
