import subprocess
import tempfile
import platform
from pathlib import Path
import os
import stat
import re


class XsigProcess:
    """
    Super-class to run the xsig analyzer/signal generator

    In general, this class shouldn't be used directly; instead use XsigInput or XsigOutput.

    The application requires a sample rate (fs), duration in seconds (can be zero for indefinite
    runtime), a configuration file and the name of the audio device to use.
    """

    def __init__(self, fs, duration, cfg_file, dev_name, ident=None, xsig_dir=None):
        self.duration = 0 if duration is None else duration

        binary_name = "xsig.exe" if platform.system() == "Windows" else "xsig"
        default_dir = Path(__file__).parents[1] / "xsig"
        self.xsig_dir = xsig_dir if xsig_dir else default_dir
        xsig = self.xsig_dir / binary_name
        assert xsig.exists(), f"xsig binary not present in {xsig.parent}"
        self.xsig_cmd = [
            xsig,
            f"{fs}",
            f"{self.duration * 1000}",
            cfg_file,
            "--device",
            dev_name,
        ]
        self.proc = None
        self.proc_output = None
        self.output_file = None
        self.ident = ident

    def __enter__(self):
        self.proc = subprocess.Popen(
            self.xsig_cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT,
            cwd=self.xsig_cmd[0].parent,
            text=True,
        )
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.proc:
            if self.proc.poll() is None:
                self.proc.terminate()

            try:
                out, _ = self.proc.communicate(timeout=15)
            except subprocess.TimeoutExpired:
                assert 0, "Error: timeout getting xsig output"
            self.proc_output = out

        # Rename any glitch data csv files
        if self.ident:
            csv_files = self.xsig_dir.glob("*.csv")
            for file in csv_files:
                filename = file.name

                # Glitch data files are called glitch.<chan>.sig.csv and glitch.<chan>.fft.csv
                # Add in the ident string to make the filename unique for the test run
                artifact_re = r"^(glitch\.)\d.*\.csv$"
                match = re.search(artifact_re, filename)
                if match:
                    pre = match.group(1)
                    target = self.xsig_dir / f"{pre}{self.ident}.{filename[len(pre):]}"
                    file.rename(target)


class XsigInput(XsigProcess):
    """
    Xsig input class

    This class contains logic that is specific to running an instance of xsig for an input test.

    As a workaround for MacOS privacy restrictions, a Python script can be created which will run
    the xsig command with a timeout; this is executed by a separate script running on the host.
    """

    def __enter__(self):
        if (
            platform.system() == "Darwin"
            and os.environ.get("USBA_MAC_PRIV_WORKAROUND", None) == "1"
        ):
            self.output_file = tempfile.NamedTemporaryFile(mode="w+")
            # Create a Python script to run the xsig command
            with tempfile.NamedTemporaryFile(
                "w+", delete=False, dir=Path.home() / "exec_all", suffix=".py"
            ) as script_file:
                # fmt: off
                script_text = (
                    'import subprocess\n'
                    'from pathlib import PosixPath\n'
                    f'with open("{self.output_file.name}", "w+") as f:\n'
                    '    try:\n'
                    f'        ret = subprocess.run({self.xsig_cmd}, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, timeout={self.duration+5}, cwd=PosixPath("{self.xsig_dir}"))\n'
                    f'    except subprocess.TimeoutExpired:\n'
                    f'        f.write("Timeout running command: {self.xsig_cmd}")\n'
                    '    else:\n'
                    '        f.write(ret.stdout.decode())\n'
                )
                # fmt: on
                script_file.write(script_text)
                script_file.flush()
                Path(script_file.name).chmod(stat.S_IREAD | stat.S_IWRITE | stat.S_IEXEC)
            return self
        else:
            return super().__enter__()

    def __exit__(self, exc_type, exc_val, exc_tb):
        if (
            platform.system() == "Darwin"
            and os.environ.get("USBA_MAC_PRIV_WORKAROUND", None) == "1"
        ):
            self.output_file.seek(0)
            proc_output = "\n".join([line.strip() for line in self.output_file.readlines()])
            del self.output_file
        super().__exit__(exc_type, exc_val, exc_tb)
        if (
            platform.system() == "Darwin"
            and os.environ.get("USBA_MAC_PRIV_WORKAROUND", None) == "1"
        ):
            self.proc_output = proc_output

    def get_output(self):
        if (
            platform.system() == "Darwin"
            and os.environ.get("USBA_MAC_PRIV_WORKAROUND", None) == "1"
        ):
            self.output_file.seek(0)
            return [line.strip() for line in self.output_file.readlines()]
        return super().get_output()


class XsigOutput(XsigProcess):
    """
    Xsig output class

    No output-specific logic is required, but this class should still be used instead of
    XsigProcess as it makes the direction of the test-code more obvious to the reader.
    """

    pass
