import subprocess
import time
import socket


class XcoreApp:
    def __init__(self, xe_path, adapter_id, attach=None, timeout=60, xflash=False, writeall=False, target=None):
        assert xe_path.exists()
        assert attach in [None, "io", "xscope", "xscope_app"]

        self.xe_path = xe_path
        self.attach = attach
        self.adapter_id = adapter_id
        self.proc = None
        self.proc_stdout = None
        self.proc_stderr = None
        self.timeout = timeout
        self.xflash = xflash
        self.writeall = writeall
        self.target = target

    # Get an available port number by binding a socket then closing it (assume nothing else will take it before it is used by xrun)
    def get_xscope_port_number(self):
        with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as sock:
            sock.bind(("localhost", 0))
            sock.listen(1)
            port = sock.getsockname()[1]
        self.xscope_port = port

    def wait_for_xscope_port(self, timeout=10):
        # Wait to allow the xrun command to start running
        time.sleep(3)

        for _ in range(timeout):
            time.sleep(1)

            with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as sock:
                try:
                    sock.bind(("localhost", self.xscope_port))
                except OSError:
                    # Failed to bind, so xrun has this port open and is ready to use
                    return

        assert 0, f"xscope port {self.xscope_port} not ready after {timeout}s"

    def __enter__(self):
        if self.xflash:
            if not self.writeall:
                xflash_cmd = [
                    "xflash",
                    "--adapter-id",
                    self.adapter_id,
                    "--factory",
                    self.xe_path,
                ]
            else:
                assert self.target is not None, "Error: target unspecified when doing xflash --write-all"
                xflash_cmd = [
                    "xflash",
                    "--adapter-id",
                    self.adapter_id,
                    "--write-all",
                    self.xe_path,
                    "--target",
                    self.target
                ]

            ret = subprocess.run(
                xflash_cmd, capture_output=True, text=True, timeout=self.timeout
            )
            self.proc_stdout = ret.stdout
            self.proc_stderr = ret.stderr
            assert ret.returncode == 0, (
                f"xflash failed, cmd {xflash_cmd}\n"
                + f"stdout:\n{ret.stdout}\n"
                + f"stderr:\n{ret.stderr}\n"
            )

            return self

        xrun_cmd = ["xrun", "--adapter-id", self.adapter_id]
        if self.attach == "io":
            xrun_cmd.append("--io")
        elif self.attach == "xscope":
            xrun_cmd.append("--xscope")
        elif self.attach == "xscope_app":
            self.get_xscope_port_number()
            xrun_cmd.append("--xscope-port")
            xrun_cmd.append(f"localhost:{self.xscope_port}")
        xrun_cmd.append(self.xe_path)

        if self.attach:
            # If attaching, store the Popen object to terminate it later
            self.proc = subprocess.Popen(
                xrun_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True
            )

            if self.attach == "xscope_app":
                self.wait_for_xscope_port()
        else:
            # If not attaching via xscope, wait for the xrun command to complete
            ret = subprocess.run(
                xrun_cmd, capture_output=True, text=True, timeout=self.timeout
            )
            self.proc_stdout = ret.stdout
            self.proc_stderr = ret.stderr
            assert ret.returncode == 0, (
                f"xrun failed, cmd: {xrun_cmd}\n"
                + f"stdout:\n{ret.stdout}\n"
                + f"stderr:\n{ret.stderr}\n"
            )

        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.terminate()

    def terminate(self):
        if self.proc:
            if self.proc.poll() is None:
                self.proc.terminate()
            self.proc_stdout, self.proc_stderr = self.proc.communicate(timeout=60)

        # Stop an application that was running by breaking in with xgdb
        # Allow a long timeout here because sometimes an XTAG can get stuck in a USB transaction when xrun
        # is killed, and this connect via xgdb can recover it but it takes a while to reboot the XTAG
        try:
            ret = subprocess.run(
                ["xgdb", "-ex", f"connect --adapter-id {self.adapter_id}", "--batch"],
                capture_output=True,
                text=True,
                timeout=120,
            )
        except subprocess.TimeoutExpired:
            assert 0, f"Connecting to XTAG {self.adapter_id} timed out"

        assert ret.returncode == 0, (
            f"Connecting to XTAG {self.adapter_id} failed\n"
            + f"stdout:\n{ret.stdout}\n"
            + f"stderr:\n{ret.stderr}\n"
        )
