import subprocess
import platform
from pathlib import Path
import time
import os
import re


class UaDfuApp:
    def __init__(self, pid, vid=0x20b1, dfu_app_type="custom"):
        assert platform.system() in [
            "Darwin",
            "Linux",
            "Windows",
        ], f"Unsupported platform: {platform.system()}"

        assert dfu_app_type in ["custom", "dfu-util"], f"Unsupported dfu_app_type {dfu_app_type}"

        self.pid = pid[0]
        self.pid_dfu = pid[1]
        self.vid = vid
        self.dfu_app_type = dfu_app_type
        self.custom_dfu_app = None

        if self.dfu_app_type == "custom": # Set the custom DFU app.
            if platform.system() == "Windows":
                self.custom_dfu_app = (
                    Path(os.environ["PROGRAMFILES"])
                    / "XMOS"
                    / "tusbaudiodsk"
                    / "DfuCons"
                    / "x64"
                    / "tlusbdfucons.exe"
                )
                assert self.custom_dfu_app.exists(), f"dfucons.exe not found in location: {self.custom_dfu_app}"
            elif (platform.system() in ["Darwin", "Linux"]): # needs to be set even for dfu_app_type = dfu-util since revertfactory is supported only on the custom apps
                xmosdfu_path = Path(__file__).parents[1] / "xmosdfu" / "xmosdfu"
                assert xmosdfu_path.exists(), f"xmosdfu not found in location: {xmosdfu_path}"
                self.custom_dfu_app = xmosdfu_path

    def _dfu_util_dfu_operation(self, dfu_op, filename=None):
        assert dfu_op in ["upload", "download", "revertfactory"], f"Invalid dfu_op {dfu_op}"
        my_env = os.environ.copy()
        my_env.pop("DYLD_LIBRARY_PATH", None)   # The copy of libusb in tools 15.2.1 doesn't work with dfu-util. TODO Remove this when migrating to tools 15.3
        if dfu_op == "download":
            cmd = ["dfu-util", "-d", f"{hex(self.vid)}:{hex(self.pid)},{hex(self.vid)}:{hex(self.pid_dfu)}", "-D", filename, "-R"]
        elif dfu_op == "upload":
            cmd = ["dfu-util", "-d", f"{hex(self.vid)}:{hex(self.pid)},{hex(self.vid)}:{hex(self.pid_dfu)}", "-U", filename, "-R"]
        else:
            # revertfactory: Download invalid upgrade image to device
            tempf = open("temp.bin", "wb")
            tempf.write(b'\xff\xff\xff\xff')
            tempf.close()
            cmd = ["dfu-util", "-d", f"{hex(self.vid)}:{hex(self.pid)},{hex(self.vid)}:{hex(self.pid_dfu)}", "-D", "temp.bin", "-R"]

        ret = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=300,
            env=my_env
        )

        try:
            Path("temp.bin").unlink()
        except FileNotFoundError:
            pass

        if ret.returncode:
            libusb_error_not_found = 251 # (-5 in 8bit 2's complement signed)
            assert ret.returncode == libusb_error_not_found, f"DFU operation {dfu_op} on image {filename} failed (error {ret.returncode})\nstdout:\n{ret.stdout}\nstderr:\n{ret.stderr}"

    def _custom_dfu_operation(self, dfu_op, filename=None):
        platform_str = platform.system()
        assert dfu_op in ["upload", "download", "revertfactory"], f"Invalid dfu_op {dfu_op}"
        assert platform_str in ["Darwin", "Linux", "Windows"], f"Invalid platform_str {platform_str}"

        if platform_str in ["Darwin", "Linux"]:
            cmd = [self.custom_dfu_app, f"{hex(self.vid)}:{hex(self.pid)},{hex(self.vid)}:{hex(self.pid_dfu)}", f"--{dfu_op}"]
            if filename:
                cmd.append(filename)
        elif platform_str == "Windows":
            thesycon_dfu_op = {"upload": "readout", "download": "upgrade", "revertfactory": "xmosrevertfactory"}
            # Issue 122: need a delay before DFU download, otherwise dfucons can fail
            time.sleep(10)
            cmd = [self.custom_dfu_app, f"{thesycon_dfu_op[dfu_op]}", f"{hex(self.vid)}:{hex(self.pid)},{hex(self.vid)}:{hex(self.pid_dfu)}"]
            if filename:
                cmd.append(filename)

        ret = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=300,
        )
        assert ret.returncode == 0, f"DFU {dfu_op} operation on image {filename} failed (error {ret.returncode})\nstdout:\n{ret.stdout}\nstderr:\n{ret.stderr}"



    def download(self, image_bin):
        if self.dfu_app_type == "dfu-util":
            self._dfu_util_dfu_operation("download", filename=image_bin)
        else:
            self._custom_dfu_operation("download", filename=image_bin)

    def upload(self, output_file):
        # delete the file if it exists
        try:
            Path(output_file).unlink()
        except FileNotFoundError:
            pass

        if self.dfu_app_type == "dfu-util":
            self._dfu_util_dfu_operation("upload", filename=output_file)
        else:
            self._custom_dfu_operation("upload", filename=output_file)

    def revert_factory(self):
        if self.dfu_app_type == "dfu-util":
            self._dfu_util_dfu_operation("revertfactory")
        else:
            self._custom_dfu_operation("revertfactory")

    def get_bcd_version(self):
        if platform.system() == "Windows":
            # Initially the version number can be reported unchanged after
            # a DFU download, so wait briefly before checking
            time.sleep(5)

        timeout = 30
        for _ in range(timeout):
            time.sleep(1)

            if platform.system() == "Darwin":
                ret = subprocess.run(
                    ["system_profiler", "SPUSBDataType"],
                    capture_output=True,
                    check=True,
                    text=True,
                )
                current_pid = None
                current_vid = None
                for i, line in enumerate(ret.stdout.splitlines()):
                    if line.strip().startswith("Product ID:"):
                        current_pid = int(line.split()[2], 16)
                    if line.strip().startswith("Vendor ID:"):
                        current_vid = int(line.split()[2], 16)
                    if line.strip().startswith("Version"):
                        if current_pid == self.pid and current_vid == self.vid:
                            return line.split()[1].strip()
            elif platform.system() == "Linux":
                ret = subprocess.run(
                    ["lsusb", "-vd", f"{hex(self.vid)}:{hex(self.pid)}"],
                    capture_output=True,
                    check=True,
                    text=True,
                )
                for line in ret.stdout.splitlines():
                    if line.strip().startswith("bcdDevice"):
                        version_str = line.split()[1]
                        return version_str.strip()
            elif platform.system() == "Windows":
                if self.dfu_app_type == "custom":
                    ret = subprocess.run(
                        [self.custom_dfu_app, "devinfo", f"0x{self.vid:04x}:0x{self.pid:04x}"],
                        stdout=subprocess.PIPE,
                        stderr=subprocess.STDOUT,
                        text=True,
                        timeout=300,
                    )
                    bcd_re = r"\s*BcdDevice:\s+0x(?P<bcd>\d{4})$"
                    match = re.search(bcd_re, ret.stdout, flags=re.MULTILINE)
                    if match:
                        match_dict = match.groupdict()
                        bcd_ver = match_dict["bcd"]
                        return f"{bcd_ver[:2]}.{bcd_ver[2:]}"
                else:
                    ret = subprocess.run(
                        ["dfu-util", "-l"],
                        stdout=subprocess.PIPE,
                        stderr=subprocess.STDOUT,
                        text=True,
                        timeout=300,
                    )
                    bcd_re = rf"\s*Found Runtime:\s+\[{self.vid:04x}:{self.pid:04x}\]\s+ver=(?P<bcd>\d{{4}})"
                    match = re.search(bcd_re, ret.stdout)
                    if match:
                        match_dict = match.groupdict()
                        bcd_ver = match_dict["bcd"]
                        return f"{bcd_ver[:2]}.{bcd_ver[2:]}"

        assert 0, f"Failed to get device version after {timeout}s; output:\n{ret.stdout}"
