// Copyright (c) 2016-2017, XMOS Ltd, All rights reserved
#include <platform.h>
#include <xs2a_registers.h>
#include <xs2a_kernel.h>
#include <syscall.h>
#include <xclib.h>
#include "loader_xn.h"

#define GLX_XCORE_HELLO 0xC1000000
#define XS1_GLX_CFG_LINK_CTRL_ADRS 0x80
#define XS1_GLX_CFG_SYS_CLK_FREQ_ADRS 0x51

void setup_network(int pll_ctl, int ref_clk_divider, const int routing_table[2])
{
  int usb_dir;
  unsafe chanend c;
  int i;

  // do nothing on tile 1
  // (symbol for calling this function is placed in both tiles' images)
  unsafe {
    // XS1_RES_TYPE_CHANEND=2 (no inline assembly immediate operands in xC)
    asm("getr %0, 2" : "=r"(c));
    asm("freer res[%0]" :: "r"(c));
    if (XS1_CHAN_ID_PROCESSOR((int)c) != 0)
      return;
  }

  // system PLL and reference clock settings
  if (!_is_simulation()) {
    write_sswitch_reg_no_ack(0, XS1_SSWITCH_REF_CLK_DIVIDER_NUM,
      ref_clk_divider);

    write_sswitch_reg_no_ack(0, XS1_SSWITCH_PLL_CTL_NUM,
      pll_ctl | XS1_PLL_CLK_DISABLE_MASK);
  }

  // set SSwitch clock to system frequency
  // (to clock down to 1MHz use system frequency minus 1)
  write_sswitch_reg(0, XS1_SSWITCH_CLK_DIVIDER_NUM, 0);

  // disable all links first (may not be necessary)
  for (i = 0; i < 8; i++) {
    write_sswitch_reg(0, XS1_SSWITCH_XLINK_0_NUM + i, 0);
  }

  // set node ID
  // (get_tile_id reads node ID written by tools in CP)
  write_sswitch_reg_no_ack(0, XS1_SSWITCH_NODE_ID_NUM, get_tile_id(tile[0]));

  // set direction of USB link
  // (find routing table entry for most significant bit of difference
  // between node ID and USB ID)
  usb_dir = (((unsigned long long)routing_table[0] +
            ((unsigned long long)routing_table[1] << 32))
            >> ((31 - clz(get_tile_id(tile[0]) ^ get_tile_id(usb_tile))) * 4))
            & 0xF;
  write_sswitch_reg(get_tile_id(tile[0]), XS1_SSWITCH_SLINK_0_NUM + USB_LINK_NUM,
                    (usb_dir << XS1_LINK_DIRECTION_SHIFT));

  // configure link to USB
  write_sswitch_reg(get_tile_id(tile[0]), XS1_SSWITCH_XLINK_0_NUM + USB_LINK_NUM,
    XS1_XLINK_ENABLE_MASK | XS1_XLINK_WIDE_MASK |
    ((USB_LINK_INTER_DELAY - 1) << XS1_XLINK_INTER_TOKEN_DELAY_SHIFT) |
    ((USB_LINK_INTRA_DELAY - 1) << XS1_XLINK_INTRA_TOKEN_DELAY_SHIFT));

  // set up routing table
  write_sswitch_reg(get_tile_id(tile[0]), XS1_SSWITCH_DIMENSION_DIRECTION0_NUM,
                    routing_table[0]);
  write_sswitch_reg(get_tile_id(tile[0]), XS1_SSWITCH_DIMENSION_DIRECTION1_NUM,
                    routing_table[1]);

  // send hello to issue link credit to USB
  write_sswitch_reg(get_tile_id(tile[0]), XS1_SSWITCH_XLINK_0_NUM + USB_LINK_NUM,
    XS1_XLINK_ENABLE_MASK | XS1_XLINK_WIDE_MASK | XS1_XLINK_HELLO_MASK |
    ((USB_LINK_INTER_DELAY - 1) << XS1_XLINK_INTER_TOKEN_DELAY_SHIFT) |
    ((USB_LINK_INTRA_DELAY - 1) << XS1_XLINK_INTRA_TOKEN_DELAY_SHIFT));

  // configure USB's PLL
  if (!_is_simulation()) {
    write_sswitch_reg(get_tile_id(usb_tile), XS1_GLX_CFG_LINK_CTRL_ADRS,
		      GLX_XCORE_HELLO);
    write_sswitch_reg(get_tile_id(usb_tile), XS1_GLX_CFG_SYS_CLK_FREQ_ADRS, 24);
  }
}
