// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <platform.h>
#include <stdio.h>
#include <assert.h>
#include <timer.h>
#include "link.h"
#include "chan.h"

#define LINK_NUM    3
#define RX_TIME_OUT_TICKS 500000000

#define	DEMO_TILE           0

#if (DEMO_TILE == 0)
#define CHANEND_TILE_ID     0x80010002
#else
#define CHANEND_TILE_ID     0x80030002
#endif

#define GET_SHARED_GLOBAL(x, g) asm volatile("ldw %0, dp[" #g "]":"=r"(x)::"memory")
#define SET_SHARED_GLOBAL(g, v) asm volatile("stw %0, dp[" #g "]"::"r"(v):"memory")

int g_data_tokens;
int g_ctrl_tokens;
int g_timeout_cnts;

void reporting_task()
{
  timer bw_tmr;
  int bw_t;
  int i = 0;
  int j = 0;
  int k = 0;
  int full_rep_cnt = 0;

  bw_tmr :> bw_t;

  while (1) {
    select {
      case bw_tmr when timerafter(bw_t + 100000000) :> void:
        GET_SHARED_GLOBAL(i, g_data_tokens);
        GET_SHARED_GLOBAL(j, g_ctrl_tokens);
        GET_SHARED_GLOBAL(k, g_timeout_cnts);
        printf("Communication rate: %d bytes per sec \t\t\t ==>\t %d Mbit/sec\n", i, ((i*8)/1000000));
        if (!(full_rep_cnt++ % 10)) {
          printf("\nApplication control token count: \t\t%d  \n", j);
          printf("Receive timeouts: \t\t\t\t%d \n\n", k);
        }
        SET_SHARED_GLOBAL(g_data_tokens, 0);
        bw_tmr :> bw_t;
        break;
    }
  }
}

#pragma select handler
void test_ct(unsafe streaming chanend c, int &j)
{
  char r = 'z';
  unsafe {
    if (testct((chanend)c)) {
      r = inct((chanend) c);
      j++;
    }
    else {
      c :> r;
      unsafe {
        volatile int * unsafe p = &g_data_tokens;
        *p = *p + 1;
      }
    }
  }
} //end of test_ct

void receive_link()
{
  unsafe streaming chanend c;
  timer rx_loop_tmr;
  int t;
  int ctrl_tkn_ctr = 0;
  int tm_out_ctr = 0;
  int rx_loop = 0;
  int comm_state = 0;

  rx_loop_tmr :> t;

  printf("Demo started...tile id: %x\n",get_local_tile_id());
  while (1) {
    switch (comm_state) {
      case 0: //setup link direction
        int reg_val = 0;
        int direction = 0x0;
        reg_val = XS1_LINK_DIRECTION_SET(reg_val, direction);
        write_node_config_reg(tile[1], XS1_SSWITCH_SLINK_0_NUM + LINK_NUM, reg_val);
        comm_state = 1;
        break;
      case 1: //channel alloc
        unsafe {
          c = chan_getr();
          //printf("Allocated chanend: %x\n", c);  //debug print, if the tile is changed
          assert((int)c == CHANEND_TILE_ID);
        }
        comm_state = 2;
        break;
      case 2:  //reconf links, leaving only one open
        for (int i=0; i<8; i++)
            link_disable(i);
        link_enable(LINK_NUM);
        /* setup static link for the communicating link */
        int ret = write_sswitch_reg(get_local_tile_id(), XS1_L_SSWITCH_XSTATIC_0_NUM + LINK_NUM, 0x80030000ul);//TODO:
        delay_milliseconds(50);
        comm_state = 3;
        break;
      case 3: //wait for transmit credits
        do {
          link_reset(LINK_NUM);
          link_hello(LINK_NUM);
          delay_microseconds(100);
        } while (!link_got_credit(LINK_NUM));
        printf("\nGot credit\n\n");
        /* setup local control variables */
        rx_loop = 1;
        tm_out_ctr++;
        rx_loop_tmr :> t;
        comm_state = 4;
        break;
      case 4: //receive data loop
        char r = 'z';
        while (rx_loop) {
          unsafe {
            select {
              case test_ct(c, ctrl_tkn_ctr):
                rx_loop_tmr :> t;
                break;
              //case c :> j: break; // breaks at unexpected control tokens
              case rx_loop_tmr when timerafter(t + RX_TIME_OUT_TICKS) :> void:
                SET_SHARED_GLOBAL(g_ctrl_tokens, ctrl_tkn_ctr);
                SET_SHARED_GLOBAL(g_timeout_cnts, tm_out_ctr);
                printf("\nTimed out...\n\n");
                rx_loop = 0;
                comm_state = 3;
                break;
            }
          }
        } //while (rx_loop)
        break;
      case 6: //end of communication; not used in this demo since its a continuous run
        chan_freer(c);
        link_disable(LINK_NUM);
        comm_state = 1;
        break;
      default:
        break;
    } //switch (comm_state)
  } // while (1)
}

int main(void)
{
  par {
    on tile[DEMO_TILE] : receive_link();
    on tile[DEMO_TILE] : reporting_task();
  }
  return 0;
}
