// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <xs1.h>
#include <stdio.h>

/* W=0 for 2bit link, W=1 for 5bit link */
#define W 0

void link_disable(int link_num)
{
  // disable link by resetting ENABLE bit in link's control register
  unsigned x;
  read_sswitch_reg(get_local_tile_id(), XS1_SSWITCH_XLINK_0_NUM + link_num, x);
  x &= ~XS1_XLINK_ENABLE_MASK;
  write_sswitch_reg(get_local_tile_id(), XS1_SSWITCH_XLINK_0_NUM + link_num, x);
}

#define INTER_DELAY 2
#define INTRA_DELAY 3

void link_enable(int link_num)
{
  unsigned x;
  read_sswitch_reg(get_local_tile_id(), XS1_SSWITCH_XLINK_0_NUM + link_num, x);
  // configure link by writing to link's control register:
  // set intertoken and intratoken delays
  // set ENABLE bit
  // set WIDE bit if 5-bit link required
  x |= XS1_L_XLINK_INTER_TOKEN_DELAY_SET(x, INTER_DELAY);
  x |= XS1_L_XLINK_INTRA_TOKEN_DELAY_SET(x, INTRA_DELAY);

  x |= XS1_XLINK_ENABLE_MASK;
  x |= W * XS1_XLINK_WIDE_MASK;
  write_sswitch_reg(get_local_tile_id(), XS1_SSWITCH_XLINK_0_NUM + link_num, x);
}

void link_reset(int link_num)
{
  // reset link by setting RESET bit in link's control register
  unsigned x;
  int l = XS1_SSWITCH_XLINK_0_NUM + link_num;
  read_sswitch_reg(get_local_tile_id(), l, x);
  x |= XS1_XLINK_RX_RESET_MASK;
  write_sswitch_reg(get_local_tile_id(), l, x);
}

void link_hello(int link_num)
{
  // send a hello by setting HELLO bit in link's control register
  unsigned x;
  int l = XS1_SSWITCH_XLINK_0_NUM + link_num;
  read_sswitch_reg(get_local_tile_id(), l, x);
  x |= XS1_XLINK_HELLO_MASK;
  write_sswitch_reg(get_local_tile_id(), l, x);
}

int link_got_credit(int link_num)
{
  unsigned x;
  int l = XS1_SSWITCH_XLINK_0_NUM + link_num;
  read_sswitch_reg(get_local_tile_id(), l, x);
  return XS1_TX_CREDIT(x);
}

static const char plink_type_str[4][16] = {
  "channel end", "error", "psctl", "idle"
};

static const char slink_type_str[3][16] = {
  "plink", "external link", "control link"
};

void print_chanend_status(tileref t, int chan_num)
{
  unsigned x;
  int l = XS1_PSWITCH_LLINK_0_NUM + chan_num;
  read_tile_config_reg(t, l, x);
  printf("chanend %d: 0x%X\n", chan_num, x);
  printf("src_inuse       %d\n", XS1_LINK_SRC_INUSE(x));
  printf("dst_inuse       %d\n", XS1_LINK_DST_INUSE(x));
  printf("junk            %d\n", XS1_LINK_JUNK(x));
  printf("network         %d\n", XS1_LINK_NETWORK(x));
  printf("src_target_id   %d\n", XS1_LINK_SRC_TARGET_ID(x));
  printf("src_target_type %d (%s)\n", XS1_LINK_SRC_TARGET_TYPE(x), plink_type_str[XS1_LINK_SRC_TARGET_TYPE(x)]);
}

void print_plink_pswitch_status(tileref t, int plink_num)
{
  unsigned x;
  int l = XS1_PSWITCH_PLINK_0_NUM + plink_num;
  read_tile_config_reg(t, l, x);
  printf("plink %d (pswitch): 0x%X\n", plink_num, x);
  printf("src_inuse       %d\n", XS1_LINK_SRC_INUSE(x));
  printf("dst_inuse       %d\n", XS1_LINK_DST_INUSE(x));
  printf("junk            %d\n", XS1_LINK_JUNK(x));
  printf("network         %d\n", XS1_LINK_NETWORK(x));
  printf("src_target_id   %d\n", XS1_LINK_SRC_TARGET_ID(x));
  printf("src_target_type %d (%s)\n", XS1_LINK_SRC_TARGET_TYPE(x), plink_type_str[XS1_LINK_SRC_TARGET_TYPE(x)]);
}

void print_plink_sswitch_status(tileref t, int plink_num)
{
  unsigned x;
  int l = XS1_SSWITCH_PLINK_0_NUM + plink_num;
  read_node_config_reg(t, l, x);
  printf("plink %d (sswitch): 0x%X\n", plink_num, x);
  printf("src_inuse       %d\n", XS1_LINK_SRC_INUSE(x));
  printf("dst_inuse       %d\n", XS1_LINK_DST_INUSE(x));
  printf("junk            %d\n", XS1_LINK_JUNK(x));
  printf("network         %d\n", XS1_LINK_NETWORK(x));
  printf("src_target_id   %d\n", XS1_LINK_SRC_TARGET_ID(x));
  printf("src_target_type %d (%s)\n", XS1_LINK_SRC_TARGET_TYPE(x), slink_type_str[XS1_LINK_SRC_TARGET_TYPE(x)]);
}

void print_slink_status(tileref t, int link_num)
{
  unsigned x;
  int l = XS1_SSWITCH_SLINK_0_NUM + link_num;
  read_node_config_reg(t, l, x);
  printf("slink %d: 0x%X\n", link_num, x);
  printf("src_inuse       %d\n", XS1_LINK_SRC_INUSE(x));
  printf("dst_inuse       %d\n", XS1_LINK_DST_INUSE(x));
  printf("junk            %d\n", XS1_LINK_JUNK(x));
  printf("network         %d\n", XS1_LINK_NETWORK(x));
  printf("direction       %d\n", XS1_LINK_DIRECTION(x));
  printf("src_target_id   %d\n", XS1_LINK_SRC_TARGET_ID(x));
  printf("src_target_type %d\n", XS1_LINK_SRC_TARGET_TYPE(x), slink_type_str[XS1_LINK_SRC_TARGET_TYPE(x)]);
}
