// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <platform.h>
#include <stdio.h>
#include <timer.h>
#include "link.h"
#include "chan.h"
#include "trycatch.h"

#define GET_SHARED_GLOBAL(x, g) asm volatile("ldw %0, dp[" #g "]":"=r"(x)::"memory")
#define SET_SHARED_GLOBAL(g, v) asm volatile("stw %0, dp[" #g "]"::"r"(v):"memory")

#define LINK_NUM    3
#define RE_ENABLE_TX_PERIOD 6
#define SEND_CTRL_TOKEN    2500000

int g_comm_state;

void re_enable_tx_link()
{
  while (1) {
    delay_seconds(RE_ENABLE_TX_PERIOD);
    SET_SHARED_GLOBAL(g_comm_state, 1);
    printf("Re-enable transmit link. Waiting for credit..\n");
    link_disable(LINK_NUM);
    link_enable(LINK_NUM);
  } //while (1)
}

void transmit_handler(int comm_state)
{
  unsafe streaming chanend c;
  int temp_state = 0;
  int err_ctr = 0;

  while (1) {
    GET_SHARED_GLOBAL(temp_state, g_comm_state);
    if (temp_state) {
      SET_SHARED_GLOBAL(g_comm_state, 0);
      comm_state = 4;
    }
    switch(comm_state) {
      case 0: //setup link direction
        int reg_val = 0;
        int direction = 0x1;
        reg_val = XS1_LINK_DIRECTION_SET(reg_val, direction);
        write_node_config_reg(tile[0], XS1_SSWITCH_SLINK_0_NUM + LINK_NUM, reg_val);
        comm_state = 1;
        break;
      case 1: //channel setup
        unsafe {
          c = chan_getr();
          chan_setd(c, 0x80030002);
        }
        comm_state = 2;
        break;
      case 2: /* reconfigure links, leaving only one open */
        for (int i=0; i<8; i++)
          link_disable(i);
        link_enable(LINK_NUM);
        comm_state = 3;
        break;
      case 3: /* Setup a static routing configuration */
        int ret = write_sswitch_reg(get_local_tile_id(), XS1_L_SSWITCH_XSTATIC_0_NUM + LINK_NUM, 0x80000000ul);
        delay_milliseconds(50);
        comm_state = 4;
        break;
      case 4: /* wait for transmit credits */
        do {
          link_reset(LINK_NUM);
          link_hello(LINK_NUM);
          delay_microseconds(100);
        } while (!link_got_credit(LINK_NUM));
        printf("Got credit\n");
        err_ctr = 0;
        comm_state = 5;
        break;
      case 5: /* send data tokens */
        /*if (!link_got_credit(LINK_NUM)) {
          comm_state = 4;
          break;
        }*/
        unsafe { c <: 'a'; }
        if (err_ctr++ == SEND_CTRL_TOKEN) {
          err_ctr = 0;
          unsafe {
            outct((chanend)c, XS1_CT_ACK);  //Send a control token as app sync point
          }
        }
        break;
      case 6: /* not used since this is a continuous running demo */
        chan_freer(c);
        link_disable(LINK_NUM);
        comm_state = 2;
        break;
      default:
        comm_state = 4;
        break;
    } //switch(comm_state)
  } //while (1)
}

void transmit_link()
{
  exception_t exception;

  TRY {
    printf("Demo started...tile id: %x\n",get_local_tile_id());
    transmit_handler(0);
  } CATCH (exception) {
    printf("Got exception.\n");
    transmit_handler(4);
  }
}

int main(void)
{
  par {
    on tile[0] : transmit_link();
    on tile[0] : re_enable_tx_link();
  }
  return 0;
}
