// Copyright (c) 2015, XMOS Ltd, All rights reserved
/*
 * Generates a 5 - 40 Hz Triangle waveform and sends it over a channel
 * Amplitude and frequency are controlled via the sliders on startKIT
 *
 * Every millisecond, it sends a new sample across
 */
#include "wavegen.h"
#include <debug_print.h>

#define TICKS_PER_MILLISECOND (XS1_TIMER_HZ / 1000)
#define TICKS_PER_SECOND (XS1_TIMER_HZ)
#define MAX_DEPTH 0x7fffffff

#define SLIDER_MIN 1000
#define SLIDER_MAX 2000

#define FREQ_MIN 5
#define FREQ_MAX 40

void wavegen(chanend c,
             client slider_if i_slider_x, client slider_if i_slider_y)
{
  int gain = MAX_DEPTH, dir = 1, time;
  int hertz = 30;
  int depth = 0; //0 = no effect, MAX_DEPTH = max depth
  int step = (MAX_DEPTH/((TICKS_PER_SECOND/TICKS_PER_MILLISECOND) / (2 * hertz)));
  timer tmr;
  tmr :> time;
  debug_printf("Amplitude modulation freq is %dHz\n", hertz);
  debug_printf("Amplitude modulation depth is 0 (i.e. disabled)\n");
  while(1){
    select{
    case tmr when timerafter(time) :> time: //Time for a new sample to be generated
      if(dir==1) gain += step;            //Rising phase of triangle
      else gain -= step;                  //Falling phase of triangle
      if (gain >= (MAX_DEPTH-step)) dir = -1;
      if (gain <= step) dir = 1;

      c <: MAX_DEPTH - (int) (((long long)depth * (long long)gain) >> 31);
      time += TICKS_PER_MILLISECOND;
      break;

    case i_slider_y.changed_state():
      sliderstate state = i_slider_y.get_slider_state();
      if (state == RIGHTING) {
        hertz -= 3;
        if (hertz < FREQ_MIN)
          hertz = FREQ_MIN;
        debug_printf("Amplitude modulation freq decreased to %dHz\n", hertz);
      }
      if (state == LEFTING) {
        hertz += 3;
        if (hertz > FREQ_MAX)
          hertz = FREQ_MAX;
        debug_printf("Amplitude modulation freq increased to %dHz\n", hertz);
      }
      step = (MAX_DEPTH/((TICKS_PER_SECOND/TICKS_PER_MILLISECOND) / (2 * hertz)));
      break;

    case i_slider_x.changed_state():
      sliderstate state = i_slider_x.get_slider_state();
      if (state == RIGHTING) {
        depth -= MAX_DEPTH/5;
        if (depth < 0)
          depth = 0;
        debug_printf("Amplitude modulation depth decreased to 0x%x\n", depth);
      }
      if (state == LEFTING) {
        if (depth > MAX_DEPTH - MAX_DEPTH/5)
          depth = MAX_DEPTH;
        else
          depth += MAX_DEPTH/5;
        debug_printf("Amplitude modulation depth increased to 0x%x\n", depth);
      }
      break;
    }
  }
}
