// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <audio_effects.h>
#include <startkit_gpio.h>
#include <stdint.h>
#include <stddef.h>
#include <debug_print.h>
#include <xscope.h>
#include <biquad.h>

/* Function that applies gain to a sample. The
   range of the gain term is 0 to 7fffffff.
*/
int apply_gain(int sample, int gain) {
  return (((int64_t) sample) * (int64_t) gain) >> 31;
}

/* This array matches the enum type defined in biquad.h.
 * It assign each biquad type to a string for debug printing.
 */
const char *filt_names[NUM_BIQUAD_TYPES] =
{ "Low pass filter (cuts out high freqs)",
  "High pass filter (cuts out low freqs)",
  "Band pass filter (cuts out low and high freqs)",
  "Notch filter (cuts out mid freqs)",
  "All pass filter (sounds the same but phase shifted)"
};

void audio_effects(streaming chanend c_dsp,
                   client startkit_led_if i_led,
                   client startkit_button_if i_button,
                   chanend c_gain,
                   static const size_t num_chans)
{
  // Unprocessed input audio sample buffer
  int32_t in_samps[num_chans] = {0};

  // Sample buffer to hold samples after processing
  int32_t out_samps[num_chans] = {0};

  int32_t gain = 0; // Start with volume at 0

  biquad_state_t biquad_state[num_chans];

  int cur_effect = 0; // This value is either 0 (meaning no effect should
                      // be applied) or a value representing one more than
                      // an effect type value as decribed by the enum in
                      // biquad.h

  debug_printf("Biquad effect off\n");
  i_led.set_multiple(0b111111111, LED_OFF);

  while(1) {
    // Send/Receive samples
    for (size_t i = 0; i < num_chans; i++) {
      c_dsp :> in_samps[i];
      c_dsp <: out_samps[i];
    }

    xscope_int(LEFT_IN, in_samps[0]);
    xscope_int(LEFT_OUT, out_samps[0]);
    xscope_int(GAIN, gain);

    select{
    case i_button.changed():
      if (i_button.get_value() == BUTTON_DOWN) {
        cur_effect++;
        if (cur_effect == NUM_BIQUAD_TYPES + 1)
          cur_effect = 0;

        i_led.set_multiple(0b111111111, LED_OFF);
        if (cur_effect == 0) {
          debug_printf("Biquad effect off\n");
          break;
        }
        enum biquad_type_t btype = cur_effect - 1;

        debug_printf("Effect: %s: Freq 1000Hz\n", filt_names[btype]);
        for (size_t i = 0; i < num_chans; i++) {
          init_biquad_state(btype, 1000, 48000, BIQUAD_Q_BUTTERWORTH,
                            biquad_state[i]);
        }
        i_led.set_multiple(1 << btype, LED_ON);
      }
      break; // end of button handling case
    case c_gain :> gain:
      break;
    default:
      break;
    }

    // Do DSP processing ...
    for (size_t i = 0; i < num_chans; i++) {
      if (cur_effect != 0)
        out_samps[i] = apply_biquad(in_samps[i], biquad_state[i]);
      else
        out_samps[i] = in_samps[i];
     }

    // Apply gain
    for (size_t i = 0; i < num_chans; i++)  {
      out_samps[i] = apply_gain(out_samps[i], gain);
    }
  }
}
