// Copyright (c) 2016, XMOS Ltd, All rights reserved

/**
How to specify a loop count in the XTA
--------------------------------------

In many cases the XTA will know the number of iterations for a given
loop. However, in some cases, this information cannot be deduced by
the tools so you must supply it.

This example shows how to set up both a 'global' and a 'local' loop
count for a given loop. (Note: The loop count is defined as the number
of times the body of the loop is executed).

'Global' in this context means that once set, this loop count is
applied to all routes subsequently created that contain this loop.

However, 'Local' means that this loop count is only applied to a given
pre-existing route, and will have no effect on either other pre-
existing routes, or subsequently created routes that contain this
loop.

For example, compile the following code:
**/


int f() {
  int i, j = 0;
  for (i = 0; i < 10; ++i) {
    #pragma xta label "loop_label"
    j += i;
  }
  return j;
}

int main() {
  f();
  return 0;
}

/**
Load the resulting executable into the XTA then click the 'Add define'
button in the toolbar. In the dialog, enter 'loop_label' in the
reference section and '10' in the num iterations section. Now time the
function 'f'. By considering the tree constructed for this route you
can see that the loop count of 10 has been correctly applied to this
route.

To add a global loop count using the command line XTA, or from an XTA
script/embedded source command, the following can be used:

   add loop loop_label 10

Load the resulting executable into the XTA then time the function 'f'.
This will create the route. Right-click in the left hand side border
of the editor on the source line inside the loop (j += i), and select
'Set loop iterations'. In the resulting dialog enter '10' in the num
iterations section.

To add a local loop count using the command line XTA, or from an XTA
script/embedded source command, the following can be used:

   set loop 0 loop_label 10

This will set a loop count of 10 on the loop containing 'loop_label'
for the route with an id of 0.
**/

