// Copyright (c) 2015, XMOS Ltd, All rights reserved

/**
How to specify an exclusion in the XTA
--------------------------------------

Not all paths of execution in a route may be timing-critical. For
example, the route may contain cases to handle errors where the timing
of the code is not important. You can tell the XTA to ignore these
paths by adding exclusions. Exclusions tell the tool to ignore all
paths which pass through that code point. Exclusions can be added
either *globally* or *locally*.

*Global* in this context means that once set, this exclusion is
applied to all routes subsequently created that contain this code
path.

However, *local* means that this exclusion is only applied to a given
pre-existing route, and will have no effect on either other pre-
existing routes, or subsequently created routes that contain this code
path.

For example, compile the following code:
**/


int add_if_not_equal(int a, int b) {
  if (a == b) {
    #pragma xta label "error_case"
    return -1;
  }
  return a + b;
}

int main() {
  add_if_not_equal(1, 2);
  return 0;
}

/**
Load the resulting executable into the XTA then right-click in the
left hand side border of the editor on the source line inside the
conditional (return -1), and select *Add to exclusion list*. Now
analyze the function 'add_if_not_equal'. The resulting route will now
only contain a single code path, i.e. the "error_case" code path will
have been excluded.

To add a global exclusion using the command line XTA, or from an XTA
script/embedded source command, the following can be used:

   add exclusion error_case

Load the resulting executable into the XTA then time the function *Add
to exclusion list*. This will create the route. Right-click in the
left-hand side border of the editor on the source line inside the
conditional (return -1), and select 'Exclude'. The code path
containing "error_case" will now be removed from the route.

To add a local exclusion using the command line XTA, or from an XTA
script/embedded source command, the following can be used:

   set exclusion 0 error_case

This will set an exclusion on the code path containing "error_case"
for the route with an id of 0.

Note: Although functionally equivalent, global exclusions can result
in faster, and more memory efficient, route creation. This is because
the global exclusions can be taken into account during route creation,
so the search space can be reduced. For local exclusions, the complete
route is created before any pruning occurs.
**/

