// Copyright (c) 2015, XMOS Ltd, All rights reserved

/**
How to specify loop iterations for specific paths in a loop
-----------------------------------------------------------

A loop may contain multiple paths through it. When a loop iteration
count has been set the tool assumes that all iterations will take the
worst-case path of execution through the loop. This is not always the
case, and a more realistic worst-case can be established by specifying
the number of iterations on individual paths through the loop.

For example, compile the following code:
**/

int f() {
  int x = 0;
  for (unsigned int i = 0; i < 10; ++i) {
    #pragma xta label "loop_label"
    if ((i & 1) == 0) {
      #pragma xta label "if_label"
      ++x;
    }
  }
  return x;
}

int main() {
  f();
  return 0;
}

/**
Load the resulting executable into the XTA then click the 'Add define'
button in the toolbar. In the dialog, On the *Loop* tab, enter 10
iterations using *loop_label* as the reference. Next switch to the
*Loop Path* tab and enter 5 iterations using 'if_label' as the
reference. Any future routes containing this loop will now correctly
assign five iterations to each side of the conditional in the loop.

To add a global loop scope using the command line XTA, or from an XTA
script/embedded source command, the following can be used:

   add loop loop_label 10
   add looppath if_label 5

Load the resulting executable into the XTA then time the function 'f'.
This will create the route. Right-click in the left hand side border
of the editor on the source line inside the conditional (++x) and
select 'Set loop iterations'. In the resulting dialog, set the number
of iterations to 10. Now right-click on the same line and select 'Set
loop path iterations', and this time enter 5 in the dialog.

To add a local loop scope using the command line XTA, or from an XTA
script/embedded source command, the following can be used:

   set loop 0 loop_label 10
   set looppath 0 if_label 5

This will set a the loop iterations for the route with an id of 0.

Note: There are some rules that need to be followed when setting loop
path iterations:

   * In a nested loop, the outer loop iterations need to be set first.

   * The sum of the loop path iterations set must be less than or
     equal to the loop iterations set on the enclosing loop.

   * If the loop path iterations set are less than that of the
     enclosing loop, then there must exist another path within the
     loop without its iterations set to which the remaining iterations
     can be allocated.
**/

