// Copyright (c) 2015, XMOS Ltd, All rights reserved

#include <platform.h>
#include <xs1.h>
#include <stdio.h>
#include <string.h>
#include <timer.h>
#include "i2c.h"
#include "xud_cdc.h"

/* App specific defines */
#define MENU_MAX_CHARS  30
#define MENU_LIST       11
#define DEBOUNCE_TIME   (XS1_TIMER_HZ/50)
#define BUTTON_PRESSED  0x00

/* GPIO slice on SQUARE slot, PORT_4A connected to the 4 LEDs */
on tile[1]: port p_led = XS1_PORT_4A;

/* PORT_4C connected to the 2 Buttons */
on tile[1]: port p_button = XS1_PORT_4C;

// Temperature to ADC look up table 
int TEMPERATURE_LUT[][2]=
{
  {-10,850},{-5,800},{0,750},{5,700},{10,650},
  {15,600},{20,550},{25,500},{30,450},{35,400},
  {40,350},{45,300},{50,250},{55,230},{60,210}
};

char app_menu[MENU_LIST][MENU_MAX_CHARS] = {
        {"\n\r-------------------------\r\n"},
        {"XMOS USB Virtual COM Demo\r\n"},
        {"-------------------------\r\n"},
        {"1. Switch to Echo mode\r\n"},
        {"2. Toggle LED 1\r\n"},
        {"3. Toggle LED 2\r\n"},
        {"4. Toggle LED 3\r\n"},
        {"5. Toggle LED 4\r\n"},
        {"6. Read Temparature\r\n"},
        {"7. Print timer ticks\r\n"},
        {"-------------------------\r\n"},
};

char echo_mode_str[3][30] = {
        {"Entered echo mode\r\n"},
        {"Press Ctrl+Z to exit it\r\n"},
        {"\r\nExit echo mode\r\n"},
};

#define ARRAY_SIZE(x) (sizeof(x)/sizeof(x[0]))

/* Sends out the App menu over CDC virtual port*/
void show_menu(client interface usb_cdc_interface cdc)
{
    unsigned length;
    for(int i = 0; i < MENU_LIST; i++) {
        length = strlen(app_menu[i]);
        cdc.write(app_menu[i], length);
    }
}

/* Function to set LED state - ON/OFF */
void set_led_state(int led_id, int val)
{
  int value;
  /* Read port value into a variable */
  p_led :> value;
  if (!val) {
      p_led <: (value | (1 << led_id));
  } else {
      p_led <: (value & ~(1 << led_id));
  }
}

/* Function to toggle LED state */
void toggle_led(int led_id)
{
    int value;
    p_led :> value;
    p_led <: (value ^ (1 << led_id));
}

/* Function to get button state (0 or 1)*/
int get_button_state(int button_id)
{
    int button_val;
    p_button :> button_val;
    button_val = (button_val >> button_id) & (0x01);
    return button_val;
}

/* Checks if a button is pressed */
int is_button_pressed(int button_id)
{
    if(get_button_state(button_id) == BUTTON_PRESSED) {
        /* Wait for debounce and check again */
        delay_ticks(DEBOUNCE_TIME);
        if(get_button_state(button_id) == BUTTON_PRESSED) {
            return 1; /* Yes button is pressed */
        }
    }
    /* No button press */
    return 0;
}

/* Uses the temperature look up table and interpolation
 * technique to determine exact temperature value */
static int linear_interpolation(int adc_value)
{
  int i=0,x1,y1,x2,y2,temperature;

  while(adc_value<TEMPERATURE_LUT[i][1]) {
    i++;
  }
  //Calculating Linear interpolation using the formula y=y1+(x-x1)*(y2-y1)/(x2-x1)
  x1=TEMPERATURE_LUT[i-1][1];
  y1=TEMPERATURE_LUT[i-1][0];
  x2=TEMPERATURE_LUT[i][1];
  y2=TEMPERATURE_LUT[i][0];
  temperature=y1+(((adc_value-x1)*(y2-y1))/(x2-x1));
  return temperature;
}

/* Read ADC value using I2C and apply linear interpolation */
static int get_temperature(client interface i2c_master_if i2c)
{
  int adc_value;
  unsigned char i2c_data[2];

  i2c.read(0x28, i2c_data, sizeof(i2c_data), 1);

  i2c_data[0] = i2c_data[0]&0x0F;
  adc_value = (i2c_data[0]<<6) | (i2c_data[1]>>2);
  return linear_interpolation(adc_value);
}

/* Initializes the Application */
void app_init(client interface i2c_master_if i2c)
{
    /* Set all LEDs to OFF (Active low)*/
    p_led <: 0x0F;
    /* ADC Config Start */
    i2c.write_reg(0x28, 0x00, 0x13);
}

/* Application task */
void app_virtual_com_extended(client interface usb_cdc_interface cdc, client interface i2c_master_if i2c)
{
    unsigned int length, led_id;
    int temperature = 0;
    char value, tmp_string[50];
    unsigned int button_1_valid, button_2_valid;
    timer tmr;
    unsigned int timer_val;

    app_init(i2c);
    show_menu(cdc);

    button_1_valid = button_2_valid = 1;

    while(1)
    {
        /* Check for a change in button 1 - Detects 1->0 transition */
        if(is_button_pressed(0)) {
            if(button_1_valid) {
                button_1_valid = 0;
                length = sprintf(tmp_string, "\r\nButton 1 Pressed!\r\n");
                cdc.write(tmp_string, length);
            }
        } else {
            button_1_valid = 1;
        }


        /* Check for a change in button 2 - Detects 1->0 transition */
        if(is_button_pressed(1)) {
            if(button_2_valid) {
                button_2_valid = 0;
                length = sprintf(tmp_string, "\r\nButton 2 Pressed!\r\n");
                cdc.write(tmp_string, length);
            }
        } else {
            button_2_valid = 1;
        }

        /* Check if user has input any character */
        if(cdc.available_bytes())
        {
            value = cdc.get_char();

            /* Do the chosen operation */
            if(value == '1') {
                length = strlen(echo_mode_str[0]);
                cdc.write(echo_mode_str[0], length);
                length = strlen(echo_mode_str[1]);
                cdc.write(echo_mode_str[1], length);

                while(value != 0x1A) { /* 0x1A = Ctrl + Z */
                    value = cdc.get_char();
                    cdc.put_char(value);
                }
                length = strlen(echo_mode_str[2]);
                cdc.write(echo_mode_str[2], length);
            }
            else if((value >= '2') && (value <= '5')) {
                /* Find out which LED to toggle */
                led_id = (value - 0x30) - 2;    // 0x30 used to convert the ascii to number
                toggle_led(led_id);
            }
            else if(value == '6') {
                /* Read temperature and print */
                temperature = get_temperature(i2c);
                length = sprintf(tmp_string, "Temperature: %d %cC\r\n", temperature, (char)0xB0); //B0 is the hex value of degree symbol
                cdc.write(tmp_string, length);
            }
            else if(value == '7') {
                /* Read 32-bit timer value */
                tmr :> timer_val;
                length = sprintf(tmp_string, "Timer ticks: %u\r\n", timer_val);
                cdc.write(tmp_string, length);
            }
            else {
                show_menu(cdc);
            }
        }
    } /* end of while(1) */
}
