// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <xs1.h>
#include <platform.h>
#include <math.h>

/**
How to use xSCOPE in real-time mode
-----------------------------------

xSCOPE is fully supported on hardware platforms which provide an XMOS
link between the target device and the XSYS development connector, it
is also supported via xSIM.

View the document (http://www.xmos.com/doc/XM-000957-PC/latest/page31
#trace-data-with-xscope) for further information on tracing data from
XMOS applications.

This example provides a simple demonstration of using the xSCOPE
continuous event type for data logging from within an XMOS
application. The continuous event type can be used to capture and log
the value of specific variables within an application to allow
debugging. xSCOPE can operate in both a post mortem and a real-time
mode where data is streamed back to the host machine and displayed as
the application is running. In this example we look at using this
mode.

The probe configuration is handled by the user providing a
config.xscope file which is picked up as part of the application
build.

This example assumes you are familiar with creating a run
configuration and enabling the associated xSCOPE options in that run
configuration in xTIMEcomposer Studio or using the command line tools.
For this example the user needs to select the real-time option in the
xSCOPE configuration.
**/


/**
    In order to used xSCOPE the correct header file must be included in
    the application
**/

#include <xscope.h>

#define INTERVAL 1000000

void wait(int delay)
{
  timer t;
  int i;
  t:>i;
  t when timerafter(i + delay):>i;
}

void run()
{
  int i = 0;
  float j = 0;
  float sin_values[64] = { 0 };
  float cos_values[64] = { 0 };
  float tan_values[64] = { 0 };
  int wait_time = INTERVAL / 63;

  for (i = 0; i < 63; i++) {
    sin_values[i] = sin(j) * 100;
    cos_values[i] = cos(j) * 100;
    tan_values[i] = tan(j) * 100;
    j += 0.1;
  }

  wait(1000000);

/**
    The xscope_float() function is used to send the contents of user
    variable values to the xSCOPE probe for real time display. In this
    example the program loops and continuously outputs events to the
    xSCOPE system for display in real time
**/

  while (1) {
    for (i = 0; i < 63; i++) {
      wait(wait_time);
      xscope_float(SIN_VALUE, sin_values[i]);
      xscope_float(COS_VALUE, cos_values[i]);
      xscope_float(TAN_VALUE, tan_values[i]);
    }
  }
}

int main(void)
{
  par {
    on tile[0]: run();
  }
  return 0;
}
