// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <platform.h>

/**
How to use the xSCOPE to capture logical core information
---------------------------------------------------------

xSCOPE is fully supported on hardware platforms which provide an XMOS
link between the target device and the XSYS development connector, it
is also supported via xSIM.

View the document (http://www.xmos.com/doc/XM-000957-PC/latest/page31
#trace-data-with-xscope) for further information on tracing data from
XMOS applications.

This example provide a simple demonstration of using the functionality
in xSCOPE to capture additional information about logical core state
when tracing events. A user can add data about the id, program counter
and node of a logical core generating an event. This allows a user to
use register a single event which the xSCOPE system will split into
multiple logical core events when processed on the host machine.

The probe configuration is handled by the user providing a
config.xscope file which is picked up as part of the application
build.

This example assumes you are familiar with creating a run
configuration and enabling the associated xSCOPE options in that run
configuration in xTIMEcomposer Studio or using the command line tools.
**/


/**
    In order to used xSCOPE the correct header file must be included in
    the application
**/

#include <xscope.h>

void output_data_1(unsigned int value)
{
  /** The xscope_core_int() function is used to send the contents of user
      variable current_state to the xSCOPE probe for logging and add the
      additional information associated with the logical core to the data
      transmitted.
   **/
  xscope_core_int(CONTINUOUS_VALUE_1, value);
}

void output_data_2(unsigned int value)
{
  xscope_core_int(CONTINUOUS_VALUE_2, value);
}

void do_output_loop(unsigned int multiplier)
{
  for (int i = 0; i < 100; i++) {
    output_data_1(i * multiplier);
    output_data_2(i * 2 * multiplier);
  }
}


int main(void)
{
  par {
    on tile[0]: do_output_loop(1);
    on tile[0]: do_output_loop(3);
  }
  return 0;
}
