// Copyright (c) 2015, XMOS Ltd, All rights reserved
#include <platform.h>

/**
How to use the xSCOPE discrete event type
-----------------------------------------

xSCOPE is fully supported on hardware platforms which provide an XMOS
link between the target device and the XSYS development connector, it
is also supported via xSIM.

View the document (http://www.xmos.com/doc/XM-000957-PC/latest/page31
#trace-data-with-xscope) for further information on tracing data from
XMOS applications.

This example provides a simple demonstration of using the xSCOPE
discrete event type for data logging from within an XMOS application.
The discrete event type is used for recording the value of a captured
variable for a length of time before that value is changed. This lets
you see the level of a particular variable over time and the points at
which it has been changed by the application.

The probe configuration is handled by the user providing a
config.xscope file which is picked up as part of the application
build.

This example assumes you are familiar with creating a run
configuration and enabling the associated xSCOPE options in that run
configuration in xTIMEcomposer Studio or using the command line tools.
**/


/**
    In order to used xSCOPE the correct header file must be included in
    the application
**/

#include <xscope.h>


void wait(int delay)
{
  timer t;
  int i;
  t :> i;
  t when timerafter(i + delay):>i;
}

/**
    Declare a variable you are going to trace as an xSCOPE discrete event
    type
**/

unsigned int memory_used = 0;

void allocate_memory(unsigned int size)
{
  memory_used += size;
  /** The xscope_int() function is used to send the contents of user
      variable memory_used to xSCOPE probe MEMORY_ALLOCATED for logging
   **/
  xscope_int(MEMORY_ALLOCATED, memory_used);
}


int main(void)
{
  par {
    on tile[0]: {
      for (int i = 1; i < 10; i++) {
        wait(1000);
        allocate_memory(i * 100);
      }
    }
  }
  return 0;
}
