// Copyright (c) 2016, XMOS Ltd, All rights reserved
/**
How to use an ordered select statement
--------------------------------------

A select statement waits for one of a set of inputs to become ready,
performs the selected input and then executes a corresponding body of
code. Each input is proceeded by the keyword case and its body must be
terminated with a break or return statement. Case statements are not
permitted to contain output operations. Selects can also be ordered so
that priority is given based on the order of the case statements. This
allows code to take a decision over which to execute first when events
occur at the same time.
**/


#include <platform.h>
#include <print.h>

void RX (chanend chnlend_a, chanend chnlend_b)
{
  unsigned chnl_input_a = 0;
  unsigned chnl_input_b = 0;

  while (1)
  {
/**
    In this example the select statement is used to wait for either an
    input on chnlend_a or an input on chnlend_b.  When an input is
    received from either channel the value of the input is printed. Using
    '#pragma ordered' the order in which events are processed is based on
    the order of case statements in the select.
**/

    
#pragma ordered
    select
    {
      case chnlend_a :>  chnl_input_a :
        printstr("Channel Input A Received ");
        printintln(chnl_input_a);
        break;
      case chnlend_b :> chnl_input_b :
        printstr("Channel Input B Received ");
        printintln(chnl_input_b);
        break;
    }

  }
}

void TX_1 (chanend chnlend_a)
{
  unsigned chnl_output = 0;

  while (1)
  {
    chnl_output++;
    chnlend_a <: chnl_output;
  }
}

void TX_2 (chanend chnlend_b)
{
  unsigned chnl_output = 0;

  while (1)
  {
    chnl_output++;
    chnlend_b <: chnl_output;
  }
}

int main(void)
{
  chan chnl_a, chnl_b;

  par
  {
    on stdcore[0] : RX(chnl_a, chnl_b);
    on stdcore[1] : TX_2(chnl_b);
    on stdcore[1] : TX_1(chnl_a);
  }

  return 0;
}

